/**
 * Copyright (c) 2012 - 2022 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package com.playertour.backend.possiblecheats.service.tests;

import static org.assertj.core.api.Assertions.assertThat;

import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.util.List;
import java.util.UUID;

import org.gecko.mongo.osgi.MongoDatabaseProvider;
import org.gecko.search.util.CommonIndexService;
import org.junit.jupiter.api.AfterAll;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.MethodOrderer;
import org.junit.jupiter.api.Order;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.TestMethodOrder;
import org.junit.jupiter.api.extension.ExtendWith;
import org.junit.platform.commons.annotation.Testable;
import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceReference;
import org.osgi.test.common.annotation.InjectBundleContext;
import org.osgi.test.common.annotation.InjectService;
import org.osgi.test.common.service.ServiceAware;
import org.osgi.test.junit5.context.BundleContextExtension;
import org.osgi.test.junit5.service.ServiceExtension;

import com.mongodb.client.MongoDatabase;
import com.playertour.backend.api.LandingZone;
import com.playertour.backend.api.PlayerProfile;
import com.playertour.backend.api.PlayerScorecard;
import com.playertour.backend.apis.course.CourseService;
import com.playertour.backend.apis.player.PlayerScorecardService;
import com.playertour.backend.apis.player.PlayerService;
import com.playertour.backend.golfcourse.model.golfcourse.GenderType;
import com.playertour.backend.golfcourse.model.golfcourse.GolfCourse;
import com.playertour.backend.golfcourse.model.golfcourse.GolfCoursePackage;
import com.playertour.backend.player.model.player.PlayerPackage;
import com.playertour.backend.possiblecheats.model.possiblecheats.PossibleCheatReport;
import com.playertour.backend.possiblecheats.model.possiblecheats.PossibleCheatReportStatus;
import com.playertour.backend.possiblecheats.model.possiblecheats.PossibleCheatReportType;
import com.playertour.backend.possiblecheats.model.possiblecheats.PossibleCheatsPackage;
import com.playertour.backend.possiblecheats.service.api.PossibleCheatsReportingService;
import com.playertour.backend.possiblecheats.service.tests.helper.PossibleCheatsReportingTestHelper;

@Testable
@ExtendWith(BundleContextExtension.class)
@ExtendWith(ServiceExtension.class)
@TestMethodOrder(MethodOrderer.OrderAnnotation.class)
public class PossibleCheatsReportingIntegrationTest {
	
	private static final String PCR_STROKE_LANDING_ZONE_DETAILS = "Detected possible cheat of type %s: stroke no. %d landed on %s";
	
	@InjectBundleContext
	BundleContext bundleContext;

	@Order(value = -1)
	@Test
	public void testServices(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PossibleCheatsReportingService> possibleCheatsReportingServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerServiceAware, 
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerScorecardService> playerScorecardServiceAware, 
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<CourseService> courseServiceAware) {

		assertThat(possibleCheatsReportingServiceAware.getServices()).hasSize(1);
		ServiceReference<PossibleCheatsReportingService> possibleCheatsReportingServiceReference = possibleCheatsReportingServiceAware
				.getServiceReference();
		assertThat(possibleCheatsReportingServiceReference).isNotNull();

		assertThat(playerServiceAware.getServices()).hasSize(1);
		ServiceReference<PlayerService> playerServiceReference = playerServiceAware.getServiceReference();
		assertThat(playerServiceReference).isNotNull();
		
		assertThat(playerScorecardServiceAware.getServices()).hasSize(1);
		ServiceReference<PlayerScorecardService> playerScorecardServiceReference = playerScorecardServiceAware.getServiceReference();
		assertThat(playerScorecardServiceReference).isNotNull();
		
		assertThat(courseServiceAware.getServices()).hasSize(1);
		ServiceReference<CourseService> courseServiceReference = courseServiceAware.getServiceReference();
		assertThat(courseServiceReference).isNotNull();		
	}

	@Test
	public void testSavePossibleCheatReport(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PossibleCheatsReportingService> possibleCheatsReportingServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerServiceAware, 
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerScorecardService> playerScorecardServiceAware, 
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<CourseService> courseServiceAware)
			throws InterruptedException {
		
		assertThat(possibleCheatsReportingServiceAware.getServices()).hasSize(1);
		PossibleCheatsReportingService possibleCheatsReportingService = possibleCheatsReportingServiceAware.getService();
		assertThat(possibleCheatsReportingService).isNotNull();
		
		assertThat(playerServiceAware.getServices()).hasSize(1);
		PlayerService playerService = playerServiceAware.getService();
		assertThat(playerService).isNotNull();

		assertThat(playerScorecardServiceAware.getServices()).hasSize(1);	
		PlayerScorecardService playerScorecardService = playerScorecardServiceAware.getService();
		assertThat(playerScorecardService).isNotNull();	
		
		assertThat(courseServiceAware.getServices()).hasSize(1);
		CourseService courseService = courseServiceAware.getService();
		assertThat(courseService).isNotNull();
		
		String loginId = UUID.randomUUID().toString();
		
		String playerScorecardId = UUID.randomUUID().toString();
				
		PossibleCheatReportStatus status = PossibleCheatReportStatus.NEW;
		PossibleCheatReportType type = PossibleCheatReportType.GREEN_HITS_PAR4S_RESULT;
		int strokeNumber = 1;
		LandingZone landingZone = LandingZone.GREEN;
		long timeStamp = LocalDateTime.now().atZone(ZoneOffset.systemDefault()).toEpochSecond();
		
		// @formatter:off
		PossibleCheatReport possibleCheatReport = PossibleCheatsReportingTestHelper.constructPossibleCheatReport(
				loginId, 
				playerScorecardId,
				status, 
				type, 
				String.format(PCR_STROKE_LANDING_ZONE_DETAILS, 
						type.getLiteral(), 
						strokeNumber, 
						landingZone), 
				timeStamp);
		// @formatter:on
		
		possibleCheatReport = possibleCheatsReportingService.savePossibleCheatReport(possibleCheatReport);
		assertThat(possibleCheatReport).isNotNull();
		assertThat(possibleCheatReport.getId()).isNotNull();
	}

	@Test
	public void testGetPossibleCheatsReports(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PossibleCheatsReportingService> possibleCheatsReportingServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerServiceAware, 
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerScorecardService> playerScorecardServiceAware, 
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<CourseService> courseServiceAware)
			throws InterruptedException {

		assertThat(possibleCheatsReportingServiceAware.getServices()).hasSize(1);
		PossibleCheatsReportingService possibleCheatsReportingService = possibleCheatsReportingServiceAware.getService();
		assertThat(possibleCheatsReportingService).isNotNull();
		
		assertThat(playerServiceAware.getServices()).hasSize(1);
		PlayerService playerService = playerServiceAware.getService();
		assertThat(playerService).isNotNull();

		assertThat(playerScorecardServiceAware.getServices()).hasSize(1);	
		PlayerScorecardService playerScorecardService = playerScorecardServiceAware.getService();
		assertThat(playerScorecardService).isNotNull();	
		
		assertThat(courseServiceAware.getServices()).hasSize(1);
		CourseService courseService = courseServiceAware.getService();
		assertThat(courseService).isNotNull();
		
		String loginId = UUID.randomUUID().toString();
		
		String playerScorecardId = UUID.randomUUID().toString();
				
		PossibleCheatReportStatus status = PossibleCheatReportStatus.NEW;
		PossibleCheatReportType type = PossibleCheatReportType.GREEN_HITS_PAR4S_RESULT;
		int strokeNumber = 1;
		LandingZone landingZone = LandingZone.GREEN;
		long timeStamp = LocalDateTime.now().atZone(ZoneOffset.systemDefault()).toEpochSecond();
		
		// @formatter:off
		PossibleCheatReport possibleCheatReport = PossibleCheatsReportingTestHelper.constructPossibleCheatReport(
				loginId, 
				playerScorecardId,
				status, 
				type, 
				String.format(PCR_STROKE_LANDING_ZONE_DETAILS, 
						type.getLiteral(), 
						strokeNumber, 
						landingZone), 
				timeStamp);
		// @formatter:on
		
		possibleCheatReport = possibleCheatsReportingService.savePossibleCheatReport(possibleCheatReport);
		assertThat(possibleCheatReport).isNotNull();
		assertThat(possibleCheatReport.getId()).isNotNull();
		
		Thread.sleep(2000);
		
		List<PossibleCheatReport> possibleCheatsReports = possibleCheatsReportingService.getPossibleCheatsReports(loginId);
		assertThat(possibleCheatsReports).isNotNull();
		assertThat(possibleCheatsReports).hasSize(1);
	}

	@Test
	public void testGetPossibleCheatsReportsWithStatus(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PossibleCheatsReportingService> possibleCheatsReportingServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerServiceAware, 
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerScorecardService> playerScorecardServiceAware, 
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<CourseService> courseServiceAware)
			throws InterruptedException {

		assertThat(possibleCheatsReportingServiceAware.getServices()).hasSize(1);
		PossibleCheatsReportingService possibleCheatsReportingService = possibleCheatsReportingServiceAware.getService();
		assertThat(possibleCheatsReportingService).isNotNull();
		
		assertThat(playerServiceAware.getServices()).hasSize(1);
		PlayerService playerService = playerServiceAware.getService();
		assertThat(playerService).isNotNull();

		assertThat(playerScorecardServiceAware.getServices()).hasSize(1);	
		PlayerScorecardService playerScorecardService = playerScorecardServiceAware.getService();
		assertThat(playerScorecardService).isNotNull();	
		
		assertThat(courseServiceAware.getServices()).hasSize(1);
		CourseService courseService = courseServiceAware.getService();
		assertThat(courseService).isNotNull();
		
		String loginId = UUID.randomUUID().toString();
		
		String playerScorecardId = UUID.randomUUID().toString();
				
		PossibleCheatReportStatus status = PossibleCheatReportStatus.NEW;
		PossibleCheatReportType type = PossibleCheatReportType.GREEN_HITS_PAR4S_RESULT;
		int strokeNumber = 1;
		LandingZone landingZone = LandingZone.GREEN;
		long timeStamp = LocalDateTime.now().atZone(ZoneOffset.systemDefault()).toEpochSecond();
		
		// @formatter:off
		PossibleCheatReport possibleCheatReport = PossibleCheatsReportingTestHelper.constructPossibleCheatReport(
				loginId, 
				playerScorecardId,
				status, 
				type, 
				String.format(PCR_STROKE_LANDING_ZONE_DETAILS, 
						type.getLiteral(), 
						strokeNumber, 
						landingZone), 
				timeStamp);
		// @formatter:on
		
		possibleCheatReport = possibleCheatsReportingService.savePossibleCheatReport(possibleCheatReport);
		assertThat(possibleCheatReport).isNotNull();
		assertThat(possibleCheatReport.getId()).isNotNull();
		
		Thread.sleep(2000);
		
		List<PossibleCheatReport> existingPossibleCheatsReports = possibleCheatsReportingService.getPossibleCheatsReportsWithStatus(loginId, status);
		assertThat(existingPossibleCheatsReports).isNotNull();
		assertThat(existingPossibleCheatsReports).hasSize(1);
		
		List<PossibleCheatReport> nonExistingPossibleCheatsReports = possibleCheatsReportingService.getPossibleCheatsReportsWithStatus(loginId, PossibleCheatReportStatus.CLOSED);
		assertThat(nonExistingPossibleCheatsReports).isNotNull();
		assertThat(nonExistingPossibleCheatsReports).hasSize(0);
	}

	@Test
	public void testGetPossibleCheatsReportsWithType(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PossibleCheatsReportingService> possibleCheatsReportingServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerServiceAware, 
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerScorecardService> playerScorecardServiceAware, 
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<CourseService> courseServiceAware)
			throws InterruptedException {
		
		assertThat(possibleCheatsReportingServiceAware.getServices()).hasSize(1);
		PossibleCheatsReportingService possibleCheatsReportingService = possibleCheatsReportingServiceAware.getService();
		assertThat(possibleCheatsReportingService).isNotNull();
		
		assertThat(playerServiceAware.getServices()).hasSize(1);
		PlayerService playerService = playerServiceAware.getService();
		assertThat(playerService).isNotNull();

		assertThat(playerScorecardServiceAware.getServices()).hasSize(1);	
		PlayerScorecardService playerScorecardService = playerScorecardServiceAware.getService();
		assertThat(playerScorecardService).isNotNull();	
		
		assertThat(courseServiceAware.getServices()).hasSize(1);
		CourseService courseService = courseServiceAware.getService();
		assertThat(courseService).isNotNull();
		
		String loginId = UUID.randomUUID().toString();
		
		String playerScorecardId = UUID.randomUUID().toString();
				
		PossibleCheatReportStatus status = PossibleCheatReportStatus.NEW;
		PossibleCheatReportType type = PossibleCheatReportType.GREEN_HITS_PAR4S_RESULT;
		int strokeNumber = 1;
		LandingZone landingZone = LandingZone.GREEN;
		long timeStamp = LocalDateTime.now().atZone(ZoneOffset.systemDefault()).toEpochSecond();
		
		// @formatter:off
		PossibleCheatReport possibleCheatReport = PossibleCheatsReportingTestHelper.constructPossibleCheatReport(
				loginId, 
				playerScorecardId,
				status, 
				type, 
				String.format(PCR_STROKE_LANDING_ZONE_DETAILS, 
						type.getLiteral(), 
						strokeNumber, 
						landingZone), 
				timeStamp);
		// @formatter:on
		
		possibleCheatReport = possibleCheatsReportingService.savePossibleCheatReport(possibleCheatReport);
		assertThat(possibleCheatReport).isNotNull();
		assertThat(possibleCheatReport.getId()).isNotNull();
		
		Thread.sleep(2000);
		
		List<PossibleCheatReport> existingPossibleCheatsReports = possibleCheatsReportingService.getPossibleCheatsReportsWithType(loginId, type);
		assertThat(existingPossibleCheatsReports).isNotNull();
		assertThat(existingPossibleCheatsReports).hasSize(1);
		
		List<PossibleCheatReport> nonExistingPossibleCheatsReports = possibleCheatsReportingService.getPossibleCheatsReportsWithType(loginId, PossibleCheatReportType.GREEN_HITS_PAR5S_RESULT);
		assertThat(nonExistingPossibleCheatsReports).isNotNull();
		assertThat(nonExistingPossibleCheatsReports).hasSize(0);		
	}

	@Test
	public void testGetPossibleCheatsReportsWithStatusAndType(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PossibleCheatsReportingService> possibleCheatsReportingServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerServiceAware, 
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerScorecardService> playerScorecardServiceAware, 
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<CourseService> courseServiceAware)
			throws InterruptedException {
		
		assertThat(possibleCheatsReportingServiceAware.getServices()).hasSize(1);
		PossibleCheatsReportingService possibleCheatsReportingService = possibleCheatsReportingServiceAware.getService();
		assertThat(possibleCheatsReportingService).isNotNull();
		
		assertThat(playerServiceAware.getServices()).hasSize(1);
		PlayerService playerService = playerServiceAware.getService();
		assertThat(playerService).isNotNull();

		assertThat(playerScorecardServiceAware.getServices()).hasSize(1);	
		PlayerScorecardService playerScorecardService = playerScorecardServiceAware.getService();
		assertThat(playerScorecardService).isNotNull();	
		
		assertThat(courseServiceAware.getServices()).hasSize(1);
		CourseService courseService = courseServiceAware.getService();
		assertThat(courseService).isNotNull();
		
		String loginId = UUID.randomUUID().toString();
		
		String playerScorecardId = UUID.randomUUID().toString();
				
		PossibleCheatReportStatus status = PossibleCheatReportStatus.NEW;
		PossibleCheatReportType type = PossibleCheatReportType.GREEN_HITS_PAR4S_RESULT;
		int strokeNumber = 1;
		LandingZone landingZone = LandingZone.GREEN;
		long timeStamp = LocalDateTime.now().atZone(ZoneOffset.systemDefault()).toEpochSecond();
		
		// @formatter:off
		PossibleCheatReport possibleCheatReport = PossibleCheatsReportingTestHelper.constructPossibleCheatReport(
				loginId, 
				playerScorecardId,
				status, 
				type, 
				String.format(PCR_STROKE_LANDING_ZONE_DETAILS, 
						type.getLiteral(), 
						strokeNumber, 
						landingZone), 
				timeStamp);
		// @formatter:on
		
		possibleCheatReport = possibleCheatsReportingService.savePossibleCheatReport(possibleCheatReport);
		assertThat(possibleCheatReport).isNotNull();
		assertThat(possibleCheatReport.getId()).isNotNull();
		
		Thread.sleep(2000);
		
		List<PossibleCheatReport> existingPossibleCheatsReports = possibleCheatsReportingService.getPossibleCheatsReportsWithStatusAndType(loginId, status, type);
		assertThat(existingPossibleCheatsReports).isNotNull();
		assertThat(existingPossibleCheatsReports).hasSize(1);
		
		List<PossibleCheatReport> nonExistingPossibleCheatsReports = possibleCheatsReportingService.getPossibleCheatsReportsWithStatusAndType(loginId, PossibleCheatReportStatus.CLOSED, PossibleCheatReportType.GREEN_HITS_PAR4S_RESULT);
		assertThat(nonExistingPossibleCheatsReports).isNotNull();
		assertThat(nonExistingPossibleCheatsReports).hasSize(0);
	}

	@Test
	public void testGetPossibleCheatsReportsByScorecardId(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PossibleCheatsReportingService> possibleCheatsReportingServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerServiceAware, 
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerScorecardService> playerScorecardServiceAware, 
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<CourseService> courseServiceAware)
			throws InterruptedException {

		assertThat(possibleCheatsReportingServiceAware.getServices()).hasSize(1);
		PossibleCheatsReportingService possibleCheatsReportingService = possibleCheatsReportingServiceAware.getService();
		assertThat(possibleCheatsReportingService).isNotNull();
		
		assertThat(playerServiceAware.getServices()).hasSize(1);
		PlayerService playerService = playerServiceAware.getService();
		assertThat(playerService).isNotNull();

		assertThat(playerScorecardServiceAware.getServices()).hasSize(1);	
		PlayerScorecardService playerScorecardService = playerScorecardServiceAware.getService();
		assertThat(playerScorecardService).isNotNull();	
		
		assertThat(courseServiceAware.getServices()).hasSize(1);
		CourseService courseService = courseServiceAware.getService();
		assertThat(courseService).isNotNull();
		
		String loginId = UUID.randomUUID().toString();
		
		String playerScorecardId = UUID.randomUUID().toString();
				
		PossibleCheatReportStatus status = PossibleCheatReportStatus.NEW;
		PossibleCheatReportType type = PossibleCheatReportType.GREEN_HITS_PAR4S_RESULT;
		int strokeNumber = 1;
		LandingZone landingZone = LandingZone.GREEN;
		long timeStamp = LocalDateTime.now().atZone(ZoneOffset.systemDefault()).toEpochSecond();
		
		// @formatter:off
		PossibleCheatReport possibleCheatReport = PossibleCheatsReportingTestHelper.constructPossibleCheatReport(
				loginId, 
				playerScorecardId,
				status, 
				type, 
				String.format(PCR_STROKE_LANDING_ZONE_DETAILS, 
						type.getLiteral(), 
						strokeNumber, 
						landingZone), 
				timeStamp);
		// @formatter:on
		
		possibleCheatReport = possibleCheatsReportingService.savePossibleCheatReport(possibleCheatReport);
		assertThat(possibleCheatReport).isNotNull();
		assertThat(possibleCheatReport.getId()).isNotNull();
		
		Thread.sleep(2000);
		
		List<PossibleCheatReport> possibleCheatsReports = possibleCheatsReportingService.getPossibleCheatsReportsByScorecardId(loginId, playerScorecardId);
		assertThat(possibleCheatsReports).isNotNull();
		assertThat(possibleCheatsReports).hasSize(1);	
	}

	@Test
	public void testGetPossibleCheatsReportsByScorecardIdWithStatus(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PossibleCheatsReportingService> possibleCheatsReportingServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerServiceAware, 
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerScorecardService> playerScorecardServiceAware, 
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<CourseService> courseServiceAware)
			throws InterruptedException {
		
		assertThat(possibleCheatsReportingServiceAware.getServices()).hasSize(1);
		PossibleCheatsReportingService possibleCheatsReportingService = possibleCheatsReportingServiceAware.getService();
		assertThat(possibleCheatsReportingService).isNotNull();
		
		assertThat(playerServiceAware.getServices()).hasSize(1);
		PlayerService playerService = playerServiceAware.getService();
		assertThat(playerService).isNotNull();

		assertThat(playerScorecardServiceAware.getServices()).hasSize(1);	
		PlayerScorecardService playerScorecardService = playerScorecardServiceAware.getService();
		assertThat(playerScorecardService).isNotNull();	
		
		assertThat(courseServiceAware.getServices()).hasSize(1);
		CourseService courseService = courseServiceAware.getService();
		assertThat(courseService).isNotNull();
		
		String loginId = UUID.randomUUID().toString();
		
		String playerScorecardId = UUID.randomUUID().toString();
				
		PossibleCheatReportStatus status = PossibleCheatReportStatus.NEW;
		PossibleCheatReportType type = PossibleCheatReportType.GREEN_HITS_PAR4S_RESULT;
		int strokeNumber = 1;
		LandingZone landingZone = LandingZone.GREEN;
		long timeStamp = LocalDateTime.now().atZone(ZoneOffset.systemDefault()).toEpochSecond();
		
		// @formatter:off
		PossibleCheatReport possibleCheatReport = PossibleCheatsReportingTestHelper.constructPossibleCheatReport(
				loginId, 
				playerScorecardId,
				status, 
				type, 
				String.format(PCR_STROKE_LANDING_ZONE_DETAILS, 
						type.getLiteral(), 
						strokeNumber, 
						landingZone), 
				timeStamp);
		// @formatter:on
		
		possibleCheatReport = possibleCheatsReportingService.savePossibleCheatReport(possibleCheatReport);
		assertThat(possibleCheatReport).isNotNull();
		assertThat(possibleCheatReport.getId()).isNotNull();
		
		Thread.sleep(2000);
		
		List<PossibleCheatReport> existingPossibleCheatsReports = possibleCheatsReportingService.getPossibleCheatsReportsByScorecardIdWithStatus(loginId, playerScorecardId, status);
		assertThat(existingPossibleCheatsReports).isNotNull();
		assertThat(existingPossibleCheatsReports).hasSize(1);
		
		List<PossibleCheatReport> nonExistingPossibleCheatsReports = possibleCheatsReportingService.getPossibleCheatsReportsByScorecardIdWithStatus(loginId, playerScorecardId, PossibleCheatReportStatus.CLOSED);
		assertThat(nonExistingPossibleCheatsReports).isNotNull();
		assertThat(nonExistingPossibleCheatsReports).hasSize(0);
	}

	@Test
	public void testGetPossibleCheatsReportsByScorecardIdWithType(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PossibleCheatsReportingService> possibleCheatsReportingServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerServiceAware, 
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerScorecardService> playerScorecardServiceAware, 
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<CourseService> courseServiceAware)
			throws InterruptedException {
		
		assertThat(possibleCheatsReportingServiceAware.getServices()).hasSize(1);
		PossibleCheatsReportingService possibleCheatsReportingService = possibleCheatsReportingServiceAware.getService();
		assertThat(possibleCheatsReportingService).isNotNull();
		
		assertThat(playerServiceAware.getServices()).hasSize(1);
		PlayerService playerService = playerServiceAware.getService();
		assertThat(playerService).isNotNull();

		assertThat(playerScorecardServiceAware.getServices()).hasSize(1);	
		PlayerScorecardService playerScorecardService = playerScorecardServiceAware.getService();
		assertThat(playerScorecardService).isNotNull();	
		
		assertThat(courseServiceAware.getServices()).hasSize(1);
		CourseService courseService = courseServiceAware.getService();
		assertThat(courseService).isNotNull();
		
		String loginId = UUID.randomUUID().toString();
		
		String playerScorecardId = UUID.randomUUID().toString();
				
		PossibleCheatReportStatus status = PossibleCheatReportStatus.NEW;
		PossibleCheatReportType type = PossibleCheatReportType.GREEN_HITS_PAR4S_RESULT;
		int strokeNumber = 1;
		LandingZone landingZone = LandingZone.GREEN;
		long timeStamp = LocalDateTime.now().atZone(ZoneOffset.systemDefault()).toEpochSecond();
		
		// @formatter:off
		PossibleCheatReport possibleCheatReport = PossibleCheatsReportingTestHelper.constructPossibleCheatReport(
				loginId, 
				playerScorecardId,
				status, 
				type, 
				String.format(PCR_STROKE_LANDING_ZONE_DETAILS, 
						type.getLiteral(), 
						strokeNumber, 
						landingZone), 
				timeStamp);
		// @formatter:on
		
		possibleCheatReport = possibleCheatsReportingService.savePossibleCheatReport(possibleCheatReport);
		assertThat(possibleCheatReport).isNotNull();
		assertThat(possibleCheatReport.getId()).isNotNull();
		
		Thread.sleep(2000);		
		
		List<PossibleCheatReport> existingPossibleCheatsReports = possibleCheatsReportingService.getPossibleCheatsReportsByScorecardIdWithType(loginId, playerScorecardId, type);
		assertThat(existingPossibleCheatsReports).isNotNull();
		assertThat(existingPossibleCheatsReports).hasSize(1);
		
		List<PossibleCheatReport> nonExistingPossibleCheatsReports = possibleCheatsReportingService.getPossibleCheatsReportsByScorecardIdWithType(loginId, playerScorecardId, PossibleCheatReportType.GREEN_HITS_PAR5S_RESULT);
		assertThat(nonExistingPossibleCheatsReports).isNotNull();
		assertThat(nonExistingPossibleCheatsReports).hasSize(0);		
	}

	@Test
	public void testGetPossibleCheatsReportsByScorecardIdWithStatusAndType(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PossibleCheatsReportingService> possibleCheatsReportingServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerServiceAware, 
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerScorecardService> playerScorecardServiceAware, 
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<CourseService> courseServiceAware)
			throws InterruptedException {

		assertThat(possibleCheatsReportingServiceAware.getServices()).hasSize(1);
		PossibleCheatsReportingService possibleCheatsReportingService = possibleCheatsReportingServiceAware.getService();
		assertThat(possibleCheatsReportingService).isNotNull();
		
		assertThat(playerServiceAware.getServices()).hasSize(1);
		PlayerService playerService = playerServiceAware.getService();
		assertThat(playerService).isNotNull();

		assertThat(playerScorecardServiceAware.getServices()).hasSize(1);	
		PlayerScorecardService playerScorecardService = playerScorecardServiceAware.getService();
		assertThat(playerScorecardService).isNotNull();	
		
		assertThat(courseServiceAware.getServices()).hasSize(1);
		CourseService courseService = courseServiceAware.getService();
		assertThat(courseService).isNotNull();
		
		String loginId = UUID.randomUUID().toString();
		
		String playerScorecardId = UUID.randomUUID().toString();
				
		PossibleCheatReportStatus status = PossibleCheatReportStatus.NEW;
		PossibleCheatReportType type = PossibleCheatReportType.GREEN_HITS_PAR4S_RESULT;
		int strokeNumber = 1;
		LandingZone landingZone = LandingZone.GREEN;
		long timeStamp = LocalDateTime.now().atZone(ZoneOffset.systemDefault()).toEpochSecond();
		
		// @formatter:off
		PossibleCheatReport possibleCheatReport = PossibleCheatsReportingTestHelper.constructPossibleCheatReport(
				loginId, 
				playerScorecardId,
				status, 
				type, 
				String.format(PCR_STROKE_LANDING_ZONE_DETAILS, 
						type.getLiteral(), 
						strokeNumber, 
						landingZone), 
				timeStamp);
		// @formatter:on
		
		possibleCheatReport = possibleCheatsReportingService.savePossibleCheatReport(possibleCheatReport);
		assertThat(possibleCheatReport).isNotNull();
		assertThat(possibleCheatReport.getId()).isNotNull();
		
		Thread.sleep(2000);		
		
		List<PossibleCheatReport> existingPossibleCheatsReports = possibleCheatsReportingService.getPossibleCheatsReportsByScorecardIdWithStatusAndType(loginId, playerScorecardId, status, type);
		assertThat(existingPossibleCheatsReports).isNotNull();
		assertThat(existingPossibleCheatsReports).hasSize(1);
		
		List<PossibleCheatReport> nonExistingPossibleCheatsReports = possibleCheatsReportingService.getPossibleCheatsReportsByScorecardIdWithStatusAndType(loginId, playerScorecardId, PossibleCheatReportStatus.CLOSED, PossibleCheatReportType.GREEN_HITS_PAR4S_RESULT);
		assertThat(nonExistingPossibleCheatsReports).isNotNull();
		assertThat(nonExistingPossibleCheatsReports).hasSize(0);		
	}
	
	@Test
	public void testGetPossibleCheatsReport(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PossibleCheatsReportingService> possibleCheatsReportingServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerServiceAware, 
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerScorecardService> playerScorecardServiceAware, 
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<CourseService> courseServiceAware)
			throws InterruptedException {
		
		assertThat(possibleCheatsReportingServiceAware.getServices()).hasSize(1);
		PossibleCheatsReportingService possibleCheatsReportingService = possibleCheatsReportingServiceAware.getService();
		assertThat(possibleCheatsReportingService).isNotNull();
		
		assertThat(playerServiceAware.getServices()).hasSize(1);
		PlayerService playerService = playerServiceAware.getService();
		assertThat(playerService).isNotNull();

		assertThat(playerScorecardServiceAware.getServices()).hasSize(1);	
		PlayerScorecardService playerScorecardService = playerScorecardServiceAware.getService();
		assertThat(playerScorecardService).isNotNull();	
		
		assertThat(courseServiceAware.getServices()).hasSize(1);
		CourseService courseService = courseServiceAware.getService();
		assertThat(courseService).isNotNull();
		
		String loginId = UUID.randomUUID().toString();
		
		String playerScorecardId = UUID.randomUUID().toString();
				
		PossibleCheatReportStatus status = PossibleCheatReportStatus.NEW;
		PossibleCheatReportType type = PossibleCheatReportType.GREEN_HITS_PAR4S_RESULT;
		int strokeNumber = 1;
		LandingZone landingZone = LandingZone.GREEN;
		long timeStamp = LocalDateTime.now().atZone(ZoneOffset.systemDefault()).toEpochSecond();
		
		// @formatter:off
		PossibleCheatReport possibleCheatReport = PossibleCheatsReportingTestHelper.constructPossibleCheatReport(
				loginId, 
				playerScorecardId,
				status, 
				type, 
				String.format(PCR_STROKE_LANDING_ZONE_DETAILS, 
						type.getLiteral(), 
						strokeNumber, 
						landingZone), 
				timeStamp);
		// @formatter:on
		
		possibleCheatReport = possibleCheatsReportingService.savePossibleCheatReport(possibleCheatReport);
		assertThat(possibleCheatReport).isNotNull();
		assertThat(possibleCheatReport.getId()).isNotNull();
		
		Thread.sleep(2000);
		
		possibleCheatReport = possibleCheatsReportingService.getPossibleCheatsReport(possibleCheatReport.getId());
		assertThat(possibleCheatReport).isNotNull();
	}

	@Test
	public void testUpdateStatus(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PossibleCheatsReportingService> possibleCheatsReportingServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerServiceAware, 
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerScorecardService> playerScorecardServiceAware, 
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<CourseService> courseServiceAware)
			throws InterruptedException {
		
		assertThat(possibleCheatsReportingServiceAware.getServices()).hasSize(1);
		PossibleCheatsReportingService possibleCheatsReportingService = possibleCheatsReportingServiceAware.getService();
		assertThat(possibleCheatsReportingService).isNotNull();
		
		assertThat(playerServiceAware.getServices()).hasSize(1);
		PlayerService playerService = playerServiceAware.getService();
		assertThat(playerService).isNotNull();

		assertThat(playerScorecardServiceAware.getServices()).hasSize(1);	
		PlayerScorecardService playerScorecardService = playerScorecardServiceAware.getService();
		assertThat(playerScorecardService).isNotNull();	
		
		assertThat(courseServiceAware.getServices()).hasSize(1);
		CourseService courseService = courseServiceAware.getService();
		assertThat(courseService).isNotNull();
		
		String loginId = UUID.randomUUID().toString();
		
		String playerScorecardId = UUID.randomUUID().toString();
				
		PossibleCheatReportStatus status = PossibleCheatReportStatus.NEW;
		PossibleCheatReportType type = PossibleCheatReportType.GREEN_HITS_PAR4S_RESULT;
		int strokeNumber = 1;
		LandingZone landingZone = LandingZone.GREEN;
		long timeStamp = LocalDateTime.now().atZone(ZoneOffset.systemDefault()).toEpochSecond();
		
		// @formatter:off
		PossibleCheatReport possibleCheatReport = PossibleCheatsReportingTestHelper.constructPossibleCheatReport(
				loginId, 
				playerScorecardId,
				status, 
				type, 
				String.format(PCR_STROKE_LANDING_ZONE_DETAILS, 
						type.getLiteral(), 
						strokeNumber, 
						landingZone), 
				timeStamp);
		// @formatter:on
		
		possibleCheatReport = possibleCheatsReportingService.savePossibleCheatReport(possibleCheatReport);
		assertThat(possibleCheatReport).isNotNull();
		assertThat(possibleCheatReport.getId()).isNotNull();
		
		Thread.sleep(2000);	
		
		possibleCheatReport = possibleCheatsReportingService.updateStatus(possibleCheatReport.getId(), PossibleCheatReportStatus.OPEN);
		assertThat(possibleCheatReport).isNotNull();
		assertThat(possibleCheatReport.getStatus()).isEqualTo(PossibleCheatReportStatus.OPEN);
	}
	
	@Test
	public void testDetectPossibleCheats(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PossibleCheatsReportingService> possibleCheatsReportingServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerServiceAware, 
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerScorecardService> playerScorecardServiceAware, 
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<CourseService> courseServiceAware)
			throws InterruptedException {
		
		assertThat(possibleCheatsReportingServiceAware.getServices()).hasSize(1);
		PossibleCheatsReportingService possibleCheatsReportingService = possibleCheatsReportingServiceAware.getService();
		assertThat(possibleCheatsReportingService).isNotNull();
		
		assertThat(playerServiceAware.getServices()).hasSize(1);
		PlayerService playerService = playerServiceAware.getService();
		assertThat(playerService).isNotNull();

		assertThat(playerScorecardServiceAware.getServices()).hasSize(1);	
		PlayerScorecardService playerScorecardService = playerScorecardServiceAware.getService();
		assertThat(playerScorecardService).isNotNull();	
		
		assertThat(courseServiceAware.getServices()).hasSize(1);
		CourseService courseService = courseServiceAware.getService();
		assertThat(courseService).isNotNull();
		
		// 1. Create a player		
		String loginId = UUID.randomUUID().toString();
		String loginName = "mario.rossi";
		PlayerProfile playerProfile = playerService.getPlayerProfile(loginId, loginName);	
		assertThat(playerProfile).isNotNull();
		
		// 2. Create a GolfCourse
		GenderType gender = GenderType.MEN;
		String courseId = UUID.randomUUID().toString();
		GolfCourse golfCourse = PossibleCheatsReportingTestHelper.constructGolfCourse(courseId, gender);
		golfCourse = courseService.saveCourse(golfCourse);
		assertThat(golfCourse).isNotNull();
		assertThat(golfCourse.getId()).isNotNull();
		
		//String courseMongoId = golfCourse.getId();
		int holesTotal = golfCourse.getCourseDetails().getHoleNum();
		
		Thread.sleep(2000);
		
		// 3. Open a PlayerScorecard, generate strokes, close it
		PlayerScorecard openedPlayerScorecard = playerScorecardService.openScorecard(loginId, courseId, gender);
		assertThat(openedPlayerScorecard).isNotNull();
		assertThat(openedPlayerScorecard.getId()).isNotNull();
		assertThat(openedPlayerScorecard.getStart()).isNotNull();
		assertThat(openedPlayerScorecard.getEnd()).isNull();
		assertThat(openedPlayerScorecard.getCourseScorecardInfo()).isNotNull();
		assertThat(openedPlayerScorecard.getCourseId()).isEqualTo(courseId);
		assertThat(openedPlayerScorecard.getHoleStats()).isNotNull();
		assertThat(openedPlayerScorecard.getHoleStats()).isNotEmpty();
		assertThat(openedPlayerScorecard.getHoleStats()).hasSize(holesTotal);
		
		String playerScorecardId = openedPlayerScorecard.getId();
		assertThat(playerScorecardId).isNotNull();
				
		Thread.sleep(2000);
		
		PlayerScorecard updatedPlayerScorecard = PossibleCheatsReportingTestHelper.generateStrokes(loginId, playerScorecardId, openedPlayerScorecard, playerScorecardService);
		assertThat(updatedPlayerScorecard).isNotNull();
		
		Thread.sleep(2000);
		
		PlayerScorecard closedPlayerScorecard = playerScorecardService.closeScorecard(loginId, playerScorecardId);
		assertThat(closedPlayerScorecard).isNotNull();
		assertThat(closedPlayerScorecard.getEnd()).isNotNull();
		assertThat(closedPlayerScorecard.isNoReturn()).isFalse();
		
		Thread.sleep(2000); // in the meantime, specialized event handler will detecet and record possible cheats...
		
		List<PossibleCheatReport> possibleCheatsReports = possibleCheatsReportingService.getPossibleCheatsReports(loginId);
		assertThat(possibleCheatsReports).isNotNull();
		assertThat(possibleCheatsReports).hasSize(4);
		
		List<PossibleCheatReport> possibleCheatsReportsByScorecardId = possibleCheatsReportingService.getPossibleCheatsReportsByScorecardId(loginId, playerScorecardId);
		assertThat(possibleCheatsReportsByScorecardId).isNotNull();
		assertThat(possibleCheatsReportsByScorecardId).hasSize(4);	
	}	

	@BeforeEach
	@AfterEach
	public void clean(
			@InjectService(cardinality = 1, timeout = 5000, filter = "(database=playertour)") ServiceAware<MongoDatabaseProvider> dbProviderAware,
			@InjectService(cardinality = 1, timeout = 5000, filter = "(component.name=PossibleCheatsReportingIndexService)") ServiceAware<CommonIndexService> possibleCheatsReportingIndexServiceAware,
			@InjectService(cardinality = 1, timeout = 5000, filter = "(component.name=PlayerIndexService)") ServiceAware<CommonIndexService> playerIndexServiceAware, 
			@InjectService(cardinality = 1, timeout = 5000, filter = "(component.name=CourseIndexService)") ServiceAware<CommonIndexService> courseIndexServiceAware) {

		assertThat(dbProviderAware.getServices()).hasSize(1);
		MongoDatabaseProvider dbProvider = dbProviderAware.getService();
		assertThat(dbProvider).isNotNull();

		MongoDatabase database = dbProvider.getDatabase();
		try {
			database.getCollection(PlayerPackage.Literals.PLAYER.getName()).drop();
			database.getCollection(GolfCoursePackage.Literals.GOLF_COURSE.getName()).drop();
			database.getCollection(PossibleCheatsPackage.Literals.POSSIBLE_CHEAT_REPORT.getName()).drop();
		} catch (IllegalArgumentException e) {
			System.out.println("Collection does not exist. Nothing to remove.");
		}

		assertThat(playerIndexServiceAware.getServices()).hasSize(1);
		CommonIndexService playerIndexService = playerIndexServiceAware.getService();
		assertThat(playerIndexService).isNotNull();

		playerIndexService.resetIndex();
		
		assertThat(courseIndexServiceAware.getServices()).hasSize(1);	
		CommonIndexService courseIndexService = courseIndexServiceAware.getService();
		assertThat(courseIndexService).isNotNull();		
		
		courseIndexService.resetIndex();

		assertThat(possibleCheatsReportingIndexServiceAware.getServices()).hasSize(1);
		CommonIndexService possibleCheatsReportingIndexService = possibleCheatsReportingIndexServiceAware.getService();
		assertThat(possibleCheatsReportingIndexService).isNotNull();

		possibleCheatsReportingIndexService.resetIndex();
	}

	@AfterAll
	public static void brutalClean(
			@InjectService(cardinality = 1, timeout = 5000, filter = "(database=playertour)") ServiceAware<MongoDatabaseProvider> dbProviderAware) {
		assertThat(dbProviderAware.getServices()).hasSize(1);
		MongoDatabaseProvider dbProvider = dbProviderAware.getService();
		assertThat(dbProvider).isNotNull();

		MongoDatabase database = dbProvider.getDatabase();
		database.drop();
	}
}
