/**
 * Copyright (c) 2012 - 2022 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package com.playertour.backend.purchases.remoteconfig.service.impl;

import java.util.concurrent.CountDownLatch;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;

import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.annotations.ServiceScope;
import org.osgi.service.component.propertytypes.ServiceRanking;
import org.osgi.service.log.Logger;
import org.osgi.service.log.LoggerFactory;
import org.osgi.util.promise.Promise;
import org.osgi.util.promise.PromiseFactory;

import com.playertour.backend.config.firebaseremoteconfig.service.api.FirebaseRemoteConfigService;
import com.playertour.backend.purchases.remoteconfig.service.api.PurchasesRemoteConfigService;
import com.playertour.backend.purchases.service.api.PurchasesAPIConfigService;

@Component(name = "PurchasesRemoteConfigService", service = {
		PurchasesRemoteConfigService.class }, scope = ServiceScope.SINGLETON, immediate = true)
@ServiceRanking(Integer.MAX_VALUE - 400)
public class PurchasesRemoteConfigServiceImpl implements PurchasesRemoteConfigService {
	private static final int DELAY = 100; // ms

	private static final String FIREBASE_REMOTE_CONFIG_GROUP_NAME = "Promotions";

	private static final String FIREBASE_REMOTE_CONFIG_PARAMETER_NAME = "promotions_doublingpoints_enabled";

	private static final String FIREBASE_REMOTE_CONFIG_CONDITION_NAME = "before_date";

	private static final String FIREBASE_REMOTE_CONFIG_PARAMETER_CONDITIONAL_VALUE_NAME = FIREBASE_REMOTE_CONFIG_CONDITION_NAME;

	@Reference(service = LoggerFactory.class)
	private Logger logger;

	@Reference
	private PurchasesAPIConfigService purchasesAPIConfigService;

	@Reference
	private FirebaseRemoteConfigService firebaseRemoteConfigService;

	private PromiseFactory factory = new PromiseFactory(Executors.newSingleThreadExecutor());

	@Activate
	public void activate() {
		factory.submit(() -> {
			CountDownLatch latch = new CountDownLatch(1);
			latch.await(DELAY, TimeUnit.MILLISECONDS);
			synchronizeConfig();
			return true;
		}).onFailure(t -> t.printStackTrace());
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.purchases.remoteconfig.service.api.PurchasesRemoteConfigService#synchronizeConfig()
	 */
	@Override
	public Promise<Boolean> synchronizeConfig() {
		logger.info("Starting purchases config synchronization");
		
		String promotionEnabledBeforeDateTime = purchasesAPIConfigService.getPromotionEnabledBeforeDateTime();

		boolean isPromotionEnabled = purchasesAPIConfigService.isPromotionEnabled();

		boolean isParamValSuccessfullySet = firebaseRemoteConfigService.setBooleanParameterValue(
				FIREBASE_REMOTE_CONFIG_GROUP_NAME, FIREBASE_REMOTE_CONFIG_PARAMETER_NAME,
				FIREBASE_REMOTE_CONFIG_PARAMETER_CONDITIONAL_VALUE_NAME, isPromotionEnabled, false);

		boolean isDateTimeConSuccessfullySet = firebaseRemoteConfigService
				.setDateTimeCondition(FIREBASE_REMOTE_CONFIG_CONDITION_NAME, promotionEnabledBeforeDateTime);

		if (isParamValSuccessfullySet && isDateTimeConSuccessfullySet) {
			logger.info("Successfully synchronized purchases configuration with Firebase Remote Config");
			return factory.resolved(true);
		} else {
			logger.error("Purchases config synchronization failed");
			return factory.resolved(false);
		}
	}
}
