/**
 * Copyright (c) 2012 - 2022 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package com.playertour.backend.purchases.service.impl;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.Objects;
import java.util.Optional;

import org.eclipse.emf.ecore.util.EcoreUtil;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.annotations.ReferenceScope;
import org.osgi.service.component.annotations.ServiceScope;
import org.osgi.service.event.annotations.RequireEventAdmin;
import org.osgi.service.log.Logger;
import org.osgi.service.log.LoggerFactory;

import com.playertour.backend.meritpoints.service.api.MeritPointsService;
import com.playertour.backend.purchases.model.purchases.Purchase;
import com.playertour.backend.purchases.model.purchases.PurchasePromotion;
import com.playertour.backend.purchases.model.purchases.PurchasesFactory;
import com.playertour.backend.purchases.service.api.PurchasesAPIConfigService;
import com.playertour.backend.purchases.service.api.PurchasesIAPsService;
import com.playertour.backend.purchases.service.api.PurchasesPromotionService;
import com.playertour.backend.purchases.service.api.PurchasesPurchaseService;

@Component(name = "PurchasesPromotionService", scope = ServiceScope.PROTOTYPE)
@RequireEventAdmin
public class PurchasesPromotionServiceImpl implements PurchasesPromotionService {
	
	@Reference(service = LoggerFactory.class)
	private Logger logger;
	
	@Reference
	private PurchasesAPIConfigService purchasesAPIConfigService;	
	
	@Reference(scope = ReferenceScope.PROTOTYPE_REQUIRED)
	private PurchasesPurchaseService purchasesPurchaseService;	
	
	@Reference(scope = ReferenceScope.PROTOTYPE_REQUIRED)
	private PurchasesIAPsService purchasesIAPsService;
		
	@Reference(scope = ReferenceScope.PROTOTYPE_REQUIRED)
	private MeritPointsService meritPointsService;
	
	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.purchases.service.api.PurchasesPromotionService#applyPromotionIfEnabled(java.lang.String)
	 */
	@Override
	public Optional<PurchasePromotion> applyPromotionIfEnabled(String orderID) {
		Objects.requireNonNull(orderID, "Purchase order ID is required!");
		
		// check if enabled is made on each call instead of during activation -
		// otherwise configuration could not be changed without restarting the app,
		// which defeats the purpose of using remote config in the first place
		if (isPromotionEnabled()) {
			logger.debug("Promotion is enabled! Applying..");
			
			Purchase purchase = purchasesPurchaseService.getPurchase(orderID);
			if (canApplyPromotion(purchase)) {
			
				PurchasePromotion purchasePromotion = applyDoublingPointsPromotion(purchase);
				return Optional.of(purchasePromotion);
	
			} else if (purchase == null) {
				logger.error("Purchase with order ID '" + orderID + "' does not exist!");
			}
		} else {
			logger.debug("Promotion is disabled! Skipping..");
		}

		return Optional.empty();
	}

	// consider using rule engine (e.g. Drools) if there are additional promotions and/or more complex rules
	private PurchasePromotion applyDoublingPointsPromotion(Purchase purchase) {
		PurchasePromotion purchasePromotion = PurchasesFactory.eINSTANCE.createPurchasePromotion();

		int meritPointsValue = purchasesIAPsService.getIAPMeritPointsValue(purchase.getProductID());
		purchasePromotion.setOriginalValue(BigInteger.valueOf(meritPointsValue));

		int promotionAppliedValue = meritPointsValue;
		purchasePromotion.setPromotionAppliedValue(BigInteger.valueOf(promotionAppliedValue));
		purchase.setPromotion(purchasePromotion);
		
		meritPointsService.deposit(purchase.getUserID(), BigDecimal.valueOf(meritPointsValue),
				String.format("Bonus of %.2f merit points", BigDecimal.valueOf(meritPointsValue)));

		purchasesPurchaseService.saveOrUpdatePurchase(purchase);

		return EcoreUtil.copy(purchasePromotion);
	}

	private boolean isPromotionEnabled() {
		return purchasesAPIConfigService.isPromotionEnabled();
	}

	private boolean canApplyPromotion(Purchase purchase) {
		// ensure promotion is only applied once (i.e. it was not applied before)
		return (purchase != null && purchase.getPromotion() == null);
	}
}
