/**
 * Copyright (c) 2012 - 2022 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package com.playertour.backend.purchases.service.impl;

import java.util.Collections;

import org.apache.lucene.document.Document;
import org.apache.lucene.document.Field.Store;
import org.apache.lucene.document.StringField;
import org.apache.lucene.index.Term;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.gecko.search.api.IndexActionType;
import org.gecko.search.document.DocumentIndexContextObject;
import org.gecko.search.document.DocumentIndexContextObjectImpl;
import org.gecko.search.util.DeferredCommitCallback;
import org.osgi.util.promise.Deferred;

import com.playertour.backend.apis.common.LuceneIndexHelper;
import com.playertour.backend.purchases.model.purchases.Purchase;
import com.playertour.backend.purchases.model.purchases.PurchasesPackage;

class PurchasesPurchaseIndexHelper {
	static final String ORDER_ID_LOWER = "order_id_lower";
	static final String USER_ID_LOWER = "user_id_lower";
	
	static final String RECEIPT_LOCAL_VERIFICATION_DATA = "receipt.localVerificationData";
	static final String RECEIPT_SERVER_VERIFICATION_DATA = "receipt.serverVerificationData";

	static DocumentIndexContextObject mapPurchaseNew(Purchase purchase) {
		return mapPurchase(purchase, IndexActionType.ADD, null);
	}

	static DocumentIndexContextObject mapPurchaseNew(Purchase purchase, Deferred<Boolean> deferred) {
		return mapPurchase(purchase, IndexActionType.ADD, deferred);
	}

	static DocumentIndexContextObject mapPurchaseUpdate(Purchase purchase, Deferred<Boolean> deferred) {
		return mapPurchase(purchase, IndexActionType.MODIFY, deferred);
	}

	static DocumentIndexContextObject mapPurchase(Purchase purchase, IndexActionType indexAction,
			Deferred<Boolean> deferred) {

		Document doc = new Document();

		doc.add(new StringField(LuceneIndexHelper.FIELD__ECLASS_URI, constructEClassUri(), Store.YES));
		doc.add(new StringField(LuceneIndexHelper.FIELD__ID, purchase.getId(), Store.YES));
		
		if (purchase.getOrderID() != null) {
			doc.add(new StringField(ORDER_ID_LOWER, purchase.getOrderID().toLowerCase(), Store.NO));
		}

		if (purchase.getUserID() != null) {
			doc.add(new StringField(USER_ID_LOWER, purchase.getUserID().toLowerCase(), Store.NO));
		}

		DocumentIndexContextObjectImpl.Builder builder = DocumentIndexContextObjectImpl.builder()
				.withDocuments(Collections.singletonList(doc)).withSourceObject(purchase)
				.withIndexActionType(indexAction);

		if (IndexActionType.MODIFY.equals(indexAction) || IndexActionType.REMOVE.equals(indexAction)) {
			builder.withIdentifyingTerm(new Term("id", purchase.getId()));
		}
		if (deferred != null) {
			builder = builder.withCommitCallback(new DeferredCommitCallback(deferred));
		}

		return builder.build();
	}
	
	private static String constructEClassUri() {
		return EcoreUtil.getURI(PurchasesPackage.Literals.PURCHASE).toString();
	}
}
