/**
 * Copyright (c) 2012 - 2018 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package com.playertour.backend.purchases.service.impl;

import java.io.IOException;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;

import org.eclipse.emf.ecore.EObject;
import org.gecko.emf.osgi.ResourceSetFactory;
import org.gecko.emf.osgi.UriMapProvider;
import org.gecko.emf.pushstream.EPushStreamProvider;
import org.gecko.emf.repository.EMFRepository;
import org.gecko.emf.repository.query.IQuery;
import org.gecko.emf.repository.query.QueryRepository;
import org.gecko.search.api.IndexActionType;
import org.gecko.search.document.DocumentIndexContextObject;
import org.gecko.search.document.LuceneIndexService;
import org.gecko.search.util.CommonIndexService;
import org.gecko.util.pushstreams.GeckoPushbackPolicyOption;
import org.osgi.service.component.ComponentServiceObjects;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.annotations.ReferenceCardinality;
import org.osgi.service.component.annotations.ServiceScope;
import org.osgi.service.log.Logger;
import org.osgi.service.log.LoggerFactory;
import org.osgi.util.promise.Promise;
import org.osgi.util.promise.PromiseFactory;
import org.osgi.util.pushstream.PushEvent;
import org.osgi.util.pushstream.PushStream;
import org.osgi.util.pushstream.QueuePolicyOption;

import com.playertour.backend.purchases.model.purchases.Purchase;
import com.playertour.backend.purchases.model.purchases.PurchasesPackage;
import com.playertour.backend.purchases.service.api.PurchasesPurchaseIndexService;

@Component(immediate = true, name = "PurchasesPurchaseIndexService", service = PurchasesPurchaseIndexService.class, scope = ServiceScope.SINGLETON, reference = {
		@Reference(name = "mongoCondition", service = UriMapProvider.class, target = "(uri.map.src=mongodb://playertour/)", cardinality = ReferenceCardinality.MANDATORY),
		@Reference(name = "modelCondition", service = ResourceSetFactory.class, target = "(emf.model.name=purchases)", cardinality = ReferenceCardinality.MANDATORY) })
public class PurchasesPurchaseIndexServiceImpl implements PurchasesPurchaseIndexService {

	@Reference(service = LoggerFactory.class)
	private Logger logger;
	
	@Reference(target = "(repo_id=playertour.playertour)", cardinality = ReferenceCardinality.MANDATORY)
	private ComponentServiceObjects<EMFRepository> repositoryServiceObjects;

	@Reference(target = "(id=purchases)")
	private LuceneIndexService purchasesIndex;
	
	private PromiseFactory factory = new PromiseFactory(Executors.newFixedThreadPool(4));
	
	@Activate
	public void activate() {
		factory.submit(() -> {
			CountDownLatch latch = new CountDownLatch(1);
			latch.await(100, TimeUnit.MILLISECONDS);
			initializeIndex();
			return true;
		}).onFailure(t -> t.printStackTrace());	
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.purchases.service.api.PurchasesPurchaseIndexService#indexPurchase(com.playertour.backend.purchases.model.purchases.Purchase, boolean)
	 */
	@Override
	public void indexPurchase(Purchase purchase, boolean isFirstSave) {
		if (isFirstSave) {
			indexPurchase(purchase, IndexActionType.ADD);
		} else {
			indexPurchase(purchase, IndexActionType.MODIFY);
		}
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.purchases.service.api.PurchasesPurchaseIndexService#deletePurchase(com.playertour.backend.purchases.model.purchases.Purchase)
	 */
	@Override
	public void deletePurchase(Purchase purchase) {
		indexPurchase(purchase, IndexActionType.REMOVE);
	}

	/* 
	 * (non-Javadoc)
	 * @see org.gecko.search.util.CommonIndexService#initializeIndex()
	 */
	@Override
	public Promise<Void> initializeIndex() {
		EMFRepository repository = repositoryServiceObjects.getService();
		QueryRepository queryRepo = (QueryRepository) repository.getAdapter(QueryRepository.class);

		IQuery query = queryRepo.createQueryBuilder().allQuery().build();
		logger.info("Starting index build for Purchases");

		resetIndex();

		EPushStreamProvider psp = queryRepo.getEObjectByQuery(PurchasesPackage.Literals.PURCHASE, query,
				CommonIndexService.getLoadOptions());
		if (psp == null) {
			return null;
		}

		// @formatter:off
		PushStream<EObject> indexNew = psp.createPushStreamBuilder()
				.withPushbackPolicy(GeckoPushbackPolicyOption.LINEAR_AFTER_THRESHOLD.getPolicy(650))
				.withQueuePolicy(QueuePolicyOption.BLOCK)
				.withExecutor(Executors.newSingleThreadExecutor())
				.withBuffer(new ArrayBlockingQueue<PushEvent<? extends EObject>>(1200))
				.build();

		return indexNew
				.map(eo -> (Purchase) eo)
				.map(PurchasesPurchaseIndexHelper::mapPurchaseNew)
				.forEach(d -> {
					repository.detach(d.getObject());
					purchasesIndex.handleContext(d);
				}).onFailure(t -> repositoryServiceObjects.ungetService(repository))
				.thenAccept(result -> logger.info("Finished index build for Purchases"))
				.thenAccept(result -> repositoryServiceObjects.ungetService(repository));
		// @formatter:on
	}

	/* 
	 * (non-Javadoc)
	 * @see org.gecko.search.util.CommonIndexService#resetIndex()
	 */
	@Override
	public void resetIndex() {
		try {
			purchasesIndex.getIndexWriter().deleteAll();
			purchasesIndex.commit();
		} catch (IOException e) {
			logger.error("Could not delete purchases index", e);
		}
	}
	
	private void indexPurchase(Purchase purchase, IndexActionType actionType) {
		DocumentIndexContextObject context = PurchasesPurchaseIndexHelper.mapPurchase(purchase, actionType, null);
		purchasesIndex.handleContextSync(context);
	}
}
