/**
 * Copyright (c) 2012 - 2018 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package com.playertour.backend.purchases.service.impl;

import java.io.IOException;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

import org.apache.lucene.document.Document;
import org.apache.lucene.index.IndexReader;
import org.apache.lucene.index.Term;
import org.apache.lucene.search.BooleanClause;
import org.apache.lucene.search.BooleanClause.Occur;
import org.apache.lucene.search.BooleanQuery;
import org.apache.lucene.search.IndexSearcher;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.TermQuery;
import org.apache.lucene.search.TopDocs;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.gecko.emf.repository.EMFRepository;
import org.gecko.search.document.LuceneIndexService;
import org.osgi.service.component.ComponentServiceObjects;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.annotations.ReferenceCardinality;
import org.osgi.service.component.annotations.ServiceScope;
import org.osgi.service.log.Logger;
import org.osgi.service.log.LoggerFactory;

import com.playertour.backend.apis.common.LuceneIndexHelper;
import com.playertour.backend.purchases.model.purchases.Purchase;
import com.playertour.backend.purchases.service.api.PurchasesPurchaseSearchService;

@Component(name = "PurchasesPurchaseSearchService", immediate = true, scope = ServiceScope.SINGLETON)
public class PurchasesPurchaseSearchServiceImpl implements PurchasesPurchaseSearchService {
	
	@Reference(target = "(repo_id=playertour.playertour)", cardinality = ReferenceCardinality.MANDATORY)
	private ComponentServiceObjects<EMFRepository> repositoryServiceObjects;

	@Reference(service = LoggerFactory.class)
	private Logger logger;

	@Reference(target = "(id=purchases)")
	private LuceneIndexService purchasesIndex;

	@Reference
	private ResourceSet resourceSet;
	
	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.purchases.service.api.PurchasesPurchaseSearchService#searchPurchasesByUserID(java.lang.String, int)
	 */
	@Override
	public List<Purchase> searchPurchasesByUserID(String userId, int maxResults) {
		IndexSearcher searcher = purchasesIndex.aquireSearch();
		EMFRepository repository = repositoryServiceObjects.getService();
		try {
			try {
				if (maxResults == -1) {
					maxResults = BooleanQuery.getMaxClauseCount();
				}
				
				String exactMatch = userId.trim().toLowerCase();
				Query q1 = new TermQuery(new Term(PurchasesPurchaseIndexHelper.USER_ID_LOWER, exactMatch));
				Query q = new BooleanQuery.Builder().add(new BooleanClause(q1, Occur.SHOULD)).build();

				TopDocs topDocs = searcher.search(q, maxResults);
				
				if (topDocs.scoreDocs.length == 0) {
					return Collections.emptyList();
				}
				
				IndexReader indexReader = searcher.getIndexReader();

				return Arrays.asList(topDocs.scoreDocs).stream().map(sd -> sd.doc).map(id -> {
					Document d;
					try {
						d = indexReader.document(id);
						return d;
					} catch (IOException e) {
						return null;
					}
				}).filter(d -> d != null).map(d -> {

					return (Purchase) LuceneIndexHelper.toEObject(d, repository);

				}).collect(Collectors.toList());

			} catch (Exception e) {
				logger.error("Exception while searching for purchases by user ID " + userId, e);

				return Collections.emptyList();
			}
		} finally {
			purchasesIndex.releaseSearcher(searcher);
			repositoryServiceObjects.ungetService(repository);
		}
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.purchases.service.api.PurchasesPurchaseSearchService#searchPurchasesByOrderID(java.lang.String)
	 */
	@Override
	public Purchase searchPurchasesByOrderID(String orderID) {
		IndexSearcher searcher = purchasesIndex.aquireSearch();
		EMFRepository repository = repositoryServiceObjects.getService();
		try {
			try {
				String exactMatch = orderID.trim().toLowerCase();
				Query q1 = new TermQuery(new Term(PurchasesPurchaseIndexHelper.ORDER_ID_LOWER, exactMatch));
				Query q = new BooleanQuery.Builder().add(new BooleanClause(q1, Occur.SHOULD)).build();
				
				TopDocs topDocs = searcher.search(q, 1);
				
				if (topDocs.scoreDocs.length == 0) {
					return null;
				}
				
				IndexReader indexReader = searcher.getIndexReader();
				
				Document doc = indexReader.document(topDocs.scoreDocs[0].doc);
				
				return (Purchase) LuceneIndexHelper.toEObject(doc, repository);
				
			} catch (Exception e) {
				logger.error("Exception while searching for purchase by order ID " + orderID, e);

				return null;
			}
		} finally {
			purchasesIndex.releaseSearcher(searcher);
			repositoryServiceObjects.ungetService(repository);
		}
	}
}
