/**
 * Copyright (c) 2012 - 2022 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package com.playertour.backend.purchases.service.impl;

import java.util.List;
import java.util.Objects;
import java.util.Optional;

import org.gecko.emf.repository.EMFRepository;
import org.gecko.emf.repository.query.IQuery;
import org.gecko.emf.repository.query.IQueryBuilder;
import org.gecko.emf.repository.query.QueryRepository;
import org.gecko.mongo.osgi.MongoDatabaseProvider;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.annotations.ReferenceScope;
import org.osgi.service.component.annotations.ServiceScope;

import com.playertour.backend.purchases.model.purchases.Purchase;
import com.playertour.backend.purchases.model.purchases.PurchasesPackage;
import com.playertour.backend.purchases.service.api.PurchasesPurchaseIndexService;
import com.playertour.backend.purchases.service.api.PurchasesPurchaseSearchService;
import com.playertour.backend.purchases.service.api.PurchasesPurchaseService;

@Component(name = "PurchasesPurchaseService", scope = ServiceScope.PROTOTYPE)
public class PurchasesPurchaseServiceImpl implements PurchasesPurchaseService {
	
	@Reference(target = "(repo_id=playertour.playertour)", scope = ReferenceScope.PROTOTYPE_REQUIRED)
	private EMFRepository repository;
	
	@Reference(target = "(database=playertour)")
	private MongoDatabaseProvider dbProvider;
	
	@Reference
	private PurchasesPurchaseIndexService purchasesPurchaseIndexService;
	
	@Reference
	private PurchasesPurchaseSearchService purchasesPurchaseSearchService;

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.purchases.service.api.PurchasesPurchaseService#saveOrUpdatePurchase(com.playertour.backend.purchases.model.purchases.Purchase)
	 */
	@Override
	public Purchase saveOrUpdatePurchase(Purchase purchase) {
		Objects.requireNonNull(purchase, "Purchase is required!");
		Objects.requireNonNull(purchase.getOrderID(), "Purchase order ID is required!");

		boolean purchaseExists = purchaseExists(purchase.getOrderID());
		if (purchaseExists) {
			Purchase existingPurchase = getPurchase(purchase.getOrderID());
			existingPurchase.setStatus(purchase.getStatus());
			if (purchase.getPromotion() != null) {
				existingPurchase.setPromotion(purchase.getPromotion());
			}
			purchase = existingPurchase;
		}

		boolean isFirstSave = purchaseExists;

		repository.save(purchase);

		purchasesPurchaseIndexService.indexPurchase(purchase, isFirstSave);

		return purchase;
	}
	
	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.purchases.service.api.PurchasesPurchaseService#getPurchases(java.lang.String, java.util.Optional)
	 */
	@Override
	public List<Purchase> getPurchases(String userId, Optional<Integer> maxResultsOptional) {
		Objects.requireNonNull(userId, "User ID is required!");

		QueryRepository queryRepo = (QueryRepository) repository.getAdapter(QueryRepository.class);

		// @formatter:off
		IQueryBuilder queryBuilder = queryRepo.createQueryBuilder()
				.column(PurchasesPackage.Literals.PURCHASE__USER_ID)
				.simpleValue(userId);
		
		IQuery query = (maxResultsOptional.isPresent()) ? 
				queryBuilder.limit(maxResultsOptional.get().intValue()).build() : 
					queryBuilder.build();
		// @formatter:on

		return queryRepo.getEObjectsByQuery(PurchasesPackage.Literals.PURCHASE, query);
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.purchases.service.api.PurchasesPurchaseService#purchaseExists(java.lang.String)
	 */
	@Override
	public boolean purchaseExists(String orderID) {
		Objects.requireNonNull(orderID, "Purchase order ID is required!");

		return (purchasesPurchaseSearchService.searchPurchasesByOrderID(orderID) != null);
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.purchases.service.api.PurchasesPurchaseService#getPurchase(java.lang.String)
	 */
	@Override
	public Purchase getPurchase(String orderID) {
		Objects.requireNonNull(orderID, "Purchase order ID is required!");

		QueryRepository queryRepo = (QueryRepository) repository.getAdapter(QueryRepository.class);

		// @formatter:off
		IQuery query = queryRepo.createQueryBuilder()
				.column(PurchasesPackage.Literals.PURCHASE__ORDER_ID)
				.simpleValue(orderID).build();
		// @formatter:on

		Purchase purchase = queryRepo.getEObjectByQuery(PurchasesPackage.Literals.PURCHASE, query, null);
		queryRepo.detach(purchase);
		return purchase;
	}
}
