/**
 * Copyright (c) 2012 - 2018 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package com.playertour.backend.purchases.service.impl;

import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeParseException;
import java.util.Objects;

import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.ComponentPropertyType;
import org.osgi.service.component.annotations.ConfigurationPolicy;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.log.Logger;
import org.osgi.service.log.LoggerFactory;
import org.osgi.service.metatype.annotations.Designate;
import org.osgi.service.metatype.annotations.ObjectClassDefinition;

import com.playertour.backend.purchases.service.api.PurchasesAPIConfigService;

@Component(immediate = true, configurationPolicy = ConfigurationPolicy.REQUIRE, configurationPid = "PurchasesAPIConfigService")
@Designate(ocd = PurchasesAPIConfigServiceImpl.APIConfig.class)
public class PurchasesAPIConfigServiceImpl implements PurchasesAPIConfigService {
	
	private static final DateTimeFormatter OFFSET_DATE_TIME_FORMATTER = DateTimeFormatter.ISO_OFFSET_DATE_TIME;

	@Reference(service = LoggerFactory.class)
	private Logger logger;	
	
	private APIConfig config;

	@ComponentPropertyType
	@ObjectClassDefinition
	public @interface APIConfig {

		String googleApplicationCredentialsPathEnv();
		
		// Apple shared secret is not required for non-subscription IAPs
		String appleSharedSecretEnv();

		String applicationName();

		String packageName();

		String[] productIDs();
		
		String promotionEnabledBeforeDateTime();
	}

	@Activate
	public void activate(APIConfig config) {
		this.config = config;
	}
	
	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.purchases.service.api.PurchasesAPIConfigService#getGoogleApplicationCredentialsPath()
	 */
	@Override
	public String getGoogleApplicationCredentialsPath() {
		return getEnvironmentVariableValue(config.googleApplicationCredentialsPathEnv());
	}
	
	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.purchases.service.api.PurchasesAPIConfigService#getAppleSharedSecret()
	 */
	// Apple shared secret is not required for non-subscription IAPs
	@Override
	public String getAppleSharedSecret() {
		return getEnvironmentVariableValue(config.appleSharedSecretEnv());
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.purchases.service.api.PurchasesAPIConfigService#getApplicationName()
	 */
	@Override
	public String getApplicationName() {
		return config.applicationName();
	}
	
	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.purchases.service.api.PurchasesAPIConfigService#getPackageName()
	 */
	@Override
	public String getPackageName() {
		return config.packageName();
	}
	
	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.purchases.service.api.PurchasesAPIConfigService#getProductIDs()
	 */
	@Override
	public String[] getProductIDs() {
		return config.productIDs();
	}
	
	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.purchases.service.api.PurchasesAPIConfigService#isPromotionEnabled()
	 */
	@Override
	public boolean isPromotionEnabled() {
		Objects.requireNonNull(config.promotionEnabledBeforeDateTime(), "Before date time is required!");

		if (isDateTimeValid(config.promotionEnabledBeforeDateTime())) {
			ZonedDateTime dateTimeNow = ZonedDateTime.now();
			ZonedDateTime parsedBeforeDateTime = ZonedDateTime.parse(config.promotionEnabledBeforeDateTime(),
					OFFSET_DATE_TIME_FORMATTER);
			return dateTimeNow.isBefore(parsedBeforeDateTime);
		} else {
			return false;
		}
	}
	
	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.purchases.service.api.PurchasesAPIConfigService#getPromotionEnabledBeforeDateTime()
	 */
	@Override
	public String getPromotionEnabledBeforeDateTime() {
		return config.promotionEnabledBeforeDateTime();
	}	

	private String getEnvironmentVariableValue(String envVariableName) {
		Objects.requireNonNull(envVariableName, "Environment variable name is required!");

		String env = System.getenv(envVariableName);
		if (env == null) {
			env = System.getProperty(envVariableName);
		}

		if (env == null) {
			String errorMessage = "No value for environment variable named '" + envVariableName
					+ "' was found in neither 'System.env' nor 'System.properties'";
			logger.error(errorMessage);
			throw new IllegalStateException(errorMessage);
		}

		return env;
	}
	
	private boolean isDateTimeValid(String dateTime) {
		Objects.requireNonNull(dateTime, "Date time is required!");
		
		try {
			
			ZonedDateTime.parse(dateTime, OFFSET_DATE_TIME_FORMATTER);
			return true;
			
		} catch (DateTimeParseException dtpe) {
			return false;
		}
	}
}
