/**
 * Copyright (c) 2012 - 2022 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package com.playertour.backend.purchases.service.impl;

import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;

import org.apache.commons.lang3.NotImplementedException;
import org.apache.http.HttpResponse;
import org.apache.http.client.ClientProtocolException;
import org.apache.http.client.HttpClient;
import org.apache.http.client.config.CookieSpecs;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import org.apache.http.util.EntityUtils;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.annotations.ReferenceScope;
import org.osgi.service.component.annotations.ServiceScope;
import org.osgi.service.log.Logger;
import org.osgi.service.log.LoggerFactory;

import com.google.gson.Gson;
import com.google.gson.JsonParser;
import com.playertour.backend.purchases.model.purchases.PurchaseReceiptVerificationStatus;
import com.playertour.backend.purchases.model.purchases.PurchasesFactory;
import com.playertour.backend.purchases.service.api.PurchasesAPIConfigService;
import com.playertour.backend.purchases.service.api.PurchasesIAPsService;
import com.playertour.backend.purchases.service.api.PurchasesPurchaseVerificationService;

@Component(name = "PurchasesAppleAppStoreService", scope = ServiceScope.SINGLETON)
public class PurchasesAppStoreServiceImpl implements PurchasesPurchaseVerificationService {

	// Sandbox URL
	private final static String SANDBOX_URL = "https://sandbox.itunes.apple.com/verifyReceipt";

	// Production URL
	private final static String PRODUCTION_URL = "https://buy.itunes.apple.com/verifyReceipt";

	private final static boolean EXCLUDE_OLD_TRANSACTIONS = true; // TODO: make this configurable if needed
	private final static int MAX_CONNECTIONS = 1;
	private final static int RESPONSE_TIMEOUT_MS = 3000;

	@Reference(service = LoggerFactory.class)
	private Logger logger;

	@Reference
	private PurchasesAPIConfigService purchasesAPIConfigService;

	@Reference(scope = ReferenceScope.PROTOTYPE_REQUIRED)
	private PurchasesIAPsService purchasesIAPsService;

	private static final Gson GSON = new Gson();

	private HttpClient client;

	
	@Activate
	public void activate() {
		// Apple shared secret is not required for non-subscription IAPs
		Objects.requireNonNull(purchasesAPIConfigService.getAppleSharedSecret(), "Apple shared secret is required!");

		try {
			client = initializeHttpClient(MAX_CONNECTIONS, RESPONSE_TIMEOUT_MS);

		} catch (Throwable t) {
			t.printStackTrace();
			logger.error("Exception was thrown while initializing HTTP client!");
		}
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.purchases.service.api.PurchasesPurchaseVerificationService#verifyNonSubscriptionPurchaseReceipt(java.lang.String, java.lang.String)
	 */
	@Override	
	public PurchaseReceiptVerificationStatus verifyNonSubscriptionPurchaseReceipt(String productID, String token) {
		Objects.requireNonNull(client, "HTTP client is required!");
		Objects.requireNonNull(productID, "In-app-product's product ID is required!");
		Objects.requireNonNull(token, "Purchase token is required!");

		PurchaseReceiptVerificationStatus purchaseVerificationStatus = PurchasesFactory.eINSTANCE
				.createPurchaseReceiptVerificationStatus();

		if (purchasesIAPsService.iapExists(productID)) {

			Map<String, String> payload = constructPayload(token, EXCLUDE_OLD_TRANSACTIONS);

			try {

				ApplePurchaseReceiptVerificationStatus status = verifyNonSubscriptionPurchase(PRODUCTION_URL, payload);
				if (status.receiptValidationStatusCode == 21007) {
					status = verifyNonSubscriptionPurchase(SANDBOX_URL, payload);
				}

				boolean isValid = mapStatus(status);

				purchaseVerificationStatus.setValid(isValid);
				purchaseVerificationStatus.setRetry(status.isRetryable);
				purchaseVerificationStatus.setErrorMessage(status.message);

			} catch (IOException e) {
				e.printStackTrace();
				logger.error(e.getMessage());

				purchaseVerificationStatus.setValid(false);
				purchaseVerificationStatus.setErrorMessage(e.getMessage());
				purchaseVerificationStatus.setRetry(true);
			}

		} else {

			purchaseVerificationStatus.setValid(false);
			purchaseVerificationStatus.setErrorMessage("Product '" + productID + "' does not exist!");
			purchaseVerificationStatus.setRetry(false);
		}

		return purchaseVerificationStatus;
	}
	
	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.purchases.service.api.PurchasesPurchaseVerificationService#verifySubscriptionPurchaseReceipt(java.lang.String, java.lang.String)
	 */
	@Override
	public PurchaseReceiptVerificationStatus verifySubscriptionPurchaseReceipt(String productID, String token) {
		throw new NotImplementedException("Subscription purchases are not supported yet!");
	}
	
	private ApplePurchaseReceiptVerificationStatus verifyNonSubscriptionPurchase(String url,
			Map<String, String> payload) throws ClientProtocolException, IOException {
		HttpResponse response = executeRequest(url, payload);

		ApplePurchaseReceiptVerificationStatus applePurchaseReceiptVerificationStatus = new ApplePurchaseReceiptVerificationStatus();

		int responseStatusCode = response.getStatusLine().getStatusCode();
		if (responseStatusCode == 200 && response.getEntity() != null) {
			String responseString = EntityUtils.toString(response.getEntity());

			applePurchaseReceiptVerificationStatus.receiptValidationStatusCode = extractStatus(responseString);
			applePurchaseReceiptVerificationStatus.isRetryable = extractIsRetryable(responseString);
			applePurchaseReceiptVerificationStatus.httpResponseStatusCode = responseStatusCode;

			return applePurchaseReceiptVerificationStatus;

		} else {

			applePurchaseReceiptVerificationStatus.receiptValidationStatusCode = -1;
			applePurchaseReceiptVerificationStatus.httpResponseStatusCode = responseStatusCode;

			if (response.getEntity() == null) {
				logger.error("Received empty response!");

				applePurchaseReceiptVerificationStatus.isRetryable = true;
				applePurchaseReceiptVerificationStatus.message = "Received empty response!";

				return applePurchaseReceiptVerificationStatus;

			} else if (responseStatusCode == 503) {
				logger.error("Service unavailable! Retry later.");

				applePurchaseReceiptVerificationStatus.isRetryable = true;
				applePurchaseReceiptVerificationStatus.message = "Service unavailable! Retry later.";

				return applePurchaseReceiptVerificationStatus;

			} else {
				logger.error("Status code is not OK: " + responseStatusCode + "("
						+ response.getStatusLine().getReasonPhrase() + ")");

				applePurchaseReceiptVerificationStatus.isRetryable = false;
				applePurchaseReceiptVerificationStatus.message = response.getStatusLine().getReasonPhrase();

				return applePurchaseReceiptVerificationStatus;
			}
		}
	}
	
	private Map<String, String> constructPayload(String receiptData, boolean excludeOldTransactions) {
		// https://developer.apple.com/documentation/appstorereceipts/requestbody

		Map<String, String> payload = new HashMap<String, String>();
		payload.put("receipt-data", receiptData);
		// Apple shared secret is not required for non-subscription IAPs
		payload.put("password", purchasesAPIConfigService.getAppleSharedSecret());
		payload.put("exclude-old-transactions", String.valueOf(excludeOldTransactions));

		return payload;
	}
	
	private int extractStatus(String responseString) {
		return JsonParser.parseString(responseString).getAsJsonObject().get("status").getAsInt();
	}
	
	private boolean extractIsRetryable(String responseString) {
		return (JsonParser.parseString(responseString).getAsJsonObject().has("is-retryable")
				&& JsonParser.parseString(responseString).getAsJsonObject().get("is-retryable").getAsInt() == 1);
	}

	private HttpResponse executeRequest(String url, Map<String, String> payload)
			throws ClientProtocolException, IOException {
		HttpPost request = new HttpPost(url);

		request.setEntity(new StringEntity(GSON.toJson(payload)));

		long now = System.currentTimeMillis();

		HttpResponse response = client.execute(request);

		logger.debug("Response took " + (System.currentTimeMillis() - now) + " ms: " + response.getStatusLine());

		return response;
	}
	
	// https://developer.apple.com/documentation/appstorereceipts/status
	private boolean mapStatus(ApplePurchaseReceiptVerificationStatus status) {
		String message = status.receiptValidationStatusCode + ": ";
		switch (status.receiptValidationStatusCode) {
		case -1:
			return false;
		case 0:
			message += "Receipt is valid";
			return true;
		case 21000:
			message += "The request to the App Store was not made using the HTTP POST request method.";
			break;
		case 21002:
			message += "The data in the receipt-data property was malformed or the service experienced a temporary issue. Try again.";
			break;
		case 21003:
			message += "The receipt could not be authenticated.";
			break;
		case 21004:
			message += "The shared secret you provided does not match the shared secret on file for your account.";
			break;
		case 21005:
			message += "The receipt server was temporarily unable to provide the receipt. Try again.";
			break;
		case 21006:
			message += "This receipt is valid but the subscription has expired.";
			break;
		case 21007:
			message += "This receipt is from the test environment, but it was sent to the production environment for verification.";
			break;
		case 21008:
			message += "This receipt is from the production environment, but it was sent to the test environment for verification.";
			break;
		case 21009:
			message += "Internal data access error. Try again later.";
			break;
		case 21010:
			message += "The user account cannot be found or has been deleted.";
			break;			
		default:
			message = "Unknown error: status code = " + status;
		}
		logger.info(message);
		status.message = message;
		return false;
	}

	private HttpClient initializeHttpClient(int maxConnections, int responseTimeoutMs) {
		PoolingHttpClientConnectionManager connectionManager = new PoolingHttpClientConnectionManager();
		connectionManager.setMaxTotal(maxConnections);
		connectionManager.setDefaultMaxPerRoute(responseTimeoutMs);

		// @formatter:off
    	RequestConfig requestConfig = RequestConfig.custom()
                .setConnectionRequestTimeout(responseTimeoutMs)
                .setSocketTimeout(responseTimeoutMs)
                .setConnectTimeout(responseTimeoutMs)
                .setCookieSpec(CookieSpecs.IGNORE_COOKIES)
                .build();
    	// @formatter:on

		// @formatter:off
        return HttpClientBuilder.create()
                .setConnectionManager(connectionManager)
                .setDefaultRequestConfig(requestConfig)
                .build();
        // @formatter:on
	}
	
	private class ApplePurchaseReceiptVerificationStatus {
		int receiptValidationStatusCode;
		int httpResponseStatusCode;
		String message;
		boolean isRetryable;
	}
}
