/**
 * Copyright (c) 2012 - 2022 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package com.playertour.backend.purchases.service.impl;

import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.security.GeneralSecurityException;
import java.util.Objects;

import org.apache.commons.lang3.NotImplementedException;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.annotations.ReferenceScope;
import org.osgi.service.component.annotations.ServiceScope;
import org.osgi.service.log.Logger;
import org.osgi.service.log.LoggerFactory;

import com.google.api.client.googleapis.javanet.GoogleNetHttpTransport;
import com.google.api.client.googleapis.json.GoogleJsonResponseException;
import com.google.api.client.json.gson.GsonFactory;
import com.google.api.services.androidpublisher.AndroidPublisher;
import com.google.api.services.androidpublisher.AndroidPublisherScopes;
import com.google.api.services.androidpublisher.model.ProductPurchase;
import com.google.auth.http.HttpCredentialsAdapter;
import com.google.auth.oauth2.GoogleCredentials;
import com.playertour.backend.purchases.model.purchases.PurchaseReceiptVerificationStatus;
import com.playertour.backend.purchases.model.purchases.PurchasesFactory;
import com.playertour.backend.purchases.service.api.PurchasesAPIConfigService;
import com.playertour.backend.purchases.service.api.PurchasesIAPsService;
import com.playertour.backend.purchases.service.api.PurchasesPurchaseVerificationService;

@Component(name = "PurchasesGooglePlayService", scope = ServiceScope.SINGLETON)
public class PurchasesGooglePlayServiceImpl implements PurchasesPurchaseVerificationService {
	
	@Reference(service = LoggerFactory.class)
	private Logger logger;
	
	@Reference
	private PurchasesAPIConfigService purchasesAPIConfigService;
	
	@Reference(scope = ReferenceScope.PROTOTYPE_REQUIRED)
	private PurchasesIAPsService purchasesIAPsService;
	
	AndroidPublisher service;
	
	@Activate
	public void activate() {
		Objects.requireNonNull(purchasesAPIConfigService.getGoogleApplicationCredentialsPath(),
				"Google application credentials path is required!");
		Objects.requireNonNull(purchasesAPIConfigService.getApplicationName(), "Application name is required!");
		Objects.requireNonNull(purchasesAPIConfigService.getPackageName(), "Package name is required!");

		try {
			service = initializeAndroidPublisherAPIService();

		} catch (IOException | GeneralSecurityException ex) {
			ex.printStackTrace();
			logger.error("Exception was thrown while initializing AndroidPublisher API service!");
		}
	}
	
	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.purchases.service.api.PurchasesPurchaseVerificationService#verifyNonSubscriptionPurchaseReceipt(java.lang.String, java.lang.String)
	 */
	@Override
	public PurchaseReceiptVerificationStatus verifyNonSubscriptionPurchaseReceipt(String productID, String token) {
		Objects.requireNonNull(service, "AndroidPublisher API service is required!");
		Objects.requireNonNull(productID, "In-app-product's product ID is required!");
		Objects.requireNonNull(token, "Purchase token is required!");

		PurchaseReceiptVerificationStatus purchaseVerificationStatus = PurchasesFactory.eINSTANCE
				.createPurchaseReceiptVerificationStatus();

		if (purchasesIAPsService.iapExists(productID)) {

			try {
				ProductPurchase productPurchase = service.purchases().products()
						.get(purchasesAPIConfigService.getPackageName(), productID, token).execute();

				logger.debug(productPurchase.toPrettyString());

				purchaseVerificationStatus.setValid(true);

			} catch (GoogleJsonResponseException e) {
				e.printStackTrace();
				logger.error(e.getMessage());
				logger.error("Status code: " + e.getStatusCode());
				logger.error("Status message: " + e.getStatusMessage());
				logger.error("Reason: " + e.getDetails().getErrors().get(0).getReason()); // Reason: invalid
				logger.error("Message: " + e.getDetails().getMessage());

				// receipt is invalid
				if (e.getStatusCode() == 400
						&& "invalid".equalsIgnoreCase(e.getDetails().getErrors().get(0).getReason())) {
					purchaseVerificationStatus.setValid(false);
					purchaseVerificationStatus.setErrorMessage(e.getDetails().getMessage());
					purchaseVerificationStatus.setRetry(false);

					// Google service in not available - verification will be retried later
				} else if (e.getStatusCode() == 503) {
					purchaseVerificationStatus.setValid(false);
					purchaseVerificationStatus.setErrorMessage("Service unavailable! Retry later.");
					purchaseVerificationStatus.setRetry(true);
				}

			} catch (IOException e) {
				e.printStackTrace();
				logger.error(e.getMessage());

				purchaseVerificationStatus.setValid(false);
				purchaseVerificationStatus.setErrorMessage(e.getMessage());
				purchaseVerificationStatus.setRetry(true);
			}

		} else {
			purchaseVerificationStatus.setValid(false);
			purchaseVerificationStatus.setErrorMessage("Product '" + productID + "' does not exist!");
			purchaseVerificationStatus.setRetry(false);
		}

		return purchaseVerificationStatus;
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.purchases.service.api.PurchasesPurchaseVerificationService#verifySubscriptionPurchaseReceipt(java.lang.String, java.lang.String)
	 */
	@Override
	public PurchaseReceiptVerificationStatus verifySubscriptionPurchaseReceipt(String productID, String token) {
		throw new NotImplementedException("Subscription purchases are not supported yet!");
	}

	private AndroidPublisher initializeAndroidPublisherAPIService()
			throws FileNotFoundException, IOException, GeneralSecurityException {
		
		// @formatter:off
		GoogleCredentials credentials = GoogleCredentials
				.fromStream(new FileInputStream(purchasesAPIConfigService.getGoogleApplicationCredentialsPath()))
				.createScoped(AndroidPublisherScopes.ANDROIDPUBLISHER);

		return new AndroidPublisher.Builder(GoogleNetHttpTransport.newTrustedTransport(),
				GsonFactory.getDefaultInstance(), 
				new HttpCredentialsAdapter(credentials))
				.setApplicationName(purchasesAPIConfigService.getApplicationName())
				.build();
		// @formatter:on
	}
}
