/**
 * Copyright (c) 2012 - 2018 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package com.playertour.backend.purchases.service.impl;

import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.annotations.ServiceScope;
import org.osgi.service.log.Logger;
import org.osgi.service.log.LoggerFactory;

import com.playertour.backend.purchases.service.api.PurchasesAPIConfigService;
import com.playertour.backend.purchases.service.api.PurchasesIAPsService;
import com.playertour.backend.purchases.service.api.PurchasesInAppProductIDUnsupportedFormatException;
import com.playertour.backend.purchases.service.api.PurchasesInAppProductIDsDoNotExistException;

@Component(name = "PurchasesIAPsService", scope = ServiceScope.PROTOTYPE)
public class PurchasesIAPsServiceImpl implements PurchasesIAPsService {

	@Reference(service = LoggerFactory.class)
	private Logger logger;

	@Reference
	private PurchasesAPIConfigService purchasesAPIConfigService;

	private List<String> productIDs;

	private Map<String, Integer> iapsMeritPoints;

	@Activate
	public void activate() throws PurchasesInAppProductIDsDoNotExistException {
		if (purchasesAPIConfigService.getProductIDs() == null
				|| Arrays.asList(purchasesAPIConfigService.getProductIDs()).isEmpty()) {
			throw new PurchasesInAppProductIDsDoNotExistException("In-app-product's product IDs are required!");
		}

		productIDs = Arrays.asList(purchasesAPIConfigService.getProductIDs());

		iapsMeritPoints = constructIAPsMeritPointsMap();
	}
	
	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.purchases.service.api.PurchasesIAPsService#getProductIDs()
	 */
	@Override
	public List<String> getProductIDs() {
		return productIDs;
	}
	
	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.purchases.service.api.PurchasesIAPsService#iapExists(java.lang.String)
	 */
	@Override
	public boolean iapExists(String productID) {
		Objects.requireNonNull(productID, "In-app-product's product ID is required!");

		return (productIDs.contains(productID));
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.purchases.service.api.PurchasesIAPsService#getIAPMeritPointsValue(java.lang.String)
	 */
	@Override
	public int getIAPMeritPointsValue(String productID) {
		Objects.requireNonNull(productID, "In-app-product's product ID is required!");

		if (iapsMeritPoints.containsKey(productID)) {
			return iapsMeritPoints.get(productID);
		}

		return 0;
	}

	private Map<String, Integer> constructIAPsMeritPointsMap() {
		return productIDs.stream().collect(Collectors.toMap(iapProductId -> iapProductId,
				iapProductId -> extractMeritPointsPurchased(iapProductId)));
	}

	private int extractMeritPointsPurchased(String productID) throws PurchasesInAppProductIDUnsupportedFormatException {
		Objects.requireNonNull(productID, "In-app-product's product ID is required!");

		if (!productID.contains("_")) {
			throw new PurchasesInAppProductIDUnsupportedFormatException(
					"Unsupported format for product ID '" + productID + "'!");
		}

		String[] productIDParts = productID.split("_");

		return Integer.parseInt(productIDParts[(productIDParts.length - 1)]);
	}
}
