/**
 * Copyright (c) 2012 - 2018 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package com.playertour.backend.purchases.service.impl;

import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

import org.apache.commons.lang3.NotImplementedException;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.annotations.ReferenceScope;
import org.osgi.service.component.annotations.ServiceScope;
import org.osgi.service.event.EventAdmin;
import org.osgi.service.event.annotations.RequireEventAdmin;
import org.osgi.service.log.Logger;
import org.osgi.service.log.LoggerFactory;

import com.playertour.backend.meritpoints.service.api.MeritPointsService;
import com.playertour.backend.purchases.model.purchases.InAppProductsIDs;
import com.playertour.backend.purchases.model.purchases.NonSubscriptionStatus;
import com.playertour.backend.purchases.model.purchases.Purchase;
import com.playertour.backend.purchases.model.purchases.PurchaseHistory;
import com.playertour.backend.purchases.model.purchases.PurchaseReceiptVerificationStatus;
import com.playertour.backend.purchases.model.purchases.PurchasesFactory;
import com.playertour.backend.purchases.service.api.PurchasesIAPsService;
import com.playertour.backend.purchases.service.api.PurchasesPurchaseService;
import com.playertour.backend.purchases.service.api.PurchasesPurchaseVerificationService;
import com.playertour.backend.purchases.service.api.PurchasesService;
import com.playertour.backend.purchases.service.impl.events.publishers.PurchaseEventPublisher;

@Component(name = "PurchasesService", scope = ServiceScope.PROTOTYPE)
@RequireEventAdmin
public class PurchasesServiceImpl implements PurchasesService {

	@Reference(service = LoggerFactory.class)
	private Logger logger;

	@Reference(scope = ReferenceScope.PROTOTYPE_REQUIRED)
	private PurchasesIAPsService purchasesIAPsService;

	@Reference(scope = ReferenceScope.PROTOTYPE_REQUIRED)
	private PurchasesPurchaseService purchasesPurchaseService;
	
	@Reference(target = "(component.name=PurchasesGooglePlayService)")
	private PurchasesPurchaseVerificationService purchasesGooglePlayService;

	@Reference(target = "(component.name=PurchasesAppleAppStoreService)")
	private PurchasesPurchaseVerificationService purchasesAppleAppStoreService;
		
	@Reference(scope = ReferenceScope.PROTOTYPE_REQUIRED)
	private MeritPointsService meritPointsService;
	
	@Reference
	private EventAdmin eventAdmin;	

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.purchases.service.api.PurchasesService#getInAppProductIDs()
	 */
	@Override
	public InAppProductsIDs getInAppProductIDs() {
		InAppProductsIDs inAppProductsIDs = PurchasesFactory.eINSTANCE.createInAppProductsIDs();
		
		List<String> productIDs = purchasesIAPsService.getProductIDs();
		
		if (productIDs != null && !productIDs.isEmpty()) {
			inAppProductsIDs.getProductIDs().addAll(productIDs);
		}

		return inAppProductsIDs;
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.purchases.service.api.PurchasesService#inAppProductExists(java.lang.String)
	 */
	@Override
	public boolean inAppProductExists(String productID) {
		Objects.requireNonNull(productID, "In-app-product's product ID is required!");
		
		return purchasesIAPsService.iapExists(productID);
	}
	
	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.purchases.service.api.PurchasesService#verifyPurchaseReceipt(java.lang.String, com.playertour.backend.purchases.model.purchases.Purchase)
	 */
	@Override
	public PurchaseReceiptVerificationStatus verifyPurchaseReceipt(String userId, Purchase purchase) {
		Objects.requireNonNull(userId, "User ID is required!");
		Objects.requireNonNull(purchase, "Purchase is required!");
		Objects.requireNonNull(purchase.getReceipt(), "Purchase receipt is required!");

		logger.info("Verifying purchase receipt for user ID: " + userId);

		PurchaseReceiptVerificationStatus purchaseReceiptVerificationStatus;

		switch (purchase.getSource()) {
		case GOOGLE_PLAY:
			purchaseReceiptVerificationStatus = purchasesGooglePlayService.verifyPurchaseReceipt(purchase.getType(),
					purchase.getProductID(), purchase.getReceipt().getServerVerificationData());
			break;
		case APP_STORE:
			purchaseReceiptVerificationStatus = purchasesAppleAppStoreService.verifyPurchaseReceipt(purchase.getType(),
					purchase.getProductID(), purchase.getReceipt().getServerVerificationData());
			break;
		default:
			throw new IllegalArgumentException("Either non-subscription or subscription IAP type is required!");
		}

		purchase.setUserID(userId);

		logger.info("Purchase receipt verification status: "
				+ (purchaseReceiptVerificationStatus.isValid() ? "valid" : "invalid"));

		if (purchaseReceiptVerificationStatus.isValid()) {
			purchase.setStatus(NonSubscriptionStatus.VERIFIED);
		} else {
			purchase.setStatus(NonSubscriptionStatus.INVALID);
		}

		if (!purchaseReceiptVerificationStatus.isValid()
				&& purchaseReceiptVerificationStatus.getErrorMessage() != null) {
			logger.error("Purchase receipt verification status error message: "
					+ purchaseReceiptVerificationStatus.getErrorMessage());
		}

		if (purchaseReceiptVerificationStatus.isRetry()) {
			logger.error("Purchase receipt verification will be retried");
		}

		if (purchaseReceiptVerificationStatus.getVerificationDateTime() == 0) {
			purchaseReceiptVerificationStatus.setVerificationDateTime(
					LocalDateTime.now().atZone(ZoneOffset.systemDefault()).toInstant().toEpochMilli());
		}

		purchase.getReceipt().getVerificationStatus().add(purchaseReceiptVerificationStatus);

		purchasesPurchaseService.saveOrUpdatePurchase(purchase);
		
		// publish event
		if (purchase.getStatus() == NonSubscriptionStatus.VERIFIED) {
			PurchaseEventPublisher.publishPurchaseVerifiedEvent(eventAdmin, 
					userId, 
					purchase.getProductID(),
					purchase.getOrderID());
		}

		return EcoreUtil.copy(purchaseReceiptVerificationStatus);
	}
	
	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.purchases.service.api.PurchasesService#savePurchase(java.lang.String, com.playertour.backend.purchases.model.purchases.Purchase)
	 */
	@Override
	public Purchase savePurchase(String userId, Purchase purchase) {
		Objects.requireNonNull(purchase, "Purchase is required!");
		Objects.requireNonNull(purchase.getSource(), "Purchase source is required!");
		Objects.requireNonNull(purchase.getType(), "Purchase type is required!");
		Objects.requireNonNull(purchase.getOrderID(), "Purchase order ID is required!");
		Objects.requireNonNull(purchase.getProductID(), "Purchase product ID is required!");
		Objects.requireNonNull(purchase.getStatus(), "Purchase status is required!");
		Objects.requireNonNull(purchase.getReceipt(), "Purchase receipt is required!");
		
		boolean isPurchaseVerified = isPurchaseVerified(purchase);
		boolean isPurchaseCompleted = isPurchaseCompleted(purchase);
		
		// prevent malicious or erroneous multiple calls to save completed purchase - saving completed purchase triggers multiple side-effects (including saving merit points, sending notifications, applying promotions, etc.)
		if (isPurchaseCompleted || 
				(!isPurchaseVerified && purchase.getStatus() == NonSubscriptionStatus.COMPLETED)) {
			
			if (isPurchaseCompleted) {
				logger.error("Purchase for order ID {} already completed! Skipping..", purchase.getOrderID());
			} else if (!isPurchaseVerified && purchase.getStatus() == NonSubscriptionStatus.COMPLETED) {
				logger.error("Purchase for order ID {} must be verified first! Skipping..", purchase.getOrderID());
			}
			
			return purchasesPurchaseService.getPurchase(purchase.getOrderID());
		}
		
		setAdditionalPurchaseAttributes(purchase, userId);
		
		Purchase savedPurchase;
		switch (purchase.getType()) {
		case NON_SUBSCRIPTION:
			savedPurchase = saveNonSubscriptionPurchase(purchase);
			break;
		case SUBSCRIPTION:
			savedPurchase = saveSubscriptionPurchase(purchase);
			break;
		default:
			throw new IllegalArgumentException("Either non-subscription or subscription IAP type is required!");
		}
		
		if (purchase.getStatus() == NonSubscriptionStatus.COMPLETED) {
			
			// update merit points balance
			updateMeritPointsBalance(userId, savedPurchase);
			
			// publish event
			PurchaseEventPublisher.publishPurchaseCompletedEvent(eventAdmin, 
					userId, 
					savedPurchase.getProductID(),
					savedPurchase.getOrderID(), 
					purchasesIAPsService.getIAPMeritPointsValue(purchase.getProductID())); // TODO: (i18n-related): format number of merit points according to player's preferred locale
		}
					
		return savedPurchase;			
	}
	
	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.purchases.service.api.PurchasesService#getPurchaseHistory(java.lang.String, java.util.Optional)
	 */
	@Override
	public PurchaseHistory getPurchaseHistory(String userId, Optional<Integer> maxResultsOptional) {
		Objects.requireNonNull(userId, "User ID is required!");

		List<Purchase> purchases = purchasesPurchaseService.getPurchases(userId, maxResultsOptional);

		PurchaseHistory purchaseHistory = PurchasesFactory.eINSTANCE.createPurchaseHistory();
		purchaseHistory.getPurchases().addAll(purchases);
		return purchaseHistory;
	}
	
	private Purchase saveNonSubscriptionPurchase(Purchase purchase) {
		return purchasesPurchaseService.saveOrUpdatePurchase(purchase);
	}

	private Purchase saveSubscriptionPurchase(Purchase purchase) {
		throw new NotImplementedException("Subscription purchases are not supported yet!");
	}
	
	private void updateMeritPointsBalance(String userId, Purchase purchase) {
		int meritpoints = purchasesIAPsService.getIAPMeritPointsValue(purchase.getProductID());

		meritPointsService.deposit(purchase.getUserID(), BigDecimal.valueOf(meritpoints),
				String.format("Purchase of %.2f merit points", BigDecimal.valueOf(meritpoints)));
	}
	
	private boolean isPurchaseCompleted(Purchase purchase) {
		Purchase existingPurchase = purchasesPurchaseService.getPurchase(purchase.getOrderID());		
		return ((existingPurchase != null) && (existingPurchase.getStatus() == NonSubscriptionStatus.COMPLETED));
	}
	
	private boolean isPurchaseVerified(Purchase purchase) {
		Purchase existingPurchase = purchasesPurchaseService.getPurchase(purchase.getOrderID());		
		return ((existingPurchase != null) && (existingPurchase.getStatus() == NonSubscriptionStatus.VERIFIED));		
	}
	
	private void setAdditionalPurchaseAttributes(Purchase purchase, String userId) {
		purchase.setUserID(userId);
		
		if (purchase.getPurchaseDateTime() == 0) {
			purchase.setPurchaseDateTime(
					LocalDateTime.now().atZone(ZoneOffset.systemDefault()).toInstant().toEpochMilli());
		}
	}	
}
