/**
 * Copyright (c) 2012 - 2022 Data In Motion and others.
 * All rights reserved.
 *
 * This program and the accompanying materials are made available under the terms of the
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package com.playertour.backend.purchases.service.tests;

import static org.assertj.core.api.Assertions.assertThat;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.TimeUnit;

import org.apache.commons.io.IOUtils;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.gecko.mongo.osgi.MongoDatabaseProvider;
import org.junit.jupiter.api.AfterAll;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Disabled;
import org.junit.jupiter.api.MethodOrderer;
import org.junit.jupiter.api.Order;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.TestMethodOrder;
import org.junit.jupiter.api.extension.ExtendWith;
import org.junit.platform.commons.annotation.Testable;
import org.osgi.framework.BundleContext;
import org.osgi.framework.FrameworkUtil;
import org.osgi.framework.ServiceReference;
import org.osgi.test.common.annotation.InjectBundleContext;
import org.osgi.test.common.annotation.InjectService;
import org.osgi.test.common.service.ServiceAware;
import org.osgi.test.junit5.context.BundleContextExtension;
import org.osgi.test.junit5.service.ServiceExtension;

import com.mongodb.client.MongoDatabase;
import com.playertour.backend.purchases.model.purchases.InAppProductSource;
import com.playertour.backend.purchases.model.purchases.InAppProductsIDs;
import com.playertour.backend.purchases.model.purchases.NonSubscriptionStatus;
import com.playertour.backend.purchases.model.purchases.Purchase;
import com.playertour.backend.purchases.model.purchases.PurchaseHistory;
import com.playertour.backend.purchases.model.purchases.PurchaseReceipt;
import com.playertour.backend.purchases.model.purchases.PurchaseReceiptVerificationStatus;
import com.playertour.backend.purchases.model.purchases.PurchaseType;
import com.playertour.backend.purchases.model.purchases.PurchasesFactory;
import com.playertour.backend.purchases.model.purchases.PurchasesPackage;
import com.playertour.backend.purchases.service.api.PurchasesAPIConfigService;
import com.playertour.backend.purchases.service.api.PurchasesIAPsService;
import com.playertour.backend.purchases.service.api.PurchasesPurchaseIndexService;
import com.playertour.backend.purchases.service.api.PurchasesPurchaseSearchService;
import com.playertour.backend.purchases.service.api.PurchasesPurchaseService;
import com.playertour.backend.purchases.service.api.PurchasesPurchaseVerificationService;
import com.playertour.backend.purchases.service.api.PurchasesService;

@Testable
@ExtendWith(BundleContextExtension.class)
@ExtendWith(ServiceExtension.class)
@TestMethodOrder(MethodOrderer.OrderAnnotation.class)
public class PurchasesIntegrationTest {

	private static final int IAP_EXISTING_PRODUCT_IDS_SIZE = 4;
	private static final String IAP_EXISTING_PRODUCT_ID_EXAMPLE = "theplayertour_bundle_1_100";
	private static final String IAP_NONEXISTING_PRODUCT_ID_EXAMPLE = "theplayertour_bundle_0_0";

	private static final String GOOGLE_EXAMPLE_RECEIPT_OK = "/data/google_example_receipt_ok";
	private static final String GOOGLE_EXAMPLE_RECEIPT_WRONG = "/data/google_example_receipt_wrong";

	private static final String APPLE_EXAMPLE_RECEIPT_OK = "/data/apple_example_receipt_ok";
	private static final String APPLE_EXAMPLE_RECEIPT_WRONG = "/data/apple_example_receipt_wrong";
	private static final String APPLE_EXAMPLE_RECEIPT_JUERGEN = "/data/apple_example_receipt_juergen";

	@InjectBundleContext
	BundleContext bundleContext;

	@Order(value = -1)
	@Test
	public void testServices(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PurchasesAPIConfigService> purchasesAPIConfigServiceAware,
			@InjectService(cardinality = 1, timeout = 5000, filter = "(component.name=PurchasesGooglePlayService)") ServiceAware<PurchasesPurchaseVerificationService> purchasesGooglePlayServiceAware,
			@InjectService(cardinality = 1, timeout = 5000, filter = "(component.name=PurchasesAppleAppStoreService)") ServiceAware<PurchasesPurchaseVerificationService> purchasesAppleAppStoreServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PurchasesIAPsService> purchasesIAPsServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PurchasesPurchaseIndexService> purchasesPurchaseIndexServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PurchasesPurchaseSearchService> purchasesPurchaseSearchServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PurchasesPurchaseService> purchasesPurchaseServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PurchasesService> purchasesServiceAware) {

		assertThat(purchasesAPIConfigServiceAware.getServices()).hasSize(1);
		ServiceReference<PurchasesAPIConfigService> purchasesAPIConfigServiceReference = purchasesAPIConfigServiceAware
				.getServiceReference();
		assertThat(purchasesAPIConfigServiceReference).isNotNull();

		assertThat(purchasesGooglePlayServiceAware.getServices()).hasSize(1);
		ServiceReference<PurchasesPurchaseVerificationService> purchasesGooglePlayServiceReference = purchasesGooglePlayServiceAware
				.getServiceReference();
		assertThat(purchasesGooglePlayServiceReference).isNotNull();

		assertThat(purchasesAppleAppStoreServiceAware.getServices()).hasSize(1);
		ServiceReference<PurchasesPurchaseVerificationService> purchasesAppleAppStoreServiceReference = purchasesAppleAppStoreServiceAware
				.getServiceReference();
		assertThat(purchasesAppleAppStoreServiceReference).isNotNull();

		assertThat(purchasesIAPsServiceAware.getServices()).hasSize(1);
		ServiceReference<PurchasesIAPsService> purchasesIAPsServiceReference = purchasesIAPsServiceAware
				.getServiceReference();
		assertThat(purchasesIAPsServiceReference).isNotNull();

		assertThat(purchasesPurchaseIndexServiceAware.getServices()).hasSize(1);
		ServiceReference<PurchasesPurchaseIndexService> purchasesPurchaseIndexServiceReference = purchasesPurchaseIndexServiceAware
				.getServiceReference();
		assertThat(purchasesPurchaseIndexServiceReference).isNotNull();

		assertThat(purchasesPurchaseSearchServiceAware.getServices()).hasSize(1);
		ServiceReference<PurchasesPurchaseSearchService> purchasesPurchaseSearchServiceReference = purchasesPurchaseSearchServiceAware
				.getServiceReference();
		assertThat(purchasesPurchaseSearchServiceReference).isNotNull();

		assertThat(purchasesPurchaseServiceAware.getServices()).hasSize(1);
		ServiceReference<PurchasesPurchaseService> purchasesPurchaseServiceReference = purchasesPurchaseServiceAware
				.getServiceReference();
		assertThat(purchasesPurchaseServiceReference).isNotNull();

		assertThat(purchasesServiceAware.getServices()).hasSize(1);
		ServiceReference<PurchasesService> purchasesServiceReference = purchasesServiceAware.getServiceReference();
		assertThat(purchasesServiceReference).isNotNull();
	}

	@Test
	public void testGetInAppProductIDs(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PurchasesService> purchasesServiceAware) {

		assertThat(purchasesServiceAware.getServices()).hasSize(1);
		PurchasesService purchasesService = purchasesServiceAware.getService();
		assertThat(purchasesService).isNotNull();

		InAppProductsIDs inAppProductsIDs = purchasesService.getInAppProductIDs();
		assertThat(inAppProductsIDs).isNotNull();
		assertThat(inAppProductsIDs.getProductIDs()).isNotNull();
		assertThat(inAppProductsIDs.getProductIDs()).isNotEmpty();
		assertThat(inAppProductsIDs.getProductIDs()).hasSize(IAP_EXISTING_PRODUCT_IDS_SIZE);
	}

	@Test
	public void testInAppProductExists(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PurchasesService> purchasesServiceAware) {

		assertThat(purchasesServiceAware.getServices()).hasSize(1);
		PurchasesService purchasesService = purchasesServiceAware.getService();
		assertThat(purchasesService).isNotNull();

		boolean iapExists = purchasesService.inAppProductExists(IAP_EXISTING_PRODUCT_ID_EXAMPLE);
		assertThat(iapExists).isTrue();
	}

	@Test
	public void testInAppProductDoesNotExist(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PurchasesService> purchasesServiceAware) {

		assertThat(purchasesServiceAware.getServices()).hasSize(1);
		PurchasesService purchasesService = purchasesServiceAware.getService();
		assertThat(purchasesService).isNotNull();

		boolean iapExists = purchasesService.inAppProductExists(IAP_NONEXISTING_PRODUCT_ID_EXAMPLE);
		assertThat(iapExists).isFalse();
	}

	@Test
	public void testVerifyGoogleValidPurchaseReceipt(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PurchasesService> purchasesServiceAware)
			throws IOException {

		assertThat(purchasesServiceAware.getServices()).hasSize(1);
		PurchasesService purchasesService = purchasesServiceAware.getService();
		assertThat(purchasesService).isNotNull();

		String validToken = IOUtils.toString(
				FrameworkUtil.getBundle(getClass()).getEntry(GOOGLE_EXAMPLE_RECEIPT_OK).openStream(),
				StandardCharsets.UTF_8.name());
		
		Purchase purchase = PurchasesFactory.eINSTANCE.createPurchase();
		purchase.setSource(InAppProductSource.GOOGLE_PLAY);
		purchase.setType(PurchaseType.NON_SUBSCRIPTION);
		purchase.setProductID(IAP_EXISTING_PRODUCT_ID_EXAMPLE);
		purchase.setStatus(NonSubscriptionStatus.PENDING);
		
		String orderID = UUID.randomUUID().toString();
		purchase.setOrderID(orderID);		
		
		String userID = UUID.randomUUID().toString();
		purchase.setUserID(userID);
		
		long purchaseDateTime = LocalDateTime.now().atZone(ZoneOffset.systemDefault()).toInstant().toEpochMilli();
		purchase.setPurchaseDateTime(purchaseDateTime);		

		PurchaseReceipt purchaseReceipt = PurchasesFactory.eINSTANCE.createPurchaseReceipt();
		purchaseReceipt.setServerVerificationData(validToken);
		purchaseReceipt.setLocalVerificationData("TEST");
		purchase.setReceipt(purchaseReceipt);
		
		PurchaseReceiptVerificationStatus purchaseReceiptVerificationStatus = purchasesService
				.verifyPurchaseReceipt(userID, purchase);
		assertThat(purchaseReceiptVerificationStatus).isNotNull();
		assertThat(purchaseReceiptVerificationStatus.isValid()).isTrue();
		assertThat(purchaseReceiptVerificationStatus.getErrorMessage()).isNull();
	}

	@Test
	public void testVerifyGoogleInvalidPurchaseReceipt(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PurchasesService> purchasesServiceAware)
			throws IOException {

		assertThat(purchasesServiceAware.getServices()).hasSize(1);
		PurchasesService purchasesService = purchasesServiceAware.getService();
		assertThat(purchasesService).isNotNull();

		String invalidToken = IOUtils.toString(
				FrameworkUtil.getBundle(getClass()).getEntry(GOOGLE_EXAMPLE_RECEIPT_WRONG).openStream(),
				StandardCharsets.UTF_8.name());
		
		Purchase purchase = PurchasesFactory.eINSTANCE.createPurchase();
		purchase.setSource(InAppProductSource.GOOGLE_PLAY);
		purchase.setType(PurchaseType.NON_SUBSCRIPTION);
		purchase.setProductID(IAP_EXISTING_PRODUCT_ID_EXAMPLE);
		purchase.setStatus(NonSubscriptionStatus.PENDING);
		
		String orderID = UUID.randomUUID().toString();
		purchase.setOrderID(orderID);		
		
		String userID = UUID.randomUUID().toString();
		purchase.setUserID(userID);
		
		long purchaseDateTime = LocalDateTime.now().atZone(ZoneOffset.systemDefault()).toInstant().toEpochMilli();
		purchase.setPurchaseDateTime(purchaseDateTime);		

		PurchaseReceipt purchaseReceipt = PurchasesFactory.eINSTANCE.createPurchaseReceipt();
		purchaseReceipt.setServerVerificationData(invalidToken);
		purchaseReceipt.setLocalVerificationData("TEST");
		purchase.setReceipt(purchaseReceipt);		

		PurchaseReceiptVerificationStatus purchaseReceiptVerificationStatus = purchasesService
				.verifyPurchaseReceipt(userID, purchase);
		assertThat(purchaseReceiptVerificationStatus).isNotNull();
		assertThat(purchaseReceiptVerificationStatus.isValid()).isFalse();
	}

	@Test
	public void testVerifyAppleValidPurchaseReceiptFromJuergen(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PurchasesService> purchasesServiceAware)
			throws IOException {

		assertThat(purchasesServiceAware.getServices()).hasSize(1);
		PurchasesService purchasesService = purchasesServiceAware.getService();
		assertThat(purchasesService).isNotNull();

		String validToken = IOUtils.toString(
				FrameworkUtil.getBundle(getClass()).getEntry(APPLE_EXAMPLE_RECEIPT_JUERGEN).openStream(),
				StandardCharsets.UTF_8.name());
		
		Purchase purchase = PurchasesFactory.eINSTANCE.createPurchase();
		purchase.setSource(InAppProductSource.APP_STORE);
		purchase.setType(PurchaseType.NON_SUBSCRIPTION);
		purchase.setProductID(IAP_EXISTING_PRODUCT_ID_EXAMPLE);
		purchase.setStatus(NonSubscriptionStatus.PENDING);
		
		String orderID = UUID.randomUUID().toString();
		purchase.setOrderID(orderID);		
		
		String userID = UUID.randomUUID().toString();
		purchase.setUserID(userID);
		
		long purchaseDateTime = LocalDateTime.now().atZone(ZoneOffset.systemDefault()).toInstant().toEpochMilli();
		purchase.setPurchaseDateTime(purchaseDateTime);		

		PurchaseReceipt purchaseReceipt = PurchasesFactory.eINSTANCE.createPurchaseReceipt();
		purchaseReceipt.setServerVerificationData(validToken);
		purchaseReceipt.setLocalVerificationData("TEST");
		purchase.setReceipt(purchaseReceipt);		

		PurchaseReceiptVerificationStatus purchaseReceiptVerificationStatus = purchasesService
				.verifyPurchaseReceipt(userID, purchase);
		assertThat(purchaseReceiptVerificationStatus).isNotNull();
		assertThat(purchaseReceiptVerificationStatus.isValid()).withFailMessage(purchaseReceiptVerificationStatus::getErrorMessage).isTrue();
		assertThat(purchaseReceiptVerificationStatus.getErrorMessage()).isNull();
	}
	@Disabled("Receipt received from Moritz on Thursday (21.04) worked then, now it stopped working... Disabled until this is resolved")
	@Test
	public void testVerifyAppleValidPurchaseReceipt(
			@InjectService(timeout = 50000) ServiceAware<PurchasesService> purchasesServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PurchasesAPIConfigService> purchasesConfigServiceAware)
					throws IOException {

		PurchasesAPIConfigService purchasesAPIConfigService = purchasesConfigServiceAware.getService();
		assertThat(purchasesAPIConfigService).isNotNull();
		
		String secret = purchasesAPIConfigService.getAppleSharedSecret();
		assertThat(secret).isNotNull();
		
		
		assertThat(purchasesServiceAware.getServices()).hasSize(1);
		PurchasesService purchasesService = purchasesServiceAware.getService();
		assertThat(purchasesService).isNotNull();
		
		
		String validToken = IOUtils.toString(
				FrameworkUtil.getBundle(getClass()).getEntry(APPLE_EXAMPLE_RECEIPT_OK).openStream(),
				StandardCharsets.UTF_8.name());
		
		Purchase purchase = PurchasesFactory.eINSTANCE.createPurchase();
		purchase.setSource(InAppProductSource.APP_STORE);
		purchase.setType(PurchaseType.NON_SUBSCRIPTION);
		purchase.setProductID(IAP_EXISTING_PRODUCT_ID_EXAMPLE);
		purchase.setStatus(NonSubscriptionStatus.PENDING);
		
		String orderID = UUID.randomUUID().toString();
		purchase.setOrderID(orderID);		
		
		String userID = UUID.randomUUID().toString();
		purchase.setUserID(userID);
		
		long purchaseDateTime = LocalDateTime.now().atZone(ZoneOffset.systemDefault()).toInstant().toEpochMilli();
		purchase.setPurchaseDateTime(purchaseDateTime);		
		
		PurchaseReceipt purchaseReceipt = PurchasesFactory.eINSTANCE.createPurchaseReceipt();
		purchaseReceipt.setServerVerificationData(validToken);
		purchaseReceipt.setLocalVerificationData("TEST");
		purchase.setReceipt(purchaseReceipt);		
		
		PurchaseReceiptVerificationStatus purchaseReceiptVerificationStatus = purchasesService
				.verifyPurchaseReceipt(userID, purchase);
		assertThat(purchaseReceiptVerificationStatus).isNotNull();
		assertThat(purchaseReceiptVerificationStatus.isValid()).isTrue();
		assertThat(purchaseReceiptVerificationStatus.getErrorMessage()).isNull();
	}

	@Test
	public void testVerifyAppleInvalidPurchaseReceipt(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PurchasesService> purchasesServiceAware)
			throws IOException {

		assertThat(purchasesServiceAware.getServices()).hasSize(1);
		PurchasesService purchasesService = purchasesServiceAware.getService();
		assertThat(purchasesService).isNotNull();

		String invalidToken = IOUtils.toString(
				FrameworkUtil.getBundle(getClass()).getEntry(APPLE_EXAMPLE_RECEIPT_WRONG).openStream(),
				StandardCharsets.UTF_8.name());
		
		Purchase purchase = PurchasesFactory.eINSTANCE.createPurchase();
		purchase.setSource(InAppProductSource.APP_STORE);
		purchase.setType(PurchaseType.NON_SUBSCRIPTION);
		purchase.setProductID(IAP_EXISTING_PRODUCT_ID_EXAMPLE);
		purchase.setStatus(NonSubscriptionStatus.PENDING);
		
		String orderID = UUID.randomUUID().toString();
		purchase.setOrderID(orderID);		
		
		String userID = UUID.randomUUID().toString();
		purchase.setUserID(userID);
		
		long purchaseDateTime = LocalDateTime.now().atZone(ZoneOffset.systemDefault()).toInstant().toEpochMilli();
		purchase.setPurchaseDateTime(purchaseDateTime);		

		PurchaseReceipt purchaseReceipt = PurchasesFactory.eINSTANCE.createPurchaseReceipt();
		purchaseReceipt.setServerVerificationData(invalidToken);
		purchaseReceipt.setLocalVerificationData("TEST");
		purchase.setReceipt(purchaseReceipt);		

		PurchaseReceiptVerificationStatus purchaseReceiptVerificationStatus = purchasesService
				.verifyPurchaseReceipt(userID, purchase);
		assertThat(purchaseReceiptVerificationStatus).isNotNull();
		assertThat(purchaseReceiptVerificationStatus.isValid()).isFalse();
	}

	@Test
	public void testSaveGoogleNonSubscriptionPurchase(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PurchasesService> purchasesServiceAware) {

		assertThat(purchasesServiceAware.getServices()).hasSize(1);
		PurchasesService purchasesService = purchasesServiceAware.getService();
		assertThat(purchasesService).isNotNull();
		
		Purchase purchase = PurchasesFactory.eINSTANCE.createPurchase();
		purchase.setSource(InAppProductSource.GOOGLE_PLAY);
		purchase.setType(PurchaseType.NON_SUBSCRIPTION);
		purchase.setProductID(IAP_EXISTING_PRODUCT_ID_EXAMPLE);
		purchase.setStatus(NonSubscriptionStatus.PENDING);
		
		String orderID = UUID.randomUUID().toString();
		purchase.setOrderID(orderID);
		
		String userID = UUID.randomUUID().toString();
		purchase.setUserID(userID);
		
		long purchaseDateTime = LocalDateTime.now().atZone(ZoneOffset.systemDefault()).toInstant().toEpochMilli();
		purchase.setPurchaseDateTime(purchaseDateTime);

		PurchaseReceipt purchaseReceipt = PurchasesFactory.eINSTANCE.createPurchaseReceipt();
		purchaseReceipt.setServerVerificationData("TEST");
		purchaseReceipt.setLocalVerificationData("TEST");
		purchase.setReceipt(purchaseReceipt);

		Purchase savedPurchase = purchasesService.savePurchase(userID, purchase);
		assertThat(savedPurchase).isNotNull();
		assertThat(savedPurchase.getSource()).isEqualTo(purchase.getSource());
		assertThat(savedPurchase.getType()).isEqualTo(purchase.getType());
		assertThat(savedPurchase.getProductID()).isEqualTo(purchase.getProductID());
		assertThat(savedPurchase.getOrderID()).isEqualTo(purchase.getOrderID());
		assertThat(savedPurchase.getPurchaseDateTime()).isEqualTo(purchase.getPurchaseDateTime());
	}
	
	@Test
	public void testUpdateGoogleNonSubscriptionPurchase(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PurchasesService> purchasesServiceAware) throws InterruptedException {

		assertThat(purchasesServiceAware.getServices()).hasSize(1);
		PurchasesService purchasesService = purchasesServiceAware.getService();
		assertThat(purchasesService).isNotNull();
		
		Purchase purchase = PurchasesFactory.eINSTANCE.createPurchase();
		purchase.setSource(InAppProductSource.GOOGLE_PLAY);
		purchase.setType(PurchaseType.NON_SUBSCRIPTION);
		purchase.setProductID(IAP_EXISTING_PRODUCT_ID_EXAMPLE);
		purchase.setStatus(NonSubscriptionStatus.VERIFIED);
		
		String orderID = UUID.randomUUID().toString();
		purchase.setOrderID(orderID);
		
		String userID = UUID.randomUUID().toString();
		purchase.setUserID(userID);
		
		long purchaseDateTime = LocalDateTime.now().atZone(ZoneOffset.systemDefault()).toInstant().toEpochMilli();
		purchase.setPurchaseDateTime(purchaseDateTime);

		PurchaseReceipt purchaseReceipt = PurchasesFactory.eINSTANCE.createPurchaseReceipt();
		purchaseReceipt.setServerVerificationData("TEST");
		purchaseReceipt.setLocalVerificationData("TEST");
		purchase.setReceipt(purchaseReceipt);

		Purchase savedPurchase = purchasesService.savePurchase(userID, purchase);
		assertThat(savedPurchase).isNotNull();
		assertThat(savedPurchase.getSource()).isEqualTo(purchase.getSource());
		assertThat(savedPurchase.getType()).isEqualTo(purchase.getType());
		assertThat(savedPurchase.getProductID()).isEqualTo(purchase.getProductID());
		assertThat(savedPurchase.getOrderID()).isEqualTo(purchase.getOrderID());
		assertThat(savedPurchase.getPurchaseDateTime()).isEqualTo(purchase.getPurchaseDateTime());
		
		TimeUnit.MILLISECONDS.sleep(2000);
		
		Purchase purchase2 = EcoreUtil.copy(savedPurchase);
		purchase2.setId(null);
		purchase2.setStatus(NonSubscriptionStatus.COMPLETED);
		
		Purchase savedPurchase2 = purchasesService.savePurchase(userID, purchase2);
		assertThat(savedPurchase2).isNotNull();
		assertThat(savedPurchase2.getId()).isEqualTo(savedPurchase.getId());
		assertThat(savedPurchase2.getSource()).isEqualTo(savedPurchase.getSource());
		assertThat(savedPurchase2.getType()).isEqualTo(savedPurchase.getType());
		assertThat(savedPurchase2.getProductID()).isEqualTo(savedPurchase.getProductID());
		assertThat(savedPurchase2.getOrderID()).isEqualTo(savedPurchase.getOrderID());
		assertThat(savedPurchase2.getPurchaseDateTime()).isEqualTo(savedPurchase.getPurchaseDateTime());
		assertThat(savedPurchase2.getStatus()).isEqualTo(purchase2.getStatus());
	}	

	@Test
	public void testSaveAppleNonSubscriptionPurchase(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PurchasesService> purchasesServiceAware) {

		assertThat(purchasesServiceAware.getServices()).hasSize(1);
		PurchasesService purchasesService = purchasesServiceAware.getService();
		assertThat(purchasesService).isNotNull();
		
		Purchase purchase = PurchasesFactory.eINSTANCE.createPurchase();
		purchase.setSource(InAppProductSource.APP_STORE);
		purchase.setType(PurchaseType.NON_SUBSCRIPTION);
		purchase.setProductID(IAP_EXISTING_PRODUCT_ID_EXAMPLE);
		purchase.setStatus(NonSubscriptionStatus.PENDING);
		
		String orderID = UUID.randomUUID().toString();
		purchase.setOrderID(orderID);
		
		String userID = UUID.randomUUID().toString();
		purchase.setUserID(userID);
		
		long purchaseDateTime = LocalDateTime.now().atZone(ZoneOffset.systemDefault()).toInstant().toEpochMilli();
		purchase.setPurchaseDateTime(purchaseDateTime);

		PurchaseReceipt purchaseReceipt = PurchasesFactory.eINSTANCE.createPurchaseReceipt();
		purchaseReceipt.setServerVerificationData("TEST");
		purchaseReceipt.setLocalVerificationData("TEST");
		purchase.setReceipt(purchaseReceipt);		

		Purchase savedPurchase = purchasesService.savePurchase(userID, purchase);
		assertThat(savedPurchase).isNotNull();
		assertThat(savedPurchase.getSource()).isEqualTo(purchase.getSource());
		assertThat(savedPurchase.getType()).isEqualTo(purchase.getType());
		assertThat(savedPurchase.getProductID()).isEqualTo(purchase.getProductID());
		assertThat(savedPurchase.getOrderID()).isEqualTo(purchase.getOrderID());
		assertThat(savedPurchase.getPurchaseDateTime()).isEqualTo(purchase.getPurchaseDateTime());
	}
	
	@Test
	public void testUpdateAppleNonSubscriptionPurchase(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PurchasesService> purchasesServiceAware) throws InterruptedException {

		assertThat(purchasesServiceAware.getServices()).hasSize(1);
		PurchasesService purchasesService = purchasesServiceAware.getService();
		assertThat(purchasesService).isNotNull();
		
		Purchase purchase = PurchasesFactory.eINSTANCE.createPurchase();
		purchase.setSource(InAppProductSource.APP_STORE);
		purchase.setType(PurchaseType.NON_SUBSCRIPTION);
		purchase.setProductID(IAP_EXISTING_PRODUCT_ID_EXAMPLE);
		purchase.setStatus(NonSubscriptionStatus.VERIFIED);
		
		String orderID = UUID.randomUUID().toString();
		purchase.setOrderID(orderID);
		
		String userID = UUID.randomUUID().toString();
		purchase.setUserID(userID);
		
		long purchaseDateTime = LocalDateTime.now().atZone(ZoneOffset.systemDefault()).toInstant().toEpochMilli();
		purchase.setPurchaseDateTime(purchaseDateTime);

		PurchaseReceipt purchaseReceipt = PurchasesFactory.eINSTANCE.createPurchaseReceipt();
		purchaseReceipt.setServerVerificationData("TEST");
		purchaseReceipt.setLocalVerificationData("TEST");
		purchase.setReceipt(purchaseReceipt);

		Purchase savedPurchase = purchasesService.savePurchase(userID, purchase);
		assertThat(savedPurchase).isNotNull();
		assertThat(savedPurchase.getSource()).isEqualTo(purchase.getSource());
		assertThat(savedPurchase.getType()).isEqualTo(purchase.getType());
		assertThat(savedPurchase.getProductID()).isEqualTo(purchase.getProductID());
		assertThat(savedPurchase.getOrderID()).isEqualTo(purchase.getOrderID());
		assertThat(savedPurchase.getPurchaseDateTime()).isEqualTo(purchase.getPurchaseDateTime());
		
		TimeUnit.MILLISECONDS.sleep(2000);
		
		Purchase purchase2 = EcoreUtil.copy(savedPurchase);
		purchase2.setId(null);
		purchase2.setStatus(NonSubscriptionStatus.COMPLETED);
		
		Purchase savedPurchase2 = purchasesService.savePurchase(userID, purchase2);
		assertThat(savedPurchase2).isNotNull();
		assertThat(savedPurchase2.getId()).isEqualTo(savedPurchase.getId());
		assertThat(savedPurchase2.getSource()).isEqualTo(savedPurchase.getSource());
		assertThat(savedPurchase2.getType()).isEqualTo(savedPurchase.getType());
		assertThat(savedPurchase2.getProductID()).isEqualTo(savedPurchase.getProductID());
		assertThat(savedPurchase2.getOrderID()).isEqualTo(savedPurchase.getOrderID());
		assertThat(savedPurchase2.getPurchaseDateTime()).isEqualTo(savedPurchase.getPurchaseDateTime());
		assertThat(savedPurchase2.getStatus()).isEqualTo(purchase2.getStatus());
	}	

	@Test
	public void testGetPurchaseHistory(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PurchasesService> purchasesServiceAware)
			throws InterruptedException {

		assertThat(purchasesServiceAware.getServices()).hasSize(1);
		PurchasesService purchasesService = purchasesServiceAware.getService();
		assertThat(purchasesService).isNotNull();

		Purchase purchase = PurchasesFactory.eINSTANCE.createPurchase();
		purchase.setSource(InAppProductSource.GOOGLE_PLAY);
		purchase.setType(PurchaseType.NON_SUBSCRIPTION);
		purchase.setProductID(IAP_EXISTING_PRODUCT_ID_EXAMPLE);
		purchase.setStatus(NonSubscriptionStatus.PENDING);
		
		String orderID = UUID.randomUUID().toString();
		purchase.setOrderID(orderID);
		
		String userID = UUID.randomUUID().toString();
		purchase.setUserID(userID);
		
		long purchaseDateTime = LocalDateTime.now().atZone(ZoneOffset.systemDefault()).toInstant().toEpochMilli();
		purchase.setPurchaseDateTime(purchaseDateTime);

		PurchaseReceipt purchaseReceipt = PurchasesFactory.eINSTANCE.createPurchaseReceipt();
		purchaseReceipt.setServerVerificationData("TEST");
		purchaseReceipt.setLocalVerificationData("TEST");
		purchase.setReceipt(purchaseReceipt);

		Purchase savedPurchase = purchasesService.savePurchase(userID, purchase);
		assertThat(savedPurchase).isNotNull();

		TimeUnit.MILLISECONDS.sleep(2000);

		PurchaseHistory purchaseHistory = purchasesService.getPurchaseHistory(userID, Optional.of(1));
		assertThat(purchaseHistory).isNotNull();
		assertThat(purchaseHistory.getPurchases()).isNotNull();
		assertThat(purchaseHistory.getPurchases()).isNotEmpty();
		assertThat(purchaseHistory.getPurchases()).hasSize(1);
	}

	@BeforeEach
	@AfterEach
	public void clean(
			@InjectService(cardinality = 1, timeout = 5000, filter = "(database=playertour)") ServiceAware<MongoDatabaseProvider> dbProviderAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PurchasesPurchaseIndexService> purchasesPurchaseIndexServiceAware) {

		assertThat(dbProviderAware.getServices()).hasSize(1);
		MongoDatabaseProvider dbProvider = dbProviderAware.getService();
		assertThat(dbProvider).isNotNull();

		assertThat(purchasesPurchaseIndexServiceAware.getServices()).hasSize(1);
		PurchasesPurchaseIndexService purchasesPurchaseIndexService = purchasesPurchaseIndexServiceAware.getService();
		assertThat(purchasesPurchaseIndexService).isNotNull();

		try {
			dbProvider.getDatabase().getCollection(PurchasesPackage.Literals.PURCHASE.getName())
					.drop();
			dbProvider.getDatabase().getCollection(PurchasesPackage.Literals.PURCHASE_RECEIPT.getName()).drop();
		} catch (IllegalArgumentException e) {
			System.out.println("Collection does not exist. Nothing to remove.");
		}

		purchasesPurchaseIndexService.resetIndex();
	}

	@AfterAll
	public static void brutalClean(
			@InjectService(cardinality = 1, timeout = 5000, filter = "(database=playertour)") ServiceAware<MongoDatabaseProvider> dbProviderAware) {
		assertThat(dbProviderAware.getServices()).hasSize(1);
		MongoDatabaseProvider dbProvider = dbProviderAware.getService();
		assertThat(dbProvider).isNotNull();

		MongoDatabase database = dbProvider.getDatabase();
		database.drop();
	}
}
