/**
import org.osgi.service.jaxrs.whiteboard.annotations.RequireJaxrsWhiteboard; * Copyright (c) 2012 - 2018 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package com.playertour.backend.rest.api.course;

import java.util.List;

import javax.ws.rs.GET;
import javax.ws.rs.Path;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.Response.Status;

import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.annotations.ServiceScope;
import org.osgi.service.http.whiteboard.annotations.RequireHttpWhiteboard;
import org.osgi.service.jaxrs.whiteboard.propertytypes.JaxrsName;
import org.osgi.service.jaxrs.whiteboard.propertytypes.JaxrsResource;

import com.playertour.backend.api.GolfCourseSearchResult;
import com.playertour.backend.api.GolfCourseSearchResults;
import com.playertour.backend.api.PlayerApiFactory;
import com.playertour.backend.apis.mmt.common.UnknownTransformationException;
import com.playertour.backend.apis.player.APIGolfCourseService;
import com.playertour.backend.rest.api.constants.ApiConstants;

import io.swagger.v3.oas.annotations.OpenAPIDefinition;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.info.Info;
import io.swagger.v3.oas.annotations.media.Content;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.responses.ApiResponse;


@RequireHttpWhiteboard
@JaxrsResource
@OpenAPIDefinition(info = @Info(title = "Playertour API", version = ApiConstants.VERSION))
@JaxrsName("Golfcourse Resource")
@Component(service = GolfCourseResource.class, enabled = true, scope = ServiceScope.PROTOTYPE)
@Path("/courses")
public class GolfCourseResource {

	@Reference
	private APIGolfCourseService service;
	
	@GET
	@Path("nearMe")
	@Produces(MediaType.APPLICATION_JSON)
	@Operation(description = "Searches for Golf Courses near a certain position",
    	responses = {  
                 @ApiResponse(
                		 responseCode = "200",
                		 description = "Returns a GolfCourseSearchResults with the list of the GolfCourse matching the search criteria.",
                		 content = {
                		 @Content(mediaType = MediaType.APPLICATION_JSON,
                				 schema = @Schema(   
                						 implementation = GolfCourseSearchResult.class
                						 ))
                 }),
                 @ApiResponse(
                		 responseCode = "401",
                		 description = "If no or Invalid token is provided."
                		 ),
                 @ApiResponse(
                		 responseCode = "500",
                		 description = "If an invalid value of latitude or longitude is provided, or if the server could not find "
                		 		+ "a suitable transformation for the data."
                		 )
				}
			)
	public Response findCoursesNearMe(@QueryParam("lat") String lat, 
			@QueryParam("lng") String lng, 
			@QueryParam("maxDistance") @Parameter(description = "In km. Default is 2 km", required = false)  Integer maxDistanceInKm) {
		List<GolfCourseSearchResult> courses;
		try {
			if(maxDistanceInKm == null) {
				maxDistanceInKm = 2;
			}
			courses = service.findCoursesNearMe(lat, lng, maxDistanceInKm);
			GolfCourseSearchResults result = PlayerApiFactory.eINSTANCE.createGolfCourseSearchResults();
			result.getResults().addAll(courses);			
			return Response.ok(result).build();
		} catch (UnknownTransformationException e) {
			return Response.status(Status.INTERNAL_SERVER_ERROR.getStatusCode(), "Unknown m2m transformation!").build();			
		} catch(IllegalArgumentException e) {
			return Response.status(Status.INTERNAL_SERVER_ERROR.getStatusCode(), "Error parsing latitude or longitude. Please, check your syntax.").build();		
		}
		
	}

	@GET
	@Path("search")
	@Produces(MediaType.APPLICATION_JSON)
	@Operation(description = "Searches for Golf Courses by name with a given query.",
	responses = {  
			@ApiResponse(
					responseCode = "200",
	                description = "Returns a GolfCourseSearchResults with the list of the GolfCourse matching the search criteria.",
					content = {
							@Content(mediaType = MediaType.APPLICATION_JSON,
									schema = @Schema(
											implementation = GolfCourseSearchResults.class))
					}),
			@ApiResponse(
					responseCode = "401",
					description = "If no or Invalid token is provided."
					),
            @ApiResponse(
           		 responseCode = "500",
           		 description = "If the server could not find a suitable transformation for the data."
           		 )
	}
			)
	public Response findCourses(@QueryParam("query") String query, 
			@QueryParam("results") @Parameter(description = "The max Number of results")  int maxResults) {
		List<GolfCourseSearchResult> courses;
		try {
			courses = service.findCourses(query, maxResults);
			GolfCourseSearchResults result = PlayerApiFactory.eINSTANCE.createGolfCourseSearchResults();
			result.getResults().addAll(courses);
			return Response.ok(result).build();
		} catch (UnknownTransformationException e) {
			return Response.status(Status.INTERNAL_SERVER_ERROR.getStatusCode(), "Unknown m2m transformation!").build();			
		} 		
	}
}
