/**
import org.osgi.service.jaxrs.whiteboard.annotations.RequireJaxrsWhiteboard; * Copyright (c) 2012 - 2018 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package com.playertour.backend.rest.api.score;

import java.util.List;

import javax.ws.rs.Consumes;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.Path;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;

import org.gecko.emf.osgi.rest.jaxrs.annotation.json.RootElement;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.annotations.ServiceScope;
import org.osgi.service.http.whiteboard.annotations.RequireHttpWhiteboard;
import org.osgi.service.jaxrs.whiteboard.propertytypes.JaxrsName;
import org.osgi.service.jaxrs.whiteboard.propertytypes.JaxrsResource;

import com.playertour.backend.api.HoleScore;
import com.playertour.backend.api.PlayerApiFactory;
import com.playertour.backend.api.ScoreCard;
import com.playertour.backend.api.ScorecardSearchResults;
import com.playertour.backend.api.Stroke;
import com.playertour.backend.apis.player.APIScorecardService;

import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.media.Content;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.responses.ApiResponse;


@RequireHttpWhiteboard
@JaxrsResource
@JaxrsName("Scorecard Resource")
@Component(service = ScoreCardResource.class, enabled = true, scope = ServiceScope.PROTOTYPE)
@Path("/scorecard")
public class ScoreCardResource {

	@Reference
	private APIScorecardService service;
	
	private static final String SCORECARD_URI = "http://theplayertour.com/backend/api/1.0#//ScoreCard";
	private static final String STROKE_URI = "http://theplayertour.com/backend/api/1.0#//Stroke";
	private static final String HOLE_SCORE_URI = "http://theplayertour.com/backend/api/1.0#//HoleScore";
	
	@GET
	@Path("scorecards")
	@Produces(MediaType.APPLICATION_JSON)
	@Operation(description = "Get all the Scorecards for this user, or null, if there is none",
    	responses = {  
                 @ApiResponse(
                		 responseCode = "200",
                   		 description = "Returns a ScorecardSearchResults with the list of ScoreCard matching the search criteria.",
                		 content = {
                		 @Content(mediaType = MediaType.APPLICATION_JSON,
                				 schema = @Schema(
                						 implementation = ScorecardSearchResults.class))
                 }),
                 @ApiResponse(
                		 responseCode = "401",
                		 description = "If no or Invalid token is provided."
                		 ),
                 @ApiResponse(
                		 responseCode = "204",
                		 description = "If the list of scorecards is empty."
                		 )
				}
			)
	public Response getScorecards() {
		String userId = "abcdef";
		List<ScoreCard> scorecards =  service.getScorecardsByUser(userId);
		ScorecardSearchResults result = PlayerApiFactory.eINSTANCE.createScorecardSearchResults();
		if(result.getResults().isEmpty()) {
			return Response.noContent().build();
		}
		result.getResults().addAll(scorecards);
		return Response.ok().entity(result).build();
	}
	
	@GET
	@Path("course/scorecards")
	@Produces(MediaType.APPLICATION_JSON)
	@Operation(description = "Get all the Scorecards for the provided course id and this user, or null, if there is none",
    	responses = {  
                 @ApiResponse(
                		 responseCode = "200",
                		 description = "Returns a ScorecardSearchResults with the list of ScoreCard matching the search criteria.",
                		 content = {
                		 @Content(mediaType = MediaType.APPLICATION_JSON,
                				 schema = @Schema(
                						 implementation = ScorecardSearchResults.class))
                 }),
                 @ApiResponse(
                		 responseCode = "401",
                		 description = "If no or Invalid token is provided."
                		 ),
     			@ApiResponse(
    					responseCode = "500",
    					description = "If an internal error occured"
    					),
                @ApiResponse(
               		 responseCode = "204",
               		 description = "If the list of scorecards is empty."
               		 )
				}
			)
	public Response getScorecardsByCourse(@QueryParam("courseId") String courseId) {
		String userId = "abcdef";
		List<ScoreCard> scorecards =  service.getScorecardsByCourse(userId, courseId);
		ScorecardSearchResults result = PlayerApiFactory.eINSTANCE.createScorecardSearchResults();
		if(result.getResults().isEmpty()) {
			return Response.noContent().build();
		}
		result.getResults().addAll(scorecards);
		return Response.ok().entity(result).build();
	}
	


	@POST
	@Path("save")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	@Operation(description = "Saves the ScoreCard for this user. This should be called when the whole Scorecard is filled "
			+ "manually after and not gradually.",
	
	responses = {  
			@ApiResponse(
					responseCode = "200",
					description = "Returns the saved ScoreCard",
					content = {
							@Content(mediaType = MediaType.APPLICATION_JSON,
									schema = @Schema(
											implementation = ScoreCard.class))
					}),
			@ApiResponse(
					responseCode = "401",
					description = "If no or Invalid token is provided."
					),
			@ApiResponse(
					responseCode = "500",
					description = "If an internal error occured"
					)
	}
			)
	public Response saveScoreCard(@RootElement(rootClassUri = SCORECARD_URI) ScoreCard scorecard) {

		String userId = "abcdef";
		ScoreCard scoreCard = service.saveScorecard(userId, scorecard);
		return Response.ok().entity(scoreCard).build();
	}
	
	@GET
	@Path("open")
	@Produces(MediaType.APPLICATION_JSON)
	@Operation(description = "Open a new Scorecard for the provided course id and this user. This should be the first step "
			+ "of a gradual filling of the Scorecard",
    	responses = {  
                 @ApiResponse(
                		 responseCode = "200",
                		 description = "Returns the new ScoreCard",
                		 content = {
                		 @Content(mediaType = MediaType.APPLICATION_JSON,
                				 schema = @Schema(
                						 implementation = ScoreCard.class))
                 }),
                 @ApiResponse(
                		 responseCode = "401",
                		 description = "If no or Invalid token is provided."
                		 ),
     			@ApiResponse(
    					responseCode = "500",
    					description = "If an internal error occured"
    					)
				}
			)
	public Response openScorecard(@QueryParam("courseId") String courseId) {
		ScoreCard scorecard =  service.openScorecard(courseId);
		return Response.ok().entity(scorecard).build();
	}
	
	@POST
	@Path("stroke")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	@Operation(description = "Save the submitted Stroke to the Scorecard with the provided id. "
			+ "Returns the updated Scorecard.",
	responses = {  
			@ApiResponse(
					responseCode = "200",
					description = "Returns the updated ScoreCard",
					content = {
							@Content(mediaType = MediaType.APPLICATION_JSON,
									schema = @Schema(
											implementation = ScoreCard.class))
					}),
			@ApiResponse(
					responseCode = "401",
					description = "If no or Invalid token is provided."
					),
			@ApiResponse(
					responseCode = "500",
					description = "If an internal error occured"
					)
	}
			)
	public Response submitStroke(@QueryParam("scorecardId") String scorecardId, @RootElement(rootClassUri = STROKE_URI)  Stroke stroke) {
		ScoreCard scoreCard = service.submitStroke(scorecardId, stroke);
		return Response.ok().entity(scoreCard).build();
	}
	
	@POST
	@Path("holescore")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	@Operation(description = "Save the submitted HoleScore to the Scorecard with the provided id. "
			+ "Returns the updated Scorecard.",
	responses = {  
			@ApiResponse(
					responseCode = "200",
					description = "Returns the updated ScoreCard",
					content = {
							@Content(mediaType = MediaType.APPLICATION_JSON,
									schema = @Schema(
											implementation = ScoreCard.class))
					}),
			@ApiResponse(
					responseCode = "401",
					description = "If no or Invalid token is provided."
					),
			@ApiResponse(
					responseCode = "500",
					description = "If an internal error occured"
					)
	}
			)
	public Response closeHole(@QueryParam("scorecardId") String scorecardId, @RootElement(rootClassUri = HOLE_SCORE_URI)  HoleScore holeScore) {
		ScoreCard scoreCard = service.closeHole(scorecardId, holeScore);
		return Response.ok().entity(scoreCard).build();
	}
	
	@GET
	@Path("close")
	@Produces(MediaType.APPLICATION_JSON)
	@Operation(description = "Close the Scorecard with the provided id for this user. This should be the last step "
			+ "of a gradual filling of the Scorecard",
    	responses = {  
                 @ApiResponse(
                		 responseCode = "200",
                		 description = "Returns the closed ScoreCard",
                		 content = {
                		 @Content(mediaType = MediaType.APPLICATION_JSON,
                				 schema = @Schema(
                						 implementation = ScoreCard.class))
                 }),
                 @ApiResponse(
                		 responseCode = "401",
                		 description = "If no or Invalid token is provided."
                		 ),
     			@ApiResponse(
    					responseCode = "500",
    					description = "If an internal error occured"
    					)
				}
			)
	public Response closeScorecard(@QueryParam("scorecardId") String scorecardId) {
		String userId = "abcdef";
		ScoreCard scorecard =  service.closeScorecard(userId, scorecardId);
		return Response.ok().entity(scorecard).build();
	}
	
}
