/**
 * Copyright (c) 2012 - 2018 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package com.playertour.backend.search.tests;

import static org.assertj.core.api.Assertions.assertThat;

import org.gecko.emf.repository.EMFRepository;
import org.gecko.mongo.osgi.MongoDatabaseProvider;
import org.junit.jupiter.api.AfterAll;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.junit.platform.commons.annotation.Testable;
import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceReference;
import org.osgi.test.common.annotation.InjectBundleContext;
import org.osgi.test.common.annotation.InjectService;
import org.osgi.test.common.service.ServiceAware;
import org.osgi.test.junit5.context.BundleContextExtension;
import org.osgi.test.junit5.service.ServiceExtension;

import com.mongodb.client.MongoDatabase;
import com.playertour.backend.apis.search.SearchService;
import com.playertour.backend.search.model.search.Search;
import com.playertour.backend.search.model.search.SearchPackage;


/**
 * Tests the implementation of the SearchService
 * @since 1.0
 */
@Testable
@ExtendWith(BundleContextExtension.class)
@ExtendWith(ServiceExtension.class)
public class SearchServiceIntegrationTest  {
	
	
	@InjectBundleContext
	BundleContext bundleContext;
	

	@Test
	public void testServices(@InjectService(cardinality = 1, timeout = 5000) ServiceAware<SearchService> searchAware, 
			@InjectService(cardinality = 1, timeout = 5000, filter = "(repo_id=playertour.playertour)") ServiceAware<EMFRepository> repoAware) {
				
		assertThat(searchAware.getServices()).hasSize(1);	
		ServiceReference<SearchService> searchRef = searchAware.getServiceReference();
		assertThat(searchRef).isNotNull();	
		
		assertThat(repoAware.getServices()).hasSize(1);	
		ServiceReference<EMFRepository> repoRef = repoAware.getServiceReference();
		assertThat(repoRef).isNotNull();	
	}
	
	@Test
	public void testSaveSearchFirstTime(@InjectService(cardinality = 1, timeout = 5000) ServiceAware<SearchService> searchAware) {
		
		assertThat(searchAware.getServices()).hasSize(1);	
		SearchService searchService = searchAware.getService();
		assertThat(searchService).isNotNull();	
		
		Search search = searchService.saveUserSearch("Club Jena");
		assertThat(search).isNotNull();
		assertThat(search.getId()).isNotNull();
		assertThat(search.getQuery()).isEqualTo("Club Jena");
		assertThat(search.getCount()).isEqualTo(1);		
	}
	
	
	@Test
	public void testSaveSearchSecondTime(@InjectService(cardinality = 1, timeout = 5000) ServiceAware<SearchService> searchAware) {
		
		assertThat(searchAware.getServices()).hasSize(1);	
		SearchService searchService = searchAware.getService();
		assertThat(searchService).isNotNull();	
		
		Search search1 = searchService.saveUserSearch("Club Jena");
		assertThat(search1).isNotNull();
		assertThat(search1.getId()).isNotNull();
		
		Search search2 = searchService.saveUserSearch(" Club Jena");
		assertThat(search2).isNotNull();
		assertThat(search1.getId()).isEqualTo(search2.getId());
		assertThat(search2.getCount()).isEqualTo(2);
		
	}
	
	@BeforeEach
	@AfterEach
	public void clean(@InjectService(cardinality = 1, timeout = 5000, filter = "(database=playertour)") ServiceAware<MongoDatabaseProvider> dbProviderAware) {
		
		assertThat(dbProviderAware.getServices()).hasSize(1);	
		MongoDatabaseProvider dbProvider = dbProviderAware.getService();
		assertThat(dbProvider).isNotNull();	
		
		MongoDatabase database = dbProvider.getDatabase();
		try {
			database.getCollection(SearchPackage.Literals.SEARCH.getName()).drop();
		} catch (IllegalArgumentException e) {
			System.out.println("Collection does not exist. Nothing to remove.");
		}
	}
	
	@AfterAll
	public static void brutalClean(@InjectService(cardinality = 1, timeout = 5000, filter = "(database=playertour)") ServiceAware<MongoDatabaseProvider> dbProviderAware) {
		assertThat(dbProviderAware.getServices()).hasSize(1);	
		MongoDatabaseProvider dbProvider = dbProviderAware.getService();
		assertThat(dbProvider).isNotNull();	
		
		MongoDatabase database = dbProvider.getDatabase();
		database.drop();
	}	
	
}
