/**
 * Copyright (c) 2012 - 2018 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package com.playertour.backend.search;

import org.gecko.emf.repository.EMFRepository;
import org.gecko.emf.repository.query.IQuery;
import org.gecko.emf.repository.query.QueryRepository;
import org.osgi.service.component.ComponentServiceObjects;
import org.osgi.service.component.annotations.*;

import com.playertour.backend.apis.search.SearchService;
import com.playertour.backend.search.model.search.Search;
import com.playertour.backend.search.model.search.SearchFactory;
import com.playertour.backend.search.model.search.SearchPackage;

@Component(name = "SearchService", scope = ServiceScope.PROTOTYPE)
public class SearchServiceImpl implements SearchService {

	@Reference(target = "(repo_id=playertour.playertour)", cardinality = ReferenceCardinality.MANDATORY)
	private ComponentServiceObjects<EMFRepository> repositoryServiceObjects;
	
	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.apis.search.SearchService#saveUserSearch(java.lang.String)
	 */
	@Override
	public Search saveUserSearch(String search) {
		
		if(search == null) {
			throw new IllegalArgumentException("Cannot save null Search object!");
		}
		EMFRepository repository = repositoryServiceObjects.getService();
		try {	
			search = search.trim();
			QueryRepository queryRepo = (QueryRepository) repository.getAdapter(QueryRepository.class);
			IQuery query = queryRepo.createQueryBuilder().column(SearchPackage.Literals.SEARCH__QUERY).simpleValue(search).build();
			Search existingSearch = queryRepo.getEObjectByQuery(SearchPackage.Literals.SEARCH, query, null);
			if(existingSearch != null) {
				existingSearch.setCount(existingSearch.getCount()+1);
				repository.save(existingSearch);
				return existingSearch;
			}
			else {
				Search newSearch = SearchFactory.eINSTANCE.createSearch();
				newSearch.setQuery(search);
				newSearch.setCount(1);
				repository.save(newSearch);
				return newSearch;
			}
		} finally {
			repositoryServiceObjects.ungetService(repository);
		}
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.apis.search.SearchService#getUserSearchById(java.lang.String)
	 */
	@Override
	public Search getUserSearchById(String searchId) {
		if(searchId == null) {
			throw new IllegalArgumentException("Cannot get Search object with null id!");
		}
		EMFRepository repository = repositoryServiceObjects.getService();
		try {	
			Search search = repository.getEObject(SearchPackage.Literals.SEARCH, searchId);
			return search;
		} finally {
			repositoryServiceObjects.ungetService(repository);
		}
	}

}
