/**
 * Copyright (c) 2012 - 2022 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package com.playertour.backend.security.keycloakadmin.service.impl;

import java.net.URI;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.TimeUnit;

import org.apache.commons.lang3.StringUtils;
import org.apache.http.Header;
import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.NameValuePair;
import org.apache.http.client.HttpClient;
import org.apache.http.client.config.CookieSpecs;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.HttpDelete;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.conn.HttpClientConnectionManager;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import org.apache.http.message.BasicNameValuePair;
import org.apache.http.util.EntityUtils;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.ComponentPropertyType;
import org.osgi.service.component.annotations.ConfigurationPolicy;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.annotations.ServiceScope;
import org.osgi.service.log.Logger;
import org.osgi.service.log.LoggerFactory;
import org.osgi.service.metatype.annotations.Designate;
import org.osgi.service.metatype.annotations.ObjectClassDefinition;

import com.google.gson.Gson;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import com.playertour.backend.security.keycloakadmin.service.api.KeycloakAdminService;

@Component(name = "KeycloakAdminService", scope = ServiceScope.SINGLETON, configurationPolicy = ConfigurationPolicy.REQUIRE, configurationPid = "KeycloakAdminService")
@Designate(ocd = KeycloakAdminServiceImpl.Config.class)
public class KeycloakAdminServiceImpl implements KeycloakAdminService {
	
	private static final int DEFAULT_MAX_CONNECTIONS_TOTAL = 200;
	private static final int DEFAULT_CONNECTIONS_PER_ROUTE = 100;

	private static final int DEFAULT_CONNECTION_TIMEOUT_MS = (10 * 1000);
	private static final int DEFAULT_READ_TIMEOUT_MS = (10 * 1000);
	private static final int DEFAULT_WAIT_TIMEOUT_MS = (10 * 1000);	
	
	private final static String KEYCLOAK_AUTH_TOKEN_MEMBER_NAME = "access_token";
	private final static String KEYCLOAK_AUTH_TOKEN_PATH = "/realms/%s/protocol/openid-connect/token";
	private final static String KEYCLOAK_ADMIN_REST_API_USERS_PATH = "/admin/realms/%s/users";
	private final static String KEYCLOAK_ADMIN_REST_API_USERS_COUNT_PATH = "/admin/realms/%s/users/count";
	
	@Reference(service = LoggerFactory.class)
	private Logger logger;
	
	private Config config;
	
	private static final Gson GSON = new Gson();

	private HttpClient httpClient;

	@ComponentPropertyType
	@ObjectClassDefinition
	public @interface Config {
		String serverBaseUri();
		
		String realmName();
		
		String clientId();
		
		String clientSecret();
		
		String adminUsername();
		
		String adminPassword();
	}

	@Activate
	public void activate(Config config) {
		Objects.requireNonNull(config.serverBaseUri(), "Keycloak server base URI is required!");
		Objects.requireNonNull(config.realmName(), "Keycloak realm name is required!");
		Objects.requireNonNull(config.clientId(), "Keycloak client id is required!");
		Objects.requireNonNull(config.clientSecret(), "Keycloak client secret is required!");
		Objects.requireNonNull(config.adminUsername(), "Keycloak admin username is required!");
		Objects.requireNonNull(config.adminPassword(), "Keycloak admin password is required!");
		
		this.config = config;
		
		try {

			int maxConnectionsTotal = DEFAULT_MAX_CONNECTIONS_TOTAL;
			int maxConnectionsPerRoute = DEFAULT_CONNECTIONS_PER_ROUTE;
			int connectTimeout = DEFAULT_CONNECTION_TIMEOUT_MS;
			int readTimeout = DEFAULT_READ_TIMEOUT_MS;
			int waitTimeout = DEFAULT_WAIT_TIMEOUT_MS;

			httpClient = initializeHttpClient(maxConnectionsTotal, maxConnectionsPerRoute, connectTimeout, readTimeout,
					waitTimeout);

		} catch (Throwable t) {
			t.printStackTrace();
			logger.error("Exception was thrown while initializing HTTP client!");
		}
	}	

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.security.keycloakadmin.service.api.KeycloakAdminService#getUsersCount()
	 */
	@Override
	public int getUsersCount() {
		try {
			String keycloakAccessToken = getKeycloakAccessToken();
			Objects.requireNonNull(keycloakAccessToken, "Keycloak access token is required!");
			
			HttpGet keycloakUsersCountRequest = new HttpGet(constructKeycloakAdminRestAPIUsersCountUrl());
			keycloakUsersCountRequest.addHeader("Authorization", "Bearer " + keycloakAccessToken);
			
			HttpResponse keycloakUsersCountResponse = httpClient.execute(keycloakUsersCountRequest);
			
			if (keycloakUsersCountResponse.getStatusLine().getStatusCode() == 200) {
				HttpEntity keycloakUsersCountResponseHttpEntity = keycloakUsersCountResponse.getEntity();
				if (keycloakUsersCountResponseHttpEntity != null) {
					
					String keycloakUsersCountResponseString = EntityUtils.toString(keycloakUsersCountResponseHttpEntity);
					
					if (StringUtils.isNumeric(keycloakUsersCountResponseString)) {
						return Integer.parseInt(keycloakUsersCountResponseString);
					}
				}
			}

		} catch (Throwable t) {
			t.printStackTrace();
		}
		
		return -1;
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.security.keycloakadmin.service.api.KeycloakAdminService#deleteUser(java.lang.String)
	 */
	@Override
	public boolean deleteUser(String userid) {
		try {
			String keycloakAccessToken = getKeycloakAccessToken();
			Objects.requireNonNull(keycloakAccessToken, "Keycloak access token is required!");			
			
			HttpDelete keycloakDeleteUserRequest = new HttpDelete(constructKeycloakAdminRestAPIDeleteUserUrl(userid));
			keycloakDeleteUserRequest.addHeader("Authorization", "Bearer " + keycloakAccessToken);
			
			HttpResponse keycloakDeleteUserResponse = httpClient.execute(keycloakDeleteUserRequest);
			
			if (keycloakDeleteUserResponse.getStatusLine().getStatusCode() == 204) {
				return true;
			}

		} catch (Throwable t) {
			t.printStackTrace();
		}
		
		return false;
	}	

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.security.keycloakadmin.service.api.KeycloakAdminService#createUser(java.lang.String)
	 */
	@Override
	public String createUser(String username) {
		try {
			String keycloakAccessToken = getKeycloakAccessToken();
			Objects.requireNonNull(keycloakAccessToken, "Keycloak access token is required!");
			
			HttpPost keycloakCreateUserRequest = new HttpPost(constructKeycloakAdminRestAPIUsersUrl());
			keycloakCreateUserRequest.addHeader("Authorization", "Bearer " + keycloakAccessToken);
			
			Map<String, String> payload = new HashMap<String, String>();
			payload.put("username", username);
			
			keycloakCreateUserRequest.setEntity(new StringEntity(GSON.toJson(payload), ContentType.APPLICATION_JSON));
			
			HttpResponse keycloakCreateUserResponse = httpClient.execute(keycloakCreateUserRequest);
		
			if (keycloakCreateUserResponse.getStatusLine().getStatusCode() == 201) {
				
				if (keycloakCreateUserResponse.containsHeader("Location")) {
					Header locationHeader = keycloakCreateUserResponse.getHeaders("Location")[0];
					
					URI locationURI = URI.create(locationHeader.getValue());
					String locationURIPath = locationURI.getPath();
					String[] locationURIPathParts = locationURIPath.split("/");
					
					String userId = locationURIPathParts[(locationURIPathParts.length - 1)];
					
					return userId;
				}
			}

		} catch (Throwable t) {
			t.printStackTrace();
		}
		
		return null;
	}
	
	private String getKeycloakAccessToken() {
		try {
			HttpPost keycloakTokenRequest = new HttpPost(constructKeycloakAccessTokenUrl());
			
		    final List<NameValuePair> params = new ArrayList<NameValuePair>();
		    params.add(new BasicNameValuePair("client_id", config.clientId()));
		    params.add(new BasicNameValuePair("grant_type", "password"));
		    params.add(new BasicNameValuePair("client_secret", config.clientSecret()));
		    params.add(new BasicNameValuePair("scope", "openid"));
		    params.add(new BasicNameValuePair("username", config.adminUsername()));
		    params.add(new BasicNameValuePair("password", config.adminPassword()));
		    
		    keycloakTokenRequest.setEntity(new UrlEncodedFormEntity(params));
			
			HttpResponse keycloakTokenResponse = httpClient.execute(keycloakTokenRequest);
			
			if (keycloakTokenResponse.getStatusLine().getStatusCode() == 200) {
				HttpEntity keycloakTokenResponseHttpEntity = keycloakTokenResponse.getEntity();
				if (keycloakTokenResponseHttpEntity != null) {
					String keycloakTokenResponseString = EntityUtils.toString(keycloakTokenResponseHttpEntity);
					JsonElement keycloakTokenResponseJsonElement = JsonParser.parseString(keycloakTokenResponseString);
					if (keycloakTokenResponseJsonElement.isJsonObject()) {
						JsonObject keycloakTokenResponseJsonObject = keycloakTokenResponseJsonElement.getAsJsonObject();
						if (keycloakTokenResponseJsonObject.has(KEYCLOAK_AUTH_TOKEN_MEMBER_NAME)) {
							return keycloakTokenResponseJsonObject.get(KEYCLOAK_AUTH_TOKEN_MEMBER_NAME).getAsString();
						}
					}
				}
			}

		} catch (Throwable t) {
			t.printStackTrace();
		}

		return null;
	}
	
	private String constructKeycloakAdminRestAPIDeleteUserUrl(String userid) {
		StringBuilder sb = new StringBuilder();
		sb.append(constructKeycloakAdminRestAPIUsersUrl());
		sb.append("/");
		sb.append(userid);
		return sb.toString();
	}	
	
	private String constructKeycloakAdminRestAPIUsersUrl() {
		StringBuilder sb = new StringBuilder();
		sb.append(config.serverBaseUri());
		sb.append(String.format(KEYCLOAK_ADMIN_REST_API_USERS_PATH, config.realmName()));		
		return sb.toString();
	}
	
	private String constructKeycloakAdminRestAPIUsersCountUrl() {
		StringBuilder sb = new StringBuilder();
		sb.append(config.serverBaseUri());
		sb.append(String.format(KEYCLOAK_ADMIN_REST_API_USERS_COUNT_PATH, config.realmName()));		
		return sb.toString();
	}
	
	private String constructKeycloakAccessTokenUrl() {
		StringBuilder sb = new StringBuilder();
		sb.append(config.serverBaseUri());
		sb.append(String.format(KEYCLOAK_AUTH_TOKEN_PATH, config.realmName()));		
		return sb.toString();
	}	
	
	private HttpClient initializeHttpClient(int maxConnectionsTotal, int maxConnectionsPerRoute, int connectTimeout,
			int readTimeout, int waitTimeout) {
		PoolingHttpClientConnectionManager connectionManager = new PoolingHttpClientConnectionManager();
		connectionManager.setMaxTotal(maxConnectionsTotal);
		connectionManager.setDefaultMaxPerRoute(maxConnectionsPerRoute);

		// @formatter:off
    	RequestConfig requestConfig = RequestConfig.custom()
    			.setConnectionRequestTimeout(waitTimeout)
    			.setSocketTimeout(readTimeout)
    			.setConnectTimeout(connectTimeout)				
                .setCookieSpec(CookieSpecs.IGNORE_COOKIES)
                .build();
    	// @formatter:on

		IdleConnectionMonitorThread staleMonitor = new IdleConnectionMonitorThread(connectionManager);
		staleMonitor.start();
		try {
			staleMonitor.join(1000);
		} catch (InterruptedException e) {
			e.printStackTrace();
			logger.error(e.getMessage());
		}

		// @formatter:off
        return HttpClientBuilder.create()
                .setConnectionManager(connectionManager)
                .setDefaultRequestConfig(requestConfig)
                .build();
        // @formatter:on
	}
	
	public static class IdleConnectionMonitorThread extends Thread {
		private final HttpClientConnectionManager connMgr;
		private volatile boolean shutdown;

		public IdleConnectionMonitorThread(HttpClientConnectionManager connMgr) {
			super();
			this.connMgr = connMgr;
		}

		@Override
		public void run() {
			try {
				while (!shutdown) {
					synchronized (this) {
						wait(5000);
						// Close expired connections
						connMgr.closeExpiredConnections();
						// Optionally, close connections that have been idle longer than 30 sec
						connMgr.closeIdleConnections(60, TimeUnit.SECONDS);
					}
				}
			} catch (InterruptedException ex) {
				// terminate
				shutdown();
			}
		}

		public void shutdown() {
			shutdown = true;
			synchronized (this) {
				notifyAll();
			}
		}
	}	
	
}
