/**
 * Copyright (c) 2012 - 2021 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package com.playertour.backend.security;

import java.io.IOException;
import java.util.Objects;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.osgi.service.cm.Configuration;
import org.osgi.service.cm.ConfigurationAdmin;
import org.pac4j.core.context.J2EContext;
import org.pac4j.core.profile.CommonProfile;
import org.pac4j.core.profile.ProfileManager;

/**
 * 
 * @author mhs
 * @since Apr 28, 2021
 */
public enum PlayerSecurityUtil {
	INSTANCE;

	public boolean isAuthProfileValid(HttpServletRequest request, HttpServletResponse response) {
		return (Objects.nonNull(request) && Objects.nonNull(response)
				&& isAuthProfileValid(new ProfileManager<CommonProfile>(new J2EContext(request, response))));
	}

	public boolean isAuthProfileValid(ProfileManager<CommonProfile> profileManager) {
		return isAuthProfileValid(profileManager.get(true).get());
	}

	public boolean isAuthProfileValid(CommonProfile authProfile) {
		return (Objects.nonNull(authProfile)
				&& authProfile.containsAttribute(PlayerSecurityConstants.PAC4J_PROFILE_USERNAME)
				&& !authProfile.getAttribute(PlayerSecurityConstants.PAC4J_PROFILE_USERNAME, String.class).isEmpty());
	}

	public String getAuthProfileUsername(HttpServletRequest request, HttpServletResponse response) {
		return getAuthProfileUsername(new ProfileManager<CommonProfile>(new J2EContext(request, response)));
	}

	public String getAuthProfileUsername(ProfileManager<CommonProfile> profileManager) {
		return getAuthProfileUsername(profileManager.get(true).get());
	}

	public String getAuthProfileUsername(CommonProfile authProfile) {
		return authProfile.getAttribute(PlayerSecurityConstants.PAC4J_PROFILE_USERNAME, String.class);
	}
	
	public String getAuthProfileId(HttpServletRequest request, HttpServletResponse response) {
		return getAuthProfileId(new ProfileManager<CommonProfile>(new J2EContext(request, response)));
	}
	
	public String getAuthProfileId(ProfileManager<CommonProfile> profileManager) {
		return getAuthProfileId(profileManager.get(true).get());
	}
	
	public String getAuthProfileId(CommonProfile authProfile) {
		return authProfile.getId();
	}

	public KeycloakConfiguration getKeycloakConfiguration(ConfigurationAdmin configurationAdmin) throws IOException {
		Objects.requireNonNull(configurationAdmin, "OSGi configuration admin unavailable!");

		return extractKeycloakConfiguration(
				configurationAdmin.getConfiguration(PlayerSecurityConstants.KEYCLOAK_CONFIGURATION_PID));
	}

	private KeycloakConfiguration extractKeycloakConfiguration(Configuration config) {
		Objects.requireNonNull(config, "Keycloak configuration object is required!");
		Objects.requireNonNull(config.getProperties(), "Keycloak configuration properties are required!");
		Objects.requireNonNull(config.getProperties().get("oid.baseUri"), "'oid.baseUri' is required!");
		Objects.requireNonNull(config.getProperties().get("oid.realm"), "'oid.realm' is required!");
		Objects.requireNonNull(config.getProperties().get("oid.clientId"), "'oid.clientId' is required!");

		return new KeycloakConfiguration((String) config.getProperties().get("oid.baseUri"),
				(String) config.getProperties().get("oid.realm"), (String) config.getProperties().get("oid.clientId"));
	}

}
