/**
 * Copyright (c) 2012 - 2022 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package com.playertour.backend.statistics.service.impl;

import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

import org.gecko.emf.repository.EMFRepository;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.annotations.ReferenceScope;
import org.osgi.service.component.annotations.ServiceScope;

import com.playertour.backend.api.HoleScore;
import com.playertour.backend.api.LandingZone;
import com.playertour.backend.api.PlayerProfile;
import com.playertour.backend.api.PlayerScorecard;
import com.playertour.backend.apis.player.PlayerScorecardService;
import com.playertour.backend.apis.player.PlayerService;
import com.playertour.backend.player.model.player.Player;
import com.playertour.backend.statistics.model.statistics.StatisticsFactory;
import com.playertour.backend.statistics.model.statistics.StatisticsFairwaysResult;
import com.playertour.backend.statistics.model.statistics.StatisticsGreenHitsResult;
import com.playertour.backend.statistics.model.statistics.StatisticsLast5ScoresResult;
import com.playertour.backend.statistics.model.statistics.StatisticsLastScoreResult;
import com.playertour.backend.statistics.model.statistics.StatisticsResults;
import com.playertour.backend.statistics.service.api.StatisticsService;

@Component(name = "StatisticsService", scope = ServiceScope.PROTOTYPE)
public class StatisticsServiceImpl implements StatisticsService {

	@Reference(target = "(repo_id=playertour.playertour)", scope = ReferenceScope.PROTOTYPE_REQUIRED)
	private EMFRepository repository;

	@Reference(scope = ReferenceScope.PROTOTYPE_REQUIRED)
	private PlayerService playerService;

	@Reference(scope = ReferenceScope.PROTOTYPE_REQUIRED)
	private PlayerScorecardService playerScorecardService;

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.statistics.service.api.StatisticsService#getResults(java.lang.String)
	 */
	@Override
	public StatisticsResults getResults(String loginId) {
		Objects.requireNonNull(loginId, "Cannot retrieve statistics results for Player with null loginId!");

		Player player = playerService.getPlayerByLoginId(loginId);
		List<PlayerScorecard> completedPlayerScorecards = getCompletedScorecards(player);

		if (!completedPlayerScorecards.isEmpty()) {
			StatisticsLast5ScoresResult last5ScoresResult = getLast5ScoresResult(player.getProfile(),
					completedPlayerScorecards);
			StatisticsFairwaysResult fairwaysResult = getFairwaysResult(completedPlayerScorecards);
			StatisticsGreenHitsResult greenHitsResult = getGreenHitsResult(completedPlayerScorecards);

			return createStatisticsResults(last5ScoresResult, fairwaysResult, greenHitsResult);

		} else {
			return createEmptyStatisticsResults();
		}
	}

	private StatisticsLast5ScoresResult getLast5ScoresResult(PlayerProfile playerProfile,
			List<PlayerScorecard> completedPlayerScorecards) {
		StatisticsLast5ScoresResult last5ScoresResult = StatisticsFactory.eINSTANCE.createStatisticsLast5ScoresResult();

		List<PlayerScorecard> latestCompletedPlayerScorecards = playerScorecardService
				.getNLatestScorecards(completedPlayerScorecards, 5);

		for (int i = 0; i < latestCompletedPlayerScorecards.size(); i++) {
			StatisticsLastScoreResult lastScoreResult = getLastScoreResult(playerProfile.getPlayerTourAbility(),
					latestCompletedPlayerScorecards.get(i), (i + 1));
			last5ScoresResult.getLast5Scores().add(lastScoreResult);
		}

		return last5ScoresResult;
	}

	private StatisticsLastScoreResult getLastScoreResult(double pta, PlayerScorecard latestCompletedPlayerScorecard,
			int order) {
		StatisticsLastScoreResult lastScoreResult = StatisticsFactory.eINSTANCE.createStatisticsLastScoreResult();

		int gross = calculateTotalStrokes(latestCompletedPlayerScorecard);
		int net = calculateNetStrokes(gross, pta);

		lastScoreResult.setOrder(order);
		lastScoreResult.setGross(gross);
		lastScoreResult.setNet(net);

		return lastScoreResult;
	}

	private StatisticsFairwaysResult getFairwaysResult(List<PlayerScorecard> completedPlayerScorecards) {
		StatisticsFairwaysResult fairwaysResult = StatisticsFactory.eINSTANCE.createStatisticsFairwaysResult();

		List<PlayerScorecard> latestCompletedPlayerScorecards = playerScorecardService
				.getNLatestScorecards(completedPlayerScorecards, 10);

		double lastRoundResult = getFairwaysLastRoundResult(latestCompletedPlayerScorecards);

		double overallResult = getFairwaysOverallResult(latestCompletedPlayerScorecards);

		fairwaysResult.setLastRound(lastRoundResult);
		fairwaysResult.setOverall(overallResult);

		return fairwaysResult;
	}

	private double getFairwaysLastRoundResult(List<PlayerScorecard> completedPlayerScorecards) {
		PlayerScorecard latestCompletedPlayerScorecard = getLatestScorecard(completedPlayerScorecards);

		if (latestCompletedPlayerScorecard != null) {
			return calculateFairwaysResult(latestCompletedPlayerScorecard);
		} else {
			return 0;
		}
	}

	private double getFairwaysOverallResult(List<PlayerScorecard> completedPlayerScorecards) {
		// @formatter:off
		return completedPlayerScorecards.stream()
				.mapToDouble(s -> calculateFairwaysResult(s))
				.average()
				.getAsDouble();
		// @formatter:on
	}

	private StatisticsGreenHitsResult getGreenHitsResult(List<PlayerScorecard> completedPlayerScorecards) {
		StatisticsGreenHitsResult greenHitsResult = StatisticsFactory.eINSTANCE.createStatisticsGreenHitsResult();

		List<PlayerScorecard> latestCompletedPlayerScorecards = playerScorecardService
				.getNLatestScorecards(completedPlayerScorecards, 10);

		double par3sResult = calculateGreenHitsPar3sResult(latestCompletedPlayerScorecards);
		double par4sResult = calculateGreenHitsPar4sResult(latestCompletedPlayerScorecards);
		double par5sResult = calculateGreenHitsPar5sResult(latestCompletedPlayerScorecards);

		greenHitsResult.setPar3s(par3sResult);
		greenHitsResult.setPar4s(par4sResult);
		greenHitsResult.setPar5s(par5sResult);

		return greenHitsResult;
	}

	private List<PlayerScorecard> getCompletedScorecards(Player player) {
		// @formatter:off
		return player.getPlayerScorecards().stream()
				.filter(s -> (s.getEnd() != null && s.isNoReturn() == false))
				.collect(Collectors.toList());
		// @formatter:on
	}

	private PlayerScorecard getLatestScorecard(List<PlayerScorecard> completedPlayerScorecards) {
		List<PlayerScorecard> latestcompletedPlayerScorecards = playerScorecardService
				.getNLatestScorecards(completedPlayerScorecards, 1);
		if (!latestcompletedPlayerScorecards.isEmpty()) {
			return latestcompletedPlayerScorecards.get(0);
		} else {
			return null;
		}
	}

	private int calculateTotalStrokes(PlayerScorecard playerScorecard) {
		// @formatter:off
		return playerScorecard.getHoleStats().stream()
				.collect(Collectors.summingInt(HoleScore::getScore))
				.intValue();
		// @formatter:on
	}

	private int calculateNetStrokes(int gross, double pta) {
		return (gross - Double.valueOf(pta).intValue());
	}

	private int calculateFairwayZoneFirstStrokes(PlayerScorecard completedPlayerScorecard) {
		// @formatter:off
		return completedPlayerScorecard.getHoleStats().stream()
				.filter(hs -> !hs.getStrokes().isEmpty() 
						&& hs.getStrokes().get(0).getZone() == LandingZone.FAIRWAY)
				.collect(Collectors.counting())
				.intValue();
		// @formatter:on
	}

	private double calculateFairwaysResult(PlayerScorecard completedPlayerScorecard) {
		int fairwayZoneFirstStrokes = calculateFairwayZoneFirstStrokes(completedPlayerScorecard);

		int holesTotal = completedPlayerScorecard.getHoleStats().size();

		return ((double) fairwayZoneFirstStrokes / (double) holesTotal);
	}

	private double calculateGreenHitsPar3sResult(List<PlayerScorecard> completedPlayerScorecards) {
		// @formatter:off
		return completedPlayerScorecards.stream()
				.mapToDouble(s -> calculateGreenHitsResult(s, 1, 3))
				.average()
				.getAsDouble();
		// @formatter:on
	}

	private double calculateGreenHitsPar4sResult(List<PlayerScorecard> completedPlayerScorecards) {
		// @formatter:off
		return completedPlayerScorecards.stream()
				.mapToDouble(s -> calculateGreenHitsResult(s, 2, 4))
				.average()
				.getAsDouble();
		// @formatter:on
	}

	private double calculateGreenHitsPar5sResult(List<PlayerScorecard> completedPlayerScorecards) {
		// @formatter:off
		return completedPlayerScorecards.stream()
				.mapToDouble(s -> calculateGreenHitsResult(s, 3, 5))
				.average()
				.getAsDouble();
		// @formatter:on
	}

	private double calculateGreenHitsResult(PlayerScorecard completedPlayerScorecard, int strokeNumber, int par) {
		int greenZoneStrokes = calculateGreenZoneStrokes(completedPlayerScorecard, strokeNumber, par);

		// take into account earlier strokes as well, i.e. for par 4, stroke number 1, in addition to stroke number 2
		if ((par == 4) && (strokeNumber == 2)) {
			greenZoneStrokes += calculateGreenZoneStrokes(completedPlayerScorecard, (strokeNumber - 1), par);
		}

		// for par 5, stroke number 1 and 2, in addition to stroke number 3
		if ((par == 5) && (strokeNumber == 3)) {
			greenZoneStrokes += calculateGreenZoneStrokes(completedPlayerScorecard, (strokeNumber - 2), par);
			greenZoneStrokes += calculateGreenZoneStrokes(completedPlayerScorecard, (strokeNumber - 1), par);
		}

		int holesTotal = completedPlayerScorecard.getHoleStats().size();

		return ((double) greenZoneStrokes / (double) holesTotal);
	}

	private int calculateGreenZoneStrokes(PlayerScorecard completedPlayerScorecard, int strokeNumber, int par) {
		// @formatter:off
		return completedPlayerScorecard.getHoleStats().stream()
				.filter(hs -> !hs.getStrokes().isEmpty() 
						&& hs.getStrokes().size() >= strokeNumber
						&& hs.getHoleInfo().getPar().intValue() == par
						&& hs.getStrokes().get(strokeNumberToIndex(strokeNumber)).getZone() == LandingZone.GREEN)
				.collect(Collectors.counting())
				.intValue();
		// @formatter:on
	}

	private int strokeNumberToIndex(int strokeNumber) {
		return (strokeNumber - 1);
	}

	private StatisticsResults createStatisticsResults(StatisticsLast5ScoresResult last5ScoresResult,
			StatisticsFairwaysResult fairwaysResult, StatisticsGreenHitsResult greenHitsResult) {
		StatisticsResults results = StatisticsFactory.eINSTANCE.createStatisticsResults();
		results.setLast5ScoresResult(last5ScoresResult);
		results.setFairwaysResult(fairwaysResult);
		results.setGreenHitsResult(greenHitsResult);

		return results;
	}

	private StatisticsResults createEmptyStatisticsResults() {
		StatisticsResults results = StatisticsFactory.eINSTANCE.createStatisticsResults();

		results.setLast5ScoresResult(StatisticsFactory.eINSTANCE.createStatisticsLast5ScoresResult());
		results.setFairwaysResult(StatisticsFactory.eINSTANCE.createStatisticsFairwaysResult());
		results.setGreenHitsResult(StatisticsFactory.eINSTANCE.createStatisticsGreenHitsResult());

		return results;
	}
}
