/**
 * Copyright (c) 2012 - 2022 Data In Motion and others.
 * All rights reserved.
 *
 * This program and the accompanying materials are made available under the terms of the
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package com.playertour.backend.statistics.service.tests;

import static org.assertj.core.api.Assertions.assertThat;

import java.util.List;
import java.util.UUID;

import org.assertj.core.data.Offset;
import org.gecko.mongo.osgi.MongoDatabaseProvider;
import org.gecko.search.util.CommonIndexService;
import org.junit.jupiter.api.AfterAll;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.MethodOrderer;
import org.junit.jupiter.api.Order;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.TestMethodOrder;
import org.junit.jupiter.api.extension.ExtendWith;
import org.junit.platform.commons.annotation.Testable;
import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceReference;
import org.osgi.test.common.annotation.InjectBundleContext;
import org.osgi.test.common.annotation.InjectService;
import org.osgi.test.common.service.ServiceAware;
import org.osgi.test.junit5.context.BundleContextExtension;
import org.osgi.test.junit5.service.ServiceExtension;

import com.mongodb.client.MongoDatabase;
import com.playertour.backend.api.HoleScore;
import com.playertour.backend.api.LandingZone;
import com.playertour.backend.api.PlayerApiFactory;
import com.playertour.backend.api.PlayerApiPackage;
import com.playertour.backend.api.PlayerProfile;
import com.playertour.backend.api.PlayerScorecard;
import com.playertour.backend.api.Stroke;
import com.playertour.backend.apis.course.CourseService;
import com.playertour.backend.apis.player.PlayerScorecardService;
import com.playertour.backend.apis.player.PlayerService;
import com.playertour.backend.apis.player.exceptions.PlayerUsernameValidationException;
import com.playertour.backend.golfcourse.model.golfcourse.CourseScorecardInfo;
import com.playertour.backend.golfcourse.model.golfcourse.CourseScorecards;
import com.playertour.backend.golfcourse.model.golfcourse.GenderType;
import com.playertour.backend.golfcourse.model.golfcourse.GolfCourseFactory;
import com.playertour.backend.golfcourse.model.golfcourse.Tee;
import com.playertour.backend.player.model.player.PlayerPackage;
import com.playertour.backend.statistics.model.statistics.StatisticsFairwaysResult;
import com.playertour.backend.statistics.model.statistics.StatisticsGreenHitsResult;
import com.playertour.backend.statistics.model.statistics.StatisticsLast5ScoresResult;
import com.playertour.backend.statistics.model.statistics.StatisticsLastScoreResult;
import com.playertour.backend.statistics.model.statistics.StatisticsResults;
import com.playertour.backend.statistics.service.api.StatisticsService;

@Testable
@ExtendWith(BundleContextExtension.class)
@ExtendWith(ServiceExtension.class)
@TestMethodOrder(MethodOrderer.OrderAnnotation.class)
public class StatisticsServiceTest {
	
	@InjectBundleContext
	BundleContext bundleContext;
	
	@Order(value = -1)
	@Test
	public void testServices(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<StatisticsService> statisticsServiceAware, 
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerServiceAware, 
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerScorecardService> playerScorecardServiceAware, 
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<CourseService> courseServiceAware) {

		assertThat(statisticsServiceAware.getServices()).hasSize(1);
		ServiceReference<StatisticsService> statisticsServiceReference = statisticsServiceAware
				.getServiceReference();
		assertThat(statisticsServiceReference).isNotNull();
		
		assertThat(playerServiceAware.getServices()).hasSize(1);
		ServiceReference<PlayerService> playerServiceRef = playerServiceAware.getServiceReference();
		assertThat(playerServiceRef).isNotNull();
		
		assertThat(playerScorecardServiceAware.getServices()).hasSize(1);	
		ServiceReference<PlayerScorecardService> playerScorecardServiceRef = playerScorecardServiceAware.getServiceReference();
		assertThat(playerScorecardServiceRef).isNotNull();
		
		assertThat(courseServiceAware.getServices()).hasSize(1);	
		ServiceReference<CourseService> courseServiceRef = courseServiceAware.getServiceReference();
		assertThat(courseServiceRef).isNotNull();
	}
	
	@Test
	public void testGetResults(@InjectService(cardinality = 1, timeout = 5000) ServiceAware<StatisticsService> statisticsServiceAware, 
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerServiceAware,  
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerScorecardService> playerScorecardServiceAware, 
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<CourseService> courseServiceAware) throws InterruptedException, PlayerUsernameValidationException {
		
		assertThat(statisticsServiceAware.getServices()).hasSize(1);	
		StatisticsService statisticsService = statisticsServiceAware.getService();
		assertThat(statisticsService).isNotNull();	
		
		assertThat(playerServiceAware.getServices()).hasSize(1);	
		PlayerService playerService = playerServiceAware.getService();
		assertThat(playerService).isNotNull();
		
		assertThat(playerScorecardServiceAware.getServices()).hasSize(1);	
		PlayerScorecardService playerScorecardService = playerScorecardServiceAware.getService();
		assertThat(playerScorecardService).isNotNull();	
		
		assertThat(courseServiceAware.getServices()).hasSize(1);	
		CourseService courseService = courseServiceAware.getService();
		assertThat(courseService).isNotNull();
		
		String loginId = UUID.randomUUID().toString();
		String loginName = "engelbert.humperdinck";
		
		PlayerProfile profileInitial = playerService.getPlayerProfile(loginId, loginName);			
		assertThat(profileInitial).isNotNull();
		
		CourseScorecardInfo scorecardInfo = GolfCourseFactory.eINSTANCE.createCourseScorecardInfo();
		scorecardInfo.setParIn(37);
		scorecardInfo.setParOut(35);
		scorecardInfo.setParTotal(72);
		scorecardInfo.getHcpHole().addAll(List.of(9, 11, 15, 1, 3, 13, 17, 5, 7, 6, 14, 18, 4, 8, 16, 2, 10, 12));
		scorecardInfo.getParHole().addAll(List.of(4, 4, 3, 5, 4, 4, 3, 4, 4, 5, 5, 4, 4, 4, 3, 5, 3, 4));
		
		CourseScorecards courseScorecards = GolfCourseFactory.eINSTANCE.createCourseScorecards();
		courseScorecards.getMenScorecard().add(scorecardInfo);
	//	courseScorecards.getWmnScorecard().add(scorecardInfo); // FIXME: adding 'CourseScorecardInfo' to 'wmnScorecard' list somehow erases 'CourseScorecardInfo' from 'menScorecard' list
				
		com.playertour.backend.golfcourse.model.golfcourse.GolfCourse golfCourse = GolfCourseFactory.eINSTANCE.createGolfCourse();
		golfCourse.setScorecards(courseScorecards);
		golfCourse.setCourseId(UUID.randomUUID().toString());
		
		int holesTotal = 18;
		GenderType gender = GenderType.MEN;
		
		Tee tee = GolfCourseFactory.eINSTANCE.createTee();
		tee.setColor("Yellow");
		tee.setGender(gender);
		golfCourse.getTee().add(tee);
		
		golfCourse = courseService.saveCourse(golfCourse);
		assertThat(golfCourse).isNotNull();
		
		String courseId = golfCourse.getId();
		assertThat(courseId).isNotNull();
		
		Thread.sleep(2000);
		
		PlayerScorecard openedPlayerScorecard = playerScorecardService.openScorecard(loginId, courseId, gender);
		assertThat(openedPlayerScorecard).isNotNull();
		assertThat(openedPlayerScorecard.getStart()).isNotNull();
		assertThat(openedPlayerScorecard.getEnd()).isNull();
		assertThat(openedPlayerScorecard.getCourseScorecardInfo()).isNotNull();
		assertThat(openedPlayerScorecard.getCourseId()).isEqualTo(courseId);
		assertThat(openedPlayerScorecard.getHoleStats()).isNotNull();
		assertThat(openedPlayerScorecard.getHoleStats()).isNotEmpty();
		assertThat(openedPlayerScorecard.getHoleStats()).hasSize(holesTotal);
		
		String playerScorecardId = openedPlayerScorecard.getId();
		assertThat(playerScorecardId).isNotNull();
				
		Thread.sleep(2000);
		
		PlayerScorecard updatedPlayerScorecard = generateStrokes(loginId, playerScorecardId, openedPlayerScorecard, playerScorecardService);
		assertThat(updatedPlayerScorecard).isNotNull();
		
		Thread.sleep(2000);
		
		PlayerScorecard closedPlayerScorecard = playerScorecardService.closeScorecard(loginId, playerScorecardId);
		assertThat(closedPlayerScorecard).isNotNull();
		assertThat(closedPlayerScorecard.getEnd()).isNotNull();
		assertThat(closedPlayerScorecard.isNoReturn()).isFalse();
		
		Thread.sleep(2000);
		
		PlayerProfile profileUpdated = playerScorecardService.updatePTA(loginId);
		assertThat(profileUpdated).isNotNull();
		assertThat(profileUpdated.getPlayerTourAbility()).isEqualTo(-52.0);
		
		Thread.sleep(2000);
		
		StatisticsResults results = statisticsService.getResults(loginId);
		assertThat(results).isNotNull();
		
		StatisticsLast5ScoresResult last5ScoresResult = results.getLast5ScoresResult();
		assertThat(last5ScoresResult).isNotNull();
		assertThat(last5ScoresResult.getLast5Scores()).hasSize(1);
		
		StatisticsLastScoreResult lastScoresResult = last5ScoresResult.getLast5Scores().get(0);
		assertThat(lastScoresResult.getGross()).isEqualTo(20);
		assertThat(lastScoresResult.getNet()).isEqualTo(72);
		
		StatisticsFairwaysResult fairwaysResult = results.getFairwaysResult();
		assertThat(fairwaysResult).isNotNull();
		assertThat(fairwaysResult.getLastRound()).isEqualTo(0.4444444444444444);
		
		StatisticsGreenHitsResult greenHitsResult = results.getGreenHitsResult();
		assertThat(greenHitsResult).isNotNull();
		
		assertThat(greenHitsResult.getPar3s()).isCloseTo(0.222222222, Offset.offset(0.000000002));
		assertThat(greenHitsResult.getPar4s()).isCloseTo(0.222222222, Offset.offset(0.000000002));
		assertThat(greenHitsResult.getPar5s()).isCloseTo(0.055555556, Offset.offset(0.000000002));
	}
	
	private PlayerScorecard generateStrokes(String loginId, String playerScorecardId, PlayerScorecard playerScorecard, PlayerScorecardService playerScorecardService) throws InterruptedException {
		int holesTotal = playerScorecard.getHoleStats().size();
		
		for (int i = 0; i < holesTotal; i++) {
			int holeNumber = (i + 1);
			
			boolean hasExtraFirstStroke = false;
			
			if (holeNumber % 9 == 0) {
				Stroke stroke = generateStroke(holeNumber, LandingZone.ROUGH);
				
				playerScorecard = playerScorecardService.submitStroke(loginId, playerScorecardId, stroke);
				assertThat(playerScorecard).isNotNull();
				assertThat(playerScorecard.getHoleStats()).isNotEmpty();
				
				hasExtraFirstStroke = true;
				
				Thread.sleep(250);
			}
			
			LandingZone landingZone;
			if (holeNumber % 2 == 0) {
				landingZone = LandingZone.FAIRWAY;
			} else {
				landingZone = LandingZone.GREEN;
			}
			
			Stroke stroke = generateStroke(holeNumber, landingZone);
			
			playerScorecard = playerScorecardService.submitStroke(loginId, playerScorecardId, stroke);
			assertThat(playerScorecard).isNotNull();
			assertThat(playerScorecard.getHoleStats()).isNotEmpty();
			
			HoleScore holeScore = playerScorecard.getHoleStats().get(i);
			assertThat(holeScore.getStrokes()).isNotEmpty();
			assertThat(holeScore.getStrokes().get(0).getHoleNumber()).isEqualTo(holeNumber);
			if (hasExtraFirstStroke) {				
				assertThat(holeScore.getStrokes()).hasSize(2);
				assertThat(holeScore.getStrokes().get(0).getZone()).isEqualTo(LandingZone.ROUGH);
				assertThat(holeScore.getStrokes().get(1).getZone()).isEqualTo(landingZone);
				
				assertThat(holeScore.getScore()).isEqualTo(2);
				assertThat(holeScore.getPTScore()).isEqualTo(2);				
			} else {
				assertThat(holeScore.getStrokes()).hasSize(1);
				assertThat(holeScore.getStrokes().get(0).getZone()).isEqualTo(landingZone);
				
				assertThat(holeScore.getScore()).isEqualTo(1);
				assertThat(holeScore.getPTScore()).isEqualTo(1);				
			}
			
			Thread.sleep(250);
		}
		
		return playerScorecard;
	}
	
	private Stroke generateStroke(int holeNumber, LandingZone landingZone) {
		Stroke stroke = PlayerApiFactory.eINSTANCE.createStroke();
		stroke.setHoleNumber(holeNumber);
		stroke.setZone(landingZone);
		return stroke;
	}
	
	@BeforeEach
	@AfterEach
	public void clean(@InjectService(cardinality = 1, timeout = 5000, filter = "(component.name=PlayerIndexService)") ServiceAware<CommonIndexService> indexAware,
			@InjectService(cardinality = 1, timeout = 5000, filter = "(database=playertour)") ServiceAware<MongoDatabaseProvider> dbProviderAware) {
		
		assertThat(indexAware.getServices()).hasSize(1);	
		CommonIndexService indexService = indexAware.getService();
		assertThat(indexService).isNotNull();		
		
		indexService.resetIndex();
		
		assertThat(dbProviderAware.getServices()).hasSize(1);	
		MongoDatabaseProvider dbProvider = dbProviderAware.getService();
		assertThat(dbProvider).isNotNull();	
		
		MongoDatabase database = dbProvider.getDatabase();
		try {
			database.getCollection(PlayerPackage.Literals.PLAYER.getName()).drop();
			database.getCollection(PlayerApiPackage.Literals.GOLF_COURSE.getName()).drop();
		} catch (IllegalArgumentException e) {
			System.out.println("Collection does not exist. Nothing to remove.");
		}
	}

	@AfterAll
	public static void brutalClean(
			@InjectService(cardinality = 1, timeout = 5000, filter = "(database=playertour)") ServiceAware<MongoDatabaseProvider> dbProviderAware) {
		assertThat(dbProviderAware.getServices()).hasSize(1);
		MongoDatabaseProvider dbProvider = dbProviderAware.getService();
		assertThat(dbProvider).isNotNull();

		MongoDatabase database = dbProvider.getDatabase();
		database.drop();
	}
}
