/**
 * Copyright (c) 2012 - 2018 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package com.playertour.backend.test.setup;

import java.util.List;
import java.util.Map;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;

import org.gecko.emf.repository.EMFRepository;
import org.gecko.search.api.IndexActionType;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.annotations.ReferenceScope;
import org.osgi.util.promise.PromiseFactory;

import com.playertour.backend.apis.course.CourseIndexService;
import com.playertour.backend.apis.course.CourseSearchService;
import com.playertour.backend.golfcourse.model.golfcourse.Address;
import com.playertour.backend.golfcourse.model.golfcourse.Contact;
import com.playertour.backend.golfcourse.model.golfcourse.CourseDetails;
import com.playertour.backend.golfcourse.model.golfcourse.CourseGPSVector;
import com.playertour.backend.golfcourse.model.golfcourse.CourseScorecard;
import com.playertour.backend.golfcourse.model.golfcourse.DressCodeType;
import com.playertour.backend.golfcourse.model.golfcourse.GolfCourse;
import com.playertour.backend.golfcourse.model.golfcourse.GolfCourseFactory;
import com.playertour.backend.golfcourse.model.golfcourse.GolfCourseType;
import com.playertour.backend.golfcourse.model.golfcourse.Hole;
import com.playertour.backend.golfcourse.model.golfcourse.Location;
import com.playertour.backend.golfcourse.model.golfcourse.PriceTable;
import com.playertour.backend.golfcourse.model.golfcourse.Scorecard;
import com.playertour.backend.golfcourse.model.golfcourse.ShapeObject;
import com.playertour.backend.golfcourse.model.golfcourse.WeekDay;

@Component(immediate=true, service = TestSetupComponent.class)
public class TestSetupComponent {
	
	@Reference(target = "(repo_id=playertour.playertour)", scope = ReferenceScope.PROTOTYPE_REQUIRED)
	private EMFRepository repository;
	
	@Reference
	CourseSearchService courseSearch;
	
	@Reference
	CourseIndexService courseIndex;
	
	private PromiseFactory factory = new PromiseFactory(Executors.newFixedThreadPool(4));
	
	private static final String[] TEST_COURSES_NAMES = new String[] {"TEST COURSE 1", "TEST COURSE 2"};

	@Activate
	public void avtivate() {
		factory.submit(() -> {
			CountDownLatch latch = new CountDownLatch(1);
			latch.await(10, TimeUnit.SECONDS);
			setupTestData();
			return true;
		})
		.onSuccess(s -> System.out.println("Test Data Saved!!"))
		.onFailure(t -> t.printStackTrace());	
	}

	
	private void setupTestData() {
		
		for(String name : TEST_COURSES_NAMES) {
			List<GolfCourse> courses = courseSearch.getCoursesByName(name, 1);
			if(courses == null || courses.isEmpty()) {
				GolfCourse course = createCourse(name);
				repository.save(course);
				courseIndex.indexGolfCourse(course, IndexActionType.ADD);
			}	
			else {
				System.out.println("We already have " + name);	
			}
		}		
	}


	private GolfCourse createCourse(String name) {
		
		GolfCourse course = GolfCourseFactory.eINSTANCE.createGolfCourse();
		course.setCourseId(name);
		createCourseDetails(course, name);
		createCourseGPSVector(course, name);
		createCourseScorecard(course, name);
		return course;
	}




	private void createCourseScorecard(GolfCourse course, String courseName) {
		
		CourseScorecard courseScorecard = GolfCourseFactory.eINSTANCE.createCourseScorecard();
		courseScorecard.setCourseId(course.getCourseId());
		Scorecard sc = GolfCourseFactory.eINSTANCE.createScorecard();
		
		if(TEST_COURSES_NAMES[0].equals(courseName)) {
			sc.setParIn(2);
			sc.setParOut(4);
			sc.setParTotal(7);
		} else {
			sc.setParIn(3);
			sc.setParOut(5);
			sc.setParTotal(8);
		}
		courseScorecard.getMenScorecard().add(sc);
		course.setScorecard(courseScorecard);
 	}


	private void createCourseGPSVector(GolfCourse course, String courseName) {
		
		CourseGPSVector gpsVector = GolfCourseFactory.eINSTANCE.createCourseGPSVector();
		gpsVector.setHoleCount(18);
		Map<String, ShapeObject> shapeObjs;
		if(TEST_COURSES_NAMES[0].equals(courseName)) {
			shapeObjs = XMLReader.extractShapeObjects("data/TestCourse1.kml");
		} else {
			shapeObjs = XMLReader.extractShapeObjects("data/TestCourse2.kml");
		}
		assignShapeObjects(shapeObjs, gpsVector, courseName);
		course.setCourseGPSVector(gpsVector);
	}


	private void assignShapeObjects(Map<String, ShapeObject> shapeObjs, CourseGPSVector gpsVector, String courseName) {
		
		ShapeObject background = shapeObjs.entrySet().stream()
				.filter(e->"Background".equals(e.getKey())).map(e->e.getValue()).findFirst().orElse(null);
		gpsVector.setBackgrund(background);
		
		ShapeObject clubHouse = shapeObjs.entrySet().stream()
				.filter(e->"Club House".equals(e.getKey())).map(e->e.getValue()).findFirst().orElse(null);
		gpsVector.setBackgrund(clubHouse);
		
		int maxHoleNum = courseName.equals(TEST_COURSES_NAMES[0]) ? 1 : 2;
		for(int i = 1; i <= maxHoleNum; i++) {
			int count = i;
			Hole hole = GolfCourseFactory.eINSTANCE.createHole();
			hole.setHoleNumber(i);
			
			ShapeObject green = shapeObjs.entrySet().stream()
					.filter(e->String.valueOf("Green "+ String.valueOf(count)).equals(e.getKey())).map(e->e.getValue()).findFirst().orElse(null);
			hole.setGreen(green);
			
			ShapeObject greenCenter = shapeObjs.entrySet().stream()
					.filter(e->String.valueOf("Green Center "+ String.valueOf(count)).equals(e.getKey())).map(e->e.getValue()).findFirst().orElse(null);
			hole.setGreenCenter(greenCenter);
			
			ShapeObject fairway = shapeObjs.entrySet().stream()
					.filter(e->String.valueOf("Fairway "+ String.valueOf(count)).equals(e.getKey())).map(e->e.getValue()).findFirst().orElse(null);
			hole.setFairway(fairway);
			
			ShapeObject centralPath = shapeObjs.entrySet().stream()
					.filter(e->String.valueOf("Central Path "+ String.valueOf(count)).equals(e.getKey())).map(e->e.getValue()).findFirst().orElse(null);
			hole.setCentralPath(centralPath);
			
			ShapeObject teeBox = shapeObjs.entrySet().stream()
					.filter(e->String.valueOf("Tee Box "+ String.valueOf(count)).equals(e.getKey())).map(e->e.getValue()).findFirst().orElse(null);
			hole.setTeebox(teeBox);
			
			ShapeObject teeBoxCenter = shapeObjs.entrySet().stream()
					.filter(e->String.valueOf("Tee Box Center "+ String.valueOf(count)).equals(e.getKey())).map(e->e.getValue()).findFirst().orElse(null);
			hole.setTeeboxCenter(teeBoxCenter);
			
			gpsVector.getHoles().add(hole);			
		}	
	}


	private void createCourseDetails(GolfCourse course, String courseName) {

		CourseDetails details = GolfCourseFactory.eINSTANCE.createCourseDetails();
		details.setCountryFullName("Germany");
		details.setCountryShortName("DE");
		details.setCourseName(courseName);
		details.setCountryId("276");
		details.setStateFullName("Thuringia");
		details.setStateShortName("TH");
		details.setStateId("7");
		
		if(TEST_COURSES_NAMES[0].equals(courseName)) {
			details.setDayClosed(WeekDay.MON);
			details.setSeasonal(true);
			details.setType(GolfCourseType.PUBLIC);
			details.setDressCode(DressCodeType.DENIM_OK);
			
			Location loc = GolfCourseFactory.eINSTANCE.createLocation();
			loc.setLatitude(50.92266955496815);
			loc.setLongitude(11.580667902049983);
			details.setLocation(loc);
			
			Address address = GolfCourseFactory.eINSTANCE.createAddress();
			address.setCity("Jena");
			address.setCountry("Germany");
			address.setRoute("Kahlaische Str.");
			address.setZipCode("07745");
			address.setState("Thuringia");
			details.setAddress(address);
			
			Contact contact = GolfCourseFactory.eINSTANCE.createContact();
			contact.setEmail("dimgolfclub@gmail.com");
			contact.setPhone("0384257485");
			contact.setUrl("www.dimgolfclub.de");
			details.setContact(contact);
			
			PriceTable price = GolfCourseFactory.eINSTANCE.createPriceTable();
			price.setTwilight(12.0);
			price.setWeekday18(15.50);
			price.setWeekday9(7.50);
			price.setWeekend18(20.50);
			price.setWeekend9(10.50);
			details.setPrincing(price);
			
		} else {
			details.setDayClosed(WeekDay.WED);
			details.setSeasonal(false);
			details.setType(GolfCourseType.SEMI_PRIVATE);
			details.setDressCode(DressCodeType.NO_DENIM);
			
			Location loc = GolfCourseFactory.eINSTANCE.createLocation();
			loc.setLatitude(50.8719859508072);
			loc.setLongitude(12.097641882256138);
			details.setLocation(loc);
			
			Address address = GolfCourseFactory.eINSTANCE.createAddress();
			address.setCity("Gera");
			address.setCountry("Germany");
			address.setRoute("Am Ferberturm");
			address.setZipCode("07546");
			address.setState("Thuringia");
			details.setAddress(address);
			
			Contact contact = GolfCourseFactory.eINSTANCE.createContact();
			contact.setEmail("geragolfclub@gmail.com");
			contact.setPhone("3748265284");
			contact.setUrl("www.geragolfclub.de");
			details.setContact(contact);
			
			PriceTable price = GolfCourseFactory.eINSTANCE.createPriceTable();
			price.setTwilight(11.50);
			price.setWeekday18(13.50);
			price.setWeekday9(7.0);
			price.setWeekend18(21.50);
			price.setWeekend9(10.0);
			details.setPrincing(price);
		}
		course.setCourseDetails(details);		
	}

}
