/**
 * Copyright (c) 2012 - 2021 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package com.playertour.backend.vaadin.security;

import static com.playertour.backend.vaadin.security.KeycloakSecuredVaadinConstants.keycloakServerJsAdapterInit;
import static com.playertour.backend.vaadin.security.KeycloakSecuredVaadinConstants.keycloakServerJsAdapterPath;

import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Objects;

import org.osgi.service.cm.ConfigurationAdmin;

import com.playertour.backend.security.KeycloakConfiguration;
import com.playertour.backend.security.PlayerSecurityUtil;
import com.vaadin.flow.component.UI;

/**
 * 
 * @author mhs
 * @since Apr 28, 2021
 */
public enum KeycloakSecuredVaadinHelper {
	INSTANCE;

	public void enableKeycloakAuthentication(ConfigurationAdmin configurationAdmin, com.vaadin.flow.component.UI ui,
			com.vaadin.flow.component.Component component) throws IOException {
		KeycloakConfiguration keycloakConfiguration = PlayerSecurityUtil.INSTANCE
				.getKeycloakConfiguration(configurationAdmin);
		
		addKeycloakJSAdapter(ui, keycloakConfiguration);
		addKeycloakJSAdapterInit(ui);
		addKeycloakJSConfig(ui, keycloakConfiguration);
		addKeycloakJSInit(component);
	}

	private String buildKeycloakJSAdapterURL(KeycloakConfiguration keycloakConfig) throws MalformedURLException {
		Objects.requireNonNull(keycloakConfig, "Keycloak configuration is required!");

		URL keycloakAuthURL = new URL(keycloakConfig.getOidBaseUrl());

		URL keycloakJSAdapterURL = new URL(keycloakAuthURL.getProtocol(), keycloakAuthURL.getHost(),
				keycloakAuthURL.getPort(), keycloakServerJsAdapterPath);

		return keycloakJSAdapterURL.toExternalForm();
	}

	private String buildKeycloakJSConfig(KeycloakConfiguration keycloakConfig) {
		Objects.requireNonNull(keycloakConfig, "Keycloak configuration is required!");

		StringBuilder keycloakJSConfig = new StringBuilder();
		keycloakJSConfig.append("keycloak = new Keycloak({ ");
		keycloakJSConfig.append("   url: '" + keycloakConfig.getOidBaseUrl() + "',");
		keycloakJSConfig.append("   realm: '" + keycloakConfig.getOidRealm() + "',");
		keycloakJSConfig.append("   clientId: '" + keycloakConfig.getOidClientId() + "'");
		keycloakJSConfig.append("});");

		return keycloakJSConfig.toString();
	}

	private String buildKeycloakJSInit() {
		return "initKeycloak($0)";
	}

	private void addKeycloakJSAdapter(UI ui, KeycloakConfiguration keycloakConfig) throws MalformedURLException {
		String keycloakJSAdapterURL = buildKeycloakJSAdapterURL(keycloakConfig);
		ui.getPage().addJavaScript(keycloakJSAdapterURL);
	}

	private void addKeycloakJSAdapterInit(UI ui) {
		ui.getPage().addJavaScript(keycloakServerJsAdapterInit);
	}

	private void addKeycloakJSConfig(UI ui, KeycloakConfiguration keycloakConfig) {
		String keycloakJSConfig = buildKeycloakJSConfig(keycloakConfig);
		ui.getPage().executeJs(keycloakJSConfig);
	}

	private void addKeycloakJSInit(com.vaadin.flow.component.Component component) {
		String keycloakJSInit = buildKeycloakJSInit();
		component.getElement().executeJs(keycloakJSInit, component);
	}
}
