/**
 * Copyright (c) 2012 - 2022 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package com.playertour.backend.vaadin.views.edit;

import java.util.Collections;
import java.util.LinkedList;
import java.util.List;

import org.gecko.vaadin.whiteboard.annotations.VaadinComponent;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.annotations.ReferenceScope;
import org.osgi.service.component.annotations.ServiceScope;

import com.playertour.backend.api.Address;
import com.playertour.backend.api.GolfCourseSearchResult;
import com.playertour.backend.apis.course.CourseService;
import com.playertour.backend.apis.igolf.IGolfCourseSearchService;
import com.playertour.backend.apis.igolf.IGolfCourseUpdateService;
import com.playertour.backend.apis.mmt.common.UnknownTransformationException;
import com.playertour.backend.apis.player.APIGolfCourseService;
import com.playertour.backend.igolf.model.igolf.CourseCompleteResponse;
import com.playertour.backend.vaadin.views.main.MainView;
import com.vaadin.flow.component.button.Button;
import com.vaadin.flow.component.button.ButtonVariant;
import com.vaadin.flow.component.checkbox.Checkbox;
import com.vaadin.flow.component.grid.Grid;
import com.vaadin.flow.component.grid.Grid.Column;
import com.vaadin.flow.component.grid.editor.Editor;
import com.vaadin.flow.component.html.Label;
import com.vaadin.flow.component.icon.VaadinIcon;
import com.vaadin.flow.component.notification.Notification;
import com.vaadin.flow.component.orderedlayout.HorizontalLayout;
import com.vaadin.flow.component.orderedlayout.VerticalLayout;
import com.vaadin.flow.component.radiobutton.RadioButtonGroup;
import com.vaadin.flow.component.radiobutton.RadioGroupVariant;
import com.vaadin.flow.component.textfield.TextField;
import com.vaadin.flow.data.binder.Binder;
import com.vaadin.flow.data.renderer.ComponentRenderer;
import com.vaadin.flow.router.PageTitle;
import com.vaadin.flow.router.Route;

/**
 * 
 * @author ilenia
 * @since Aug 11, 2022
 */
@Route(value = "edit", layout = MainView.class)
@PageTitle("Edit")
@Component(name = "EditView", service = EditView.class, scope = ServiceScope.PROTOTYPE)
@VaadinComponent()
public class EditView extends VerticalLayout {

	/** serialVersionUID */
	private static final long serialVersionUID = 2642203026459479108L;

	@Reference(scope = ReferenceScope.PROTOTYPE_REQUIRED)
	APIGolfCourseService apiCourseService;
	
	@Reference(scope = ReferenceScope.PROTOTYPE_REQUIRED)
	CourseService courseService;
	
	@Reference(scope = ReferenceScope.PROTOTYPE_REQUIRED)
	IGolfCourseUpdateService iGolfUpdateService;

	@Reference
	IGolfCourseSearchService iGolfCourseSearchService;

	private static final String PLAYERTOUR_DB_OPTION = "playertour db";
	private static final String IGOLF_DB_OPTION = "igolf db";
	private static final String UNKNOWN_COURSE_NAME = "unknown";
	private static final String GRID_COLUMN_KEY_DELETED = "grid-deleted-column";
	private static final String GRID_COLUMN_KEY_INVALID = "grid-invalid-column";
	private static final String GRID_COLUMN_KEY_NAME = "grid-name-column";
	private static final String GRID_COLUMN_KEY_ADDRESS = "grid-address-column";
	private static final String GRID_COLUMN_KEY_EDIT = "grid-edit-column";
	private static final String GRID_COLUMN_HEADER_DELETED = "Deleted";
	private static final String GRID_COLUMN_HEADER_INVALID = "Totally Invalid";
	private static final String GRID_COLUMN_HEADER_NAME = "Name";
	private static final String GRID_COLUMN_HEADER_ADDRESS = "Address";
	private static final String GRID_COLUMN_HEADER_EDIT = "Edit";


	private RadioButtonGroup<String> dbOptionsRadioGroup = new RadioButtonGroup<String>();
	private Grid<DisplayedCourse> courseGrid = new Grid<DisplayedCourse>();

	@Activate
	public void renderView() {

		setSizeFull();

		//		DB options: select the db from which courses will be displayed on the map
		VerticalLayout dbOptionsLayout = new VerticalLayout();
		dbOptionsLayout.setWidthFull();
		dbOptionsLayout.setHeight("20%");
		
		dbOptionsRadioGroup.setItems(PLAYERTOUR_DB_OPTION, IGOLF_DB_OPTION);
		dbOptionsRadioGroup.setLabel("Select the db from which existing courses will be displayed:");
		dbOptionsRadioGroup.addThemeVariants(RadioGroupVariant.LUMO_VERTICAL);	
		dbOptionsLayout.setVisible(true);
		dbOptionsLayout.add(dbOptionsRadioGroup);

		VerticalLayout searchAndGridLayout = new VerticalLayout();
		searchAndGridLayout.setSizeFull();
		searchAndGridLayout.setVisible(false);

		HorizontalLayout searchLayout = new HorizontalLayout();
		searchLayout.setHeight("20%");

		Label searchLabel = new Label("Search Courses by name:");
		searchLayout.add(searchLabel);

		TextField searchText = new TextField();
		searchLayout.add(searchText);

		dbOptionsRadioGroup.addValueChangeListener(evt -> {

			if(dbOptionsRadioGroup.getValue() == null) {
				searchAndGridLayout.setVisible(false);
			}
			else {
				searchAndGridLayout.setVisible(true);	
				searchText.setValue("");
				courseGrid.setVisible(false);
				courseGrid.setItems(Collections.emptyList());
			}
		});

		Button searchButton = new Button("Search", 
				event ->  {
					String query = searchText.getValue();
					executeSearch(query);
					
				});
		searchLayout.add(searchButton);
		add(searchLayout);

		setupGrid();
		setGridEditor();

		searchAndGridLayout.add(searchLayout, courseGrid);
		add(dbOptionsLayout, searchAndGridLayout);

	}

	private void executeSearch(String query) {
		try {
			switch(dbOptionsRadioGroup.getValue()) {
			case PLAYERTOUR_DB_OPTION:
				displayPlayertourResults(apiCourseService.findAllCourses(query, Integer.MAX_VALUE));						
				break;
			case IGOLF_DB_OPTION:
				displayIGolfResults(iGolfCourseSearchService.searchCourseCompleteResponses(query, Integer.MAX_VALUE));						
				break;							
			}
		} catch (UnknownTransformationException e) {
			Notification.show("Error searching for Golf Courses!");
		}
	}

	private void setupGrid() {
		courseGrid.addColumn(DisplayedCourse::getName).setAutoWidth(true).setHeader(GRID_COLUMN_HEADER_NAME).setKey(GRID_COLUMN_KEY_NAME);

		courseGrid.addColumn(DisplayedCourse::getAddress).setAutoWidth(true).setHeader(GRID_COLUMN_HEADER_ADDRESS).setKey(GRID_COLUMN_KEY_ADDRESS);

		courseGrid.addColumn(new ComponentRenderer<>(e -> {
			boolean value = e.isTotallyInvalid();	
			Checkbox checkbox = new Checkbox(value);
			checkbox.setReadOnly(true);		
			checkbox.setEnabled(false);
			return checkbox;
		})).setAutoWidth(true).setHeader(GRID_COLUMN_HEADER_INVALID).setKey(GRID_COLUMN_KEY_INVALID);

		courseGrid.addColumn(new ComponentRenderer<>(e -> {
			boolean value = e.isDeleted();			
			Checkbox checkbox = new Checkbox(value);
			checkbox.setReadOnly(true);		
			checkbox.setEnabled(false);
			return checkbox;
		})).setAutoWidth(true).setHeader(GRID_COLUMN_HEADER_DELETED).setKey(GRID_COLUMN_KEY_DELETED);

		courseGrid.setVisible(false);
		courseGrid.setWidthFull();
	}

	private void setGridEditor() {
		Editor<DisplayedCourse> gridEditor = courseGrid.getEditor();
		Column<DisplayedCourse> editColumn = courseGrid.addColumn(new ComponentRenderer<>(course -> {
			Button editButton = new Button("Edit");
			editButton.addClickListener(e -> {
				if (gridEditor.isOpen())
					gridEditor.cancel();
				courseGrid.getEditor().editItem(course);
			});
			return editButton;
		})).setAutoWidth(true).setHeader(GRID_COLUMN_HEADER_EDIT).setKey(GRID_COLUMN_KEY_EDIT);

		Binder<DisplayedCourse> binder = new Binder<>(DisplayedCourse.class);
		gridEditor.setBinder(binder);
		gridEditor.setBuffered(true);

		TextField courseNameField = new TextField();
		courseNameField.setWidthFull();
		binder.forField(courseNameField)
		.asRequired("Course name must not be empty")
		.withValidator(value -> value != null && !value.isEmpty(), "The course name cannot be left empty!")
		.bind(DisplayedCourse::getName, DisplayedCourse::setName);
		courseGrid.getColumnByKey(GRID_COLUMN_KEY_NAME).setEditorComponent(courseNameField);

		Checkbox courseDeletedField = new Checkbox();
		courseDeletedField.setWidthFull();
		binder.forField(courseDeletedField)
		.bind(DisplayedCourse::isDeleted, DisplayedCourse::setDeleted);
		courseGrid.getColumnByKey(GRID_COLUMN_KEY_DELETED).setEditorComponent(courseDeletedField);		

		Button saveButton = new Button("Save", e -> gridEditor.save());
		Button cancelButton = new Button(VaadinIcon.CLOSE.create(),
				e -> gridEditor.cancel());
		cancelButton.addThemeVariants(ButtonVariant.LUMO_ICON,
				ButtonVariant.LUMO_ERROR);
		HorizontalLayout actions = new HorizontalLayout(saveButton,
				cancelButton);
		actions.setPadding(false);
		editColumn.setEditorComponent(actions);
		
		gridEditor.addSaveListener(event -> {
			DisplayedCourse item = event.getItem();
			switch(dbOptionsRadioGroup.getValue()) {
			case PLAYERTOUR_DB_OPTION:
				courseService.editCourseName(item.getCourseId(), item.getName());
				courseService.markCourseAsDeleted(item.getCourseId(), item.isDeleted());
				break;
			case IGOLF_DB_OPTION:
				iGolfUpdateService.editIGolfCourseName(item.getCourseId(), item.getName());
				iGolfUpdateService.markIGolfCourseAsDeleted(item.getCourseId(), item.isDeleted());
				break;		
			}
			gridEditor.refresh();
		});
	}
	private void displayPlayertourResults(List<GolfCourseSearchResult> result) {
		List<DisplayedCourse> displayedCourses = new LinkedList<>();
		result.stream().forEach(r -> {
			DisplayedCourse displayedCourse = 
					new DisplayedCourse(r.getCourse().getId(), r.getCourse().getName(), 
							r.getCourse().getAddress(), r.getCourse().isDeleted(), r.getCourse().isTotallyInvalid());
			displayedCourses.add(displayedCourse);			
		});
		courseGrid.getColumnByKey(GRID_COLUMN_KEY_INVALID).setVisible(true);
		courseGrid.setItems(displayedCourses);	
		courseGrid.setVisible(true);
		Notification.show("Found " + result.size() + " results!");
	}

	private void displayIGolfResults(List<CourseCompleteResponse> result) {		
		List<DisplayedCourse> displayedCourses = new LinkedList<>();
		result.stream().forEach(r -> {
			String courseName = r.getCourseDetails() != null ? (r.getCourseDetails().getCourseName() != null ?
					r.getCourseDetails().getCourseName() : UNKNOWN_COURSE_NAME) : UNKNOWN_COURSE_NAME;
			if(UNKNOWN_COURSE_NAME.equals(courseName)) {
				courseName = r.getCourseInfo() != null ? (r.getCourseInfo().getCourseName() != null ?
						r.getCourseInfo().getCourseName() : UNKNOWN_COURSE_NAME) : UNKNOWN_COURSE_NAME;
			}
			String address = r.getCourseDetails() != null ? 
					(r.getCourseDetails().getAddress1() + ", " + 
							(r.getCourseDetails().getZipCode() != null ? r.getCourseDetails().getZipCode() + ", " : ", ") + 
							(r.getCourseDetails().getCity() != null ? r.getCourseDetails().getCity() + ", " : ", ") + 
							(r.getCourseDetails().getStateFull() != null ? r.getCourseDetails().getStateFull() + ", " : ", ") + 
							(r.getCourseDetails().getCountryFull() != null ? r.getCourseDetails().getCountryFull(): "")) : UNKNOWN_COURSE_NAME;
			DisplayedCourse displayedCourse = 
					new DisplayedCourse(r.getCourseInfo().getId_course(), courseName, address, r.isDeleted());
			displayedCourses.add(displayedCourse);			
		});
		courseGrid.getColumnByKey(GRID_COLUMN_KEY_INVALID).setVisible(false);
		courseGrid.setItems(displayedCourses);	
		courseGrid.setVisible(true);
		Notification.show("Found " + result.size() + " results!");
	}

	class DisplayedCourse {

		private String courseId;
		private String name;
		private String address;
		private boolean deleted;
		private boolean totallyInvalid;


		public DisplayedCourse(String courseId, String name, Address address, boolean deleted, boolean totallyInvalid) {
			this.courseId = courseId;
			this.name = name;
			if(address == null) {
				this.address = "";
			}
			else {
				StringBuilder sb = new StringBuilder();
				sb.append(address.getRoute() != null ? address.getRoute() + "," : "");
				sb.append(address.getZipCode() != null ? address.getZipCode() + "," : "");
				sb.append(address.getCity() != null ? address.getCity() + "," : "");
				sb.append(address.getState() != null ? address.getState() + "," : "");
				sb.append(address.getCountry() != null ? address.getCountry() : "");				
				this.address = sb.toString();	
			}		
			this.deleted = deleted;
			this.totallyInvalid = totallyInvalid;
		}

		public DisplayedCourse(String courseId, String name, String address, boolean deleted) {
			this.courseId = courseId;
			this.name = name;
			this.address = address == null  ? "" : address;
			this.deleted = deleted;
		}

		public String getCourseId() {
			return courseId;
		}

		public String getName() {
			return name;
		}

		public void setName(String name) {
			this.name = name;
		}

		public String getAddress() {
			return address;
		}

		public boolean isDeleted() {
			return deleted;
		}

		public void setDeleted(boolean deleted) {
			this.deleted = deleted;
		}

		public boolean isTotallyInvalid() {
			return totallyInvalid;
		}

	}


}
