/**
 * Copyright (c) 2012 - 2021 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package com.playertour.backend.vaadin.views.main;

import java.util.Optional;

import org.gecko.vaadin.whiteboard.annotations.VaadinComponent;
import org.osgi.service.cm.ConfigurationAdmin;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.annotations.ServiceScope;
import org.osgi.service.log.Logger;
import org.osgi.service.log.LoggerFactory;

import com.playertour.backend.vaadin.security.AbstractKeycloakSecuredVaadinMainView;
import com.playertour.backend.vaadin.simulator.GameSimulatorView;
import com.playertour.backend.vaadin.views.edit.EditView;
import com.playertour.backend.vaadin.views.index.IndexView;
import com.playertour.backend.vaadin.views.manual.course.ManualCourseView;
import com.playertour.backend.vaadin.views.map.MapView;
import com.playertour.backend.vaadin.views.notification.NotificationView;
import com.playertour.backend.vaadin.views.purchases.PurchasesView;
import com.playertour.backend.vaadin.views.search.SearchView;
import com.playertour.backend.vaadin.views.update.UpdateView;
import com.playertour.backend.vaadin.views.users.UserView;
import com.playertour.backend.vaadin.views.validation.ValidationView;
import com.vaadin.flow.component.ComponentUtil;
import com.vaadin.flow.component.UI;
import com.vaadin.flow.component.applayout.DrawerToggle;
import com.vaadin.flow.component.button.Button;
import com.vaadin.flow.component.dependency.NpmPackage;
import com.vaadin.flow.component.html.H1;
import com.vaadin.flow.component.html.Image;
import com.vaadin.flow.component.icon.Icon;
import com.vaadin.flow.component.icon.VaadinIcon;
import com.vaadin.flow.component.notification.Notification;
import com.vaadin.flow.component.orderedlayout.FlexComponent;
import com.vaadin.flow.component.orderedlayout.HorizontalLayout;
import com.vaadin.flow.component.orderedlayout.VerticalLayout;
import com.vaadin.flow.component.tabs.Tab;
import com.vaadin.flow.component.tabs.Tabs;
import com.vaadin.flow.component.tabs.TabsVariant;
import com.vaadin.flow.router.PageTitle;
import com.vaadin.flow.router.Route;
import com.vaadin.flow.router.RouterLink;


/**
 * 
 * @author ilenia, mhs
 * @since Apr 15, 2021
 */
@Component(service=MainView.class, scope = ServiceScope.PROTOTYPE)
@Route("")
@VaadinComponent()
@NpmPackage(value = "lumo-css-framework", version = "3.0.11") //this is a temporarely workaround for a Vaadin bug in order to make our generator work
public class MainView extends AbstractKeycloakSecuredVaadinMainView {

	private static final long serialVersionUID = 7203879464859515711L;
	private Tabs menu = null;
    private H1 viewTitle;
    private boolean retriggerAfterNavigation = false;
    
    @Reference(service = ConfigurationAdmin.class)
	void setConfigurationAdmin(ConfigurationAdmin configurationAdmin) {
		this.configurationAdmin = configurationAdmin;
	}

    @Reference(service = LoggerFactory.class)
	void setLogger(Logger logger) {
		this.logger = logger;
	}

    private com.vaadin.flow.component.Component createHeaderContent() {
        HorizontalLayout layout = new HorizontalLayout();
        layout.setId("header");
        layout.getThemeList().set("dark", true);
        layout.setWidthFull();
        layout.setSpacing(false);
        layout.setAlignItems(FlexComponent.Alignment.CENTER);
        layout.add(new DrawerToggle());
        viewTitle = new H1();
        layout.add(viewTitle);
        
        Icon logoutIcon = new Icon(VaadinIcon.SIGN_OUT);
        logoutIcon.setColor("white");
        Button logoutBtn = new Button();
        logoutBtn.setIcon(logoutIcon);
        logoutBtn.addClickListener(event -> {
        	Notification.show("Logout!");
			UI.getCurrent().getPage().executeJs("keycloak.logout()");
		});
        logoutBtn.getStyle().set(
        		"margin-left", "auto");
        layout.add(logoutBtn);
        return layout;
    }

    private com.vaadin.flow.component.Component createDrawerContent(Tabs menu) {
        VerticalLayout layout = new VerticalLayout();
        layout.setSizeFull();
        layout.setPadding(false);
        layout.setSpacing(false);
        layout.getThemeList().set("spacing-s", true);
        layout.setAlignItems(FlexComponent.Alignment.STRETCH);
        VerticalLayout logoLayout = new VerticalLayout();
        logoLayout.setId("logo");
        logoLayout.setAlignItems(FlexComponent.Alignment.CENTER);
        logoLayout.add(new Image("images/logo_100x63.png", "Playertour logo"));
        layout.add(logoLayout, menu);
        return layout;
    }

    private Tabs createMenu() {
        final Tabs tabs = new Tabs();
        tabs.setOrientation(Tabs.Orientation.VERTICAL);
        tabs.addThemeVariants(TabsVariant.LUMO_MINIMAL);
        tabs.setId("tabs");
        tabs.add(createMenuItems());
        tabs.setSelectedTab(null);
        return tabs;
    }

    private com.vaadin.flow.component.Component[] createMenuItems() {
        return new Tab[]{
        		createTab("Update", UpdateView.class), 
        		createTab("Validate", ValidationView.class), 
        		createTab("Index", IndexView.class),        		
        		createTab("Search", SearchView.class),
        		createTab("Map", MapView.class),
        		createTab("Edit", EditView.class),
        		createTab("Users", UserView.class),
        		createTab("Purchases", PurchasesView.class),
        		createTab("Notification", NotificationView.class),
        		createTab("Game Simulation", GameSimulatorView.class),
        		createTab("Manual Course Creation", ManualCourseView.class)
        		};
    }

    private static Tab createTab(String text, Class<? extends com.vaadin.flow.component.Component> navigationTarget) {
        final Tab tab = new Tab();
        tab.add(new RouterLink(text, navigationTarget));
        ComponentUtil.setData(tab, Class.class, navigationTarget);
        return tab;
    }

    @Override
    protected void afterNavigation() {
        super.afterNavigation();
        if(menu != null) {
        	getTabForComponent(getContent()).ifPresent(menu::setSelectedTab);
        	retriggerAfterNavigation = false;
        }
        else {
        	retriggerAfterNavigation = true;
        }
        if(viewTitle != null) {
        	viewTitle.setText(getCurrentPageTitle());
        	retriggerAfterNavigation = false;
        }       
    }

    private Optional<Tab> getTabForComponent(com.vaadin.flow.component.Component component) {    	
    	Optional<Tab> optional = menu.getChildren().filter(tab -> ComponentUtil.getData(tab, Class.class).equals(component.getClass()))
                .findFirst().map(Tab.class::cast);
        return optional;
    }

    private String getCurrentPageTitle() {
        PageTitle title = getContent().getClass().getAnnotation(PageTitle.class);
        return title == null ? "" : title.value();
    }

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.vaadin.security.AbstractKeycloakSecuredVaadinMainView#renderView()
	 */
	@Override
	public void renderView() {
        setPrimarySection(Section.DRAWER);
        addToNavbar(true, createHeaderContent());
        menu = createMenu();
        addToDrawer(createDrawerContent(menu));
        if(retriggerAfterNavigation) {
        	afterNavigation();
        }
	}
}
