/**
 * Copyright (c) 2012 - 2022 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package com.playertour.backend.vaadin.views.map;

import java.lang.reflect.InvocationTargetException;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.Callable;
import java.util.concurrent.Executors;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import org.gecko.util.pushstreams.GeckoPushbackPolicyOption;
import org.gecko.vaadin.whiteboard.annotations.VaadinComponent;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.annotations.ReferenceScope;
import org.osgi.service.component.annotations.ServiceScope;
import org.osgi.util.pushstream.PushEvent;
import org.osgi.util.pushstream.PushEventSource;
import org.osgi.util.pushstream.PushStream;
import org.osgi.util.pushstream.PushStreamProvider;
import org.osgi.util.pushstream.QueuePolicyOption;

import com.playertour.backend.apis.course.CourseSearchService;
import com.playertour.backend.apis.geoapify.GeoapifyService;
import com.playertour.backend.apis.igolf.IGolfCourseService;
import com.playertour.backend.apis.igolf.IGolfCourseUpdateService;
import com.playertour.backend.apis.igolf.exceptions.IGolfUpdateException;
import com.playertour.backend.geoapify.model.geoapify.FeatureCollection;
import com.playertour.backend.golfcourse.model.golfcourse.GolfCourse;
import com.playertour.backend.igolf.model.igolf.CourseCompleteResponse;
import com.playertour.backend.vaadin.helper.UiUpdateThread;
import com.playertour.backend.vaadin.helper.VaadinViewProgressMonitor;
import com.playertour.backend.vaadin.views.main.MainView;
import com.vaadin.flow.component.UI;
import com.vaadin.flow.component.button.Button;
import com.vaadin.flow.component.dependency.NpmPackage;
import com.vaadin.flow.component.grid.Grid;
import com.vaadin.flow.component.html.Label;
import com.vaadin.flow.component.notification.Notification;
import com.vaadin.flow.component.notification.NotificationVariant;
import com.vaadin.flow.component.orderedlayout.HorizontalLayout;
import com.vaadin.flow.component.orderedlayout.VerticalLayout;
import com.vaadin.flow.component.progressbar.ProgressBar;
import com.vaadin.flow.component.progressbar.ProgressBarVariant;
import com.vaadin.flow.component.radiobutton.RadioButtonGroup;
import com.vaadin.flow.component.radiobutton.RadioGroupVariant;
import com.vaadin.flow.component.textfield.TextField;
import com.vaadin.flow.router.PageTitle;
import com.vaadin.flow.router.Route;

/**
 * 
 * @author ilenia
 * @since May 6, 2022
 */
@Route(value = "map", layout = MainView.class)
@PageTitle("Map")
@NpmPackage(value = "leaflet", version = "^1.7.1")
@Component(name = "MapView", service = MapView.class, scope = ServiceScope.PROTOTYPE)
@VaadinComponent()
public class MapView extends VerticalLayout{

	@Reference
	private GeoapifyService geoapifyService;

	@Reference(target = "(component.name=IGolfDBCourseService)", scope = ReferenceScope.PROTOTYPE_REQUIRED)
	private IGolfCourseService iGolfDBCourseService;

	@Reference(target = "(component.name=IGolfCourseUpdateService)", scope = ReferenceScope.PROTOTYPE_REQUIRED)
	private IGolfCourseUpdateService iGolfCourseUpdateService;

	@Reference
	private CourseSearchService courseSearchService;

	/** serialVersionUID */
	private static final long serialVersionUID = -5323563277713860651L;
	private static final String PLAYERTOUR_DB_OPTION = "playertour db";
	private static final String IGOLF_DB_OPTION = "igolf db";
	private static final String RADIUS_OPTION_50 = "50 km";
	private static final String RADIUS_OPTION_500 = "500 km";
	private static final String COURSES_ALL_OPTION = "All courses";
	private static final String COURSES_VALID_OPTION = "Only valid courses";
	private static final String COURSES_INVALID_OPTION = "Only invalid courses";
	private static final String COURSES_TOT_INVALID_OPTION = "Only totally invalid courses";
	private static final Double INITIAL_MAP_LATITUDE = 52.5157; //Berlin
	private static final Double INITIAL_MAP_LONGITUDE = 13.39379; //Berlin
	private static final Integer INITIAL_MAP_ZOOM_LEVEL = 10;
	private static final Double KM_TO_MILES_CONVERSION_FACTOR = 0.621371;

	private PushStreamProvider pushStreamProvider = new PushStreamProvider();
	private LeafletGolfMap map = new LeafletGolfMap();

	@Activate
	public void renderView() {

		setSizeFull();

//		DB options: select the db from which courses will be displayed on the map
		
		VerticalLayout dbOptionsLayout = new VerticalLayout();
		dbOptionsLayout.setWidthFull();
		dbOptionsLayout.setHeight("40%");
		RadioButtonGroup<String> dbOptionsRadioGroup = new RadioButtonGroup<String>();
		dbOptionsRadioGroup.setItems(PLAYERTOUR_DB_OPTION, IGOLF_DB_OPTION);
		dbOptionsRadioGroup.setLabel("Select the db from which existing courses will be displayed:");
		dbOptionsRadioGroup.addThemeVariants(RadioGroupVariant.LUMO_VERTICAL);	
		dbOptionsLayout.setVisible(true);
		dbOptionsLayout.add(dbOptionsRadioGroup);
		
//		SEARCH FIELD: to search for a certain location to center the map on and show existing courses

		HorizontalLayout searchLayout = new HorizontalLayout();
		searchLayout.setHeight("20%");
		searchLayout.setWidthFull();
		searchLayout.setVisible(false);

		Label searchLabel = new Label("Enter Location:");
		searchLabel.setWidth("20%");

		TextField searchText = new TextField();
		searchText.setWidthFull();

		Button gridBtn = new Button("Show Grid");
		gridBtn.setWidth("20%");

		Grid<DisplayedSearchResult> searchGrid = new Grid<DisplayedSearchResult>();
		searchGrid.addColumn(DisplayedSearchResult::getFormattedAddress);		
		searchGrid.setSizeFull();
		searchGrid.setVisible(false);
		searchGrid.addItemClickListener(evt -> {
			map.removeUpdateArea();
			map.setView(evt.getItem().getLatitude(), evt.getItem().getLongitude(), INITIAL_MAP_ZOOM_LEVEL);
			switch(dbOptionsRadioGroup.getValue()) {
			case PLAYERTOUR_DB_OPTION:
				Map<GolfCourse, Double> golfCoursesMap = courseSearchService.searchCoursesWithinDist(evt.getItem().getLatitude(), evt.getItem().getLongitude(), 50000);
				showPlayertourCourses(golfCoursesMap, true, COURSES_ALL_OPTION);
				break;
			case IGOLF_DB_OPTION:
				try {
					PushEventSource<CourseCompleteResponse> iGolfDBCourseEventSource = 
							iGolfDBCourseService.getCoursesNearMe(evt.getItem().getLatitude(), evt.getItem().getLongitude(), (int) (50.*KM_TO_MILES_CONVERSION_FACTOR));
					showIGolfCourses(iGolfDBCourseEventSource, true);
				} catch (Exception e) {
					Notification.show("Exception while searching for existing IGolf courses in db!").addThemeVariants(NotificationVariant.LUMO_ERROR);
				}
				break;
			}
			

			searchGrid.setVisible(false);
			gridBtn.setVisible(true);			
		});		

		gridBtn.setVisible(true);
		gridBtn.addClickListener(evt -> {
			searchGrid.setVisible(true);
			gridBtn.setVisible(false);
		});

		Button searchButton = new Button("Search", 
				event ->  {
					try {
						String query = searchText.getValue();
						FeatureCollection featureColl = geoapifyService.getAddressByAutocompletion(query);
						List<DisplayedSearchResult> results = featureColl.getFeatures().stream()
								.map(f -> new DisplayedSearchResult(f.getProperties().getFormatted(), f.getProperties().getLat(), f.getProperties().getLon()))
								.collect(Collectors.toList());
						searchGrid.setItems(results);
						searchGrid.setVisible(true);
						gridBtn.setVisible(false);
					} catch (Exception e) {
						Notification.show("Error searching for Location!").addThemeVariants(NotificationVariant.LUMO_ERROR);
					}
				});
		searchButton.setWidth("20%");
		searchLayout.add(searchLabel, searchText, searchButton, searchGrid, gridBtn);

//		MAP: to display existing courses and the area within which igolf update can be triggered
		
		VerticalLayout mapLayout = new VerticalLayout();
		mapLayout.setSizeFull();
		mapLayout.setVisible(false);

		map.setSizeFull();
		map.setView(INITIAL_MAP_LATITUDE, INITIAL_MAP_LONGITUDE, INITIAL_MAP_ZOOM_LEVEL);
		mapLayout.add(map);

//		BUTTONS: to show the area within which an igolf update can be triggered and existing courses are displayed

		Button updateViewBtn = new Button("Update View");
		updateViewBtn.setHeight("20%");
		updateViewBtn.setEnabled(false);

		Button updateIGolfBtn = new Button("Update IGolf DB");
		updateIGolfBtn.setHeight("20%");
		updateIGolfBtn.setEnabled(false);
		updateIGolfBtn.setVisible(false);

//		OPTIONS: to adjust the displayed options (valid/invalid playetour courses) and to adjust the radius of the updated/displayed area
		
		VerticalLayout optionsLayout = new VerticalLayout();
		optionsLayout.setSizeFull();
		optionsLayout.setVisible(false);

		RadioButtonGroup<String> radiusRadioButton = new RadioButtonGroup<String>();
		radiusRadioButton.addThemeVariants(RadioGroupVariant.LUMO_VERTICAL);	
		radiusRadioButton.setItems(RADIUS_OPTION_50, RADIUS_OPTION_500);
		radiusRadioButton.setLabel("Select update radius:");
		radiusRadioButton.setHelperText("This is the radius within which the update will be performed.");
		

		RadioButtonGroup<String> coursesRadioButton = new RadioButtonGroup<String>();
		coursesRadioButton.addThemeVariants(RadioGroupVariant.LUMO_VERTICAL);	
		coursesRadioButton.setItems(COURSES_ALL_OPTION, COURSES_VALID_OPTION, COURSES_INVALID_OPTION, COURSES_TOT_INVALID_OPTION);
		coursesRadioButton.setLabel("Select the courses you want to see:");
		coursesRadioButton.setVisible(false);
		coursesRadioButton.addValueChangeListener(evt -> {
			if(evt.getValue() == null) {
				updateViewBtn.setEnabled(false);
				return;
			}
			if(radiusRadioButton.getValue() == null) {
				return;
			}
			updateViewBtn.setEnabled(true);
		});
		
		radiusRadioButton.addValueChangeListener(evt -> {
			if(evt.getValue() == null) {
				updateViewBtn.setEnabled(false);
				updateIGolfBtn.setEnabled(false);
				return;
			}
			switch(dbOptionsRadioGroup.getValue()) {
			case PLAYERTOUR_DB_OPTION:
				if(coursesRadioButton.getValue() == null) {
					return;
				}
				updateViewBtn.setEnabled(true);
				break;			
			case IGOLF_DB_OPTION:
				updateViewBtn.setEnabled(true);
				updateIGolfBtn.setEnabled(true);
				break;
			}
		});

		updateViewBtn.addClickListener(evt -> {
			map.setUpdateRadius(radiusRadioButton.getValue() == RADIUS_OPTION_500 ? 500000. : 50000.);			
			map.getCurrentLatitude().then(latRes -> {
				double lat = latRes.asNumber();
				map.getCurrentLongitude().then(lngRes -> {
					double lng = lngRes.asNumber();
					switch(dbOptionsRadioGroup.getValue()) {
					case PLAYERTOUR_DB_OPTION:
						int radiusInMeters = radiusRadioButton.getValue() == RADIUS_OPTION_500 ? 500000 : 50000;
						Map<GolfCourse, Double> golfCoursesMap = courseSearchService.searchCoursesWithinDist(lat, lng, radiusInMeters);
						showPlayertourCourses(golfCoursesMap, true, coursesRadioButton.getValue());
						break;
					case IGOLF_DB_OPTION:
						map.showUpdateArea();
						int radiusInMiles = radiusRadioButton.getValue() == RADIUS_OPTION_500 ? (int) (500.*KM_TO_MILES_CONVERSION_FACTOR) : 
							(int) (50.*KM_TO_MILES_CONVERSION_FACTOR);
						try {
							PushEventSource<CourseCompleteResponse> iGolfDBCourseEventSource = 
									iGolfDBCourseService.getCoursesNearMe(lat, lng,radiusInMiles);
							showIGolfCourses(iGolfDBCourseEventSource, true);
						} catch (InvocationTargetException | IGolfUpdateException | InterruptedException e) {
							Notification.show("Error updating exisiting courses on map!").addThemeVariants(NotificationVariant.LUMO_ERROR);
						}
						break;
					}
					
				});
			});			
		});	     

		
//		PROGRESS BAR: progress bar to display the progress when triggering an igolf update
		
		HorizontalLayout iGolfCoursesBarLayout = new HorizontalLayout();
		iGolfCoursesBarLayout.setSizeFull();
		iGolfCoursesBarLayout.setVisible(false);

		ProgressBar iGolfCoursesProgressBar = new ProgressBar(0., 100.);
		iGolfCoursesProgressBar.setIndeterminate(true);
		iGolfCoursesProgressBar.setVisible(false);
		iGolfCoursesProgressBar.setWidthFull();
		iGolfCoursesProgressBar.setHeight("30%");
		iGolfCoursesProgressBar.addThemeVariants(ProgressBarVariant.LUMO_CONTRAST);

		Label iGolfCoursesBarLabel = new Label();
		iGolfCoursesBarLabel.setSizeFull();
		iGolfCoursesBarLabel.setVisible(false);
		iGolfCoursesBarLayout.add(iGolfCoursesProgressBar, iGolfCoursesBarLabel);

		VaadinViewProgressMonitor iGolfCoursesProgressMonitor = 
				new VaadinViewProgressMonitor(iGolfCoursesProgressBar, iGolfCoursesBarLabel, UI.getCurrent());

		updateIGolfBtn.addClickListener(evt -> {
			int radiusInMiles = radiusRadioButton.getValue() == RADIUS_OPTION_500 ? (int) (500.*KM_TO_MILES_CONVERSION_FACTOR) : 
				(int) (50.*KM_TO_MILES_CONVERSION_FACTOR);

			iGolfCourseUpdateService.setForceUpdate(false);

			map.getCurrentLatitude().then(latRes -> {
				double lat = latRes.asNumber();
				map.getCurrentLongitude().then(lngRes -> {
					double lng = lngRes.asNumber();
					try {
						iGolfCoursesProgressBar.setVisible(true);
						iGolfCoursesBarLabel.setVisible(true);
						Callable<Void> courseUpdateCall =  new Callable<Void>() {
							@Override
							public Void call() throws Exception {
								iGolfCourseUpdateService.grabAndSaveIGolfCoursesWithinDist(lat, lng, radiusInMiles, iGolfCoursesProgressMonitor);
								return null;
							}					
						};
						Callable<Void> endCall =  new Callable<Void>() {
							@Override
							public Void call() throws Exception {
								PushEventSource<CourseCompleteResponse> iGolfDBCourseEventSource = iGolfDBCourseService.getCoursesNearMe(lat, lng,radiusInMiles);
								showIGolfCourses(iGolfDBCourseEventSource, true);
								return null;
							}					
						};
						UI.getCurrent().setPollInterval(500);				
						UiUpdateThread updateThread = 
								new UiUpdateThread(courseUpdateCall, UI.getCurrent(), iGolfCoursesProgressBar, iGolfCoursesProgressMonitor, 
										updateIGolfBtn, iGolfCoursesBarLabel);
						updateThread.setEndCall(endCall);
						updateThread.start();						
					} catch (Exception e) {
						Notification.show("Error updating exisiting courses on map!").addThemeVariants(NotificationVariant.LUMO_ERROR);
					}
				});
			});						
		});
		
		dbOptionsRadioGroup.addValueChangeListener(evt -> {
			map.removeCourses();
			map.removeUpdateArea();
			radiusRadioButton.setValue(null);
			coursesRadioButton.setValue(null);
			searchLayout.setVisible(true);
			mapLayout.setVisible(true);
			optionsLayout.setVisible(true);
			switch(evt.getValue()) {
			case PLAYERTOUR_DB_OPTION:
				coursesRadioButton.setVisible(true);
				updateViewBtn.setVisible(true);
				iGolfCoursesBarLayout.setVisible(false);
				updateIGolfBtn.setVisible(false);
				break;
			case IGOLF_DB_OPTION:
				coursesRadioButton.setVisible(false);
				updateViewBtn.setVisible(true);
				iGolfCoursesBarLayout.setVisible(true);
				updateIGolfBtn.setVisible(true);
				break;
			}
		});

		optionsLayout.add(radiusRadioButton, coursesRadioButton, updateViewBtn, updateIGolfBtn, iGolfCoursesBarLayout);
		add(dbOptionsLayout, searchLayout, mapLayout, optionsLayout);
	}

	private void  showIGolfCourses(PushEventSource<CourseCompleteResponse> iGolfCourseEventSource, boolean cleanDisplayedCourses) throws IGolfUpdateException, InvocationTargetException, InterruptedException {

		if(cleanDisplayedCourses) {
			map.removeCourses();
		}
		PushStream<CourseCompleteResponse> iGolfCoursePushStream = pushStreamProvider.buildStream(iGolfCourseEventSource)
				.withPushbackPolicy(GeckoPushbackPolicyOption.LINEAR_AFTER_THRESHOLD.getPolicy(25))
				.withQueuePolicy(QueuePolicyOption.BLOCK)
				.withExecutor(Executors.newSingleThreadExecutor())
				.withBuffer(new ArrayBlockingQueue<PushEvent<? extends CourseCompleteResponse>>(50))
				.build();
		iGolfCoursePushStream.forEach(ccr -> {
			DisplayedCourse course = new DisplayedCourse(ccr.getCourseDetails().getCourseName(), ccr.getCourseDetails().getLatitude(), ccr.getCourseDetails().getLongitude());
			map.showCourse(course);
		});
	}

	private void showPlayertourCourses(Map<GolfCourse, Double> playertourCoursesMap, boolean cleanDisplayedCourses, String displayOption) {
		if(cleanDisplayedCourses) {
			map.removeCourses();
		}
		Predicate<GolfCourse> filter;
		switch(displayOption) {
		case COURSES_INVALID_OPTION:
			filter = gc -> gc.isInvalid();
			break;
		case COURSES_TOT_INVALID_OPTION:
			filter = gc -> gc.isTotallyInvalid();
			break;
		case COURSES_VALID_OPTION:
			filter = gc -> !gc.isInvalid();
			break;
		case COURSES_ALL_OPTION: default:
			filter = gc -> true;
			break;
		}
		playertourCoursesMap.keySet().stream().filter(filter).forEach(gc -> {
			DisplayedCourse course = 
					new DisplayedCourse(gc.getCourseDetails().getCourseName(), gc.getCourseDetails().getLocation().getLatitude(), 
							gc.getCourseDetails().getLocation().getLongitude(), gc.isInvalid(), gc.isTotallyInvalid());
			map.showCourse(course);
		});
	}




	public class DisplayedCourse {
		private String name;
		private Double latitude;
		private Double longitude;
		private Boolean isInvalid;
		private Boolean isTotallyInvalid;

		public DisplayedCourse(String name, Double latitude, Double longitude) {
			this.name = name;
			this.latitude = latitude;
			this.longitude = longitude;
		}
		
		public DisplayedCourse(String name, Double latitude, Double longitude, boolean isInvalid, boolean isTotallyInvalid) {
			this.name = name;
			this.latitude = latitude;
			this.longitude = longitude;
			this.isInvalid = isInvalid;
			this.isTotallyInvalid = isTotallyInvalid;
		}
		
		/**
		 * Returns the name.
		 * @return the name
		 */
		public String getName() {
			return name;
		}
		/**
		 * Returns the latitude.
		 * @return the latitude
		 */
		public Double getLatitude() {
			return latitude;
		}
		/**
		 * Returns the longitude.
		 * @return the longitude
		 */
		public Double getLongitude() {
			return longitude;
		}
		
		public Boolean isInvalid() {
			return isInvalid;
		}

		public Boolean isTotallyInvalid() {
			return isTotallyInvalid;
		}
	}

	class DisplayedSearchResult {
		private String formattedAddress;
		private Double latitude;
		private Double longitude;

		public DisplayedSearchResult(String formattedAddress, double latitude, double longitude) {
			this.formattedAddress = formattedAddress;
			this.latitude = latitude;
			this.longitude = longitude;
		}

		public String getFormattedAddress() {
			return this.formattedAddress;
		}

		public Double getLatitude() {
			return this.latitude;
		}

		public Double getLongitude() {
			return this.longitude;
		}
	}
}
