/**
 * Copyright (c) 2012 - 2022 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package com.playertour.backend.vaadin.views.purchases;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Collections;
import java.util.Date;
import java.util.Optional;

import org.gecko.vaadin.whiteboard.annotations.VaadinComponent;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.annotations.ReferenceScope;
import org.osgi.service.component.annotations.ServiceScope;

import com.playertour.backend.apis.player.PlayerSearchService;
import com.playertour.backend.player.model.player.Player;
import com.playertour.backend.purchases.model.purchases.Purchase;
import com.playertour.backend.purchases.model.purchases.PurchaseHistory;
import com.playertour.backend.purchases.model.purchases.PurchaseReceipt;
import com.playertour.backend.purchases.model.purchases.PurchaseReceiptVerificationStatus;
import com.playertour.backend.purchases.service.api.PurchasesService;
import com.playertour.backend.vaadin.views.main.MainView;
import com.vaadin.flow.component.button.Button;
import com.vaadin.flow.component.checkbox.Checkbox;
import com.vaadin.flow.component.grid.Grid;
import com.vaadin.flow.component.html.Div;
import com.vaadin.flow.component.html.Label;
import com.vaadin.flow.component.notification.Notification;
import com.vaadin.flow.component.notification.NotificationVariant;
import com.vaadin.flow.component.orderedlayout.HorizontalLayout;
import com.vaadin.flow.component.orderedlayout.VerticalLayout;
import com.vaadin.flow.component.textfield.TextField;
import com.vaadin.flow.data.binder.Binder;
import com.vaadin.flow.data.binder.ReadOnlyHasValue;
import com.vaadin.flow.data.provider.SortDirection;
import com.vaadin.flow.data.renderer.ComponentRenderer;
import com.vaadin.flow.router.PageTitle;
import com.vaadin.flow.router.Route;

/**
 * 
 * @author ilenia
 * @since Oct 5, 2022
 */
@Route(value = "purchases", layout = MainView.class)
@PageTitle("Purchases")
@Component(name = "PurchasesView", service = PurchasesView.class, scope = ServiceScope.PROTOTYPE)
@VaadinComponent()
public class PurchasesView extends VerticalLayout {

	/** serialVersionUID */
	private static final long serialVersionUID = -6329204762467722476L;

	@Reference
	PlayerSearchService playerSearchService;

	@Reference(scope = ReferenceScope.PROTOTYPE_REQUIRED)
	PurchasesService purchasesService;
	
	private static final DateFormat DATE_FORMAT = new SimpleDateFormat("dd MMM yyyy HH:mm:ss");   
	
	private Grid<Purchase> purchasesGrid = new Grid<>();
	private Purchase gridOldSelectedPurchase;
	private Grid<PurchaseReceiptVerificationStatus> purchaseReceiptVerStatusGrid = new Grid<>();
	private VerticalLayout receiptLayout = new VerticalLayout();
	private Binder<PurchaseReceipt> receiptBinder = new Binder<>();

	@Activate
	public void renderView() {

		setSizeFull();

		HorizontalLayout searchLayout = new HorizontalLayout();
		Label searchLabel = new Label("Search User by username:");
		TextField searchText = new TextField();
		Button searchButton = new Button("Search", 
				event ->  {
					String query = searchText.getValue();
					Player player = playerSearchService.searchPlayerByExactUsername(query);
					if(player == null) {
						Notification.show("No Player found with the provided username!").addThemeVariants(NotificationVariant.LUMO_ERROR);
					}
					PurchaseHistory purchaseHistory = purchasesService.getPurchaseHistory(player.getLoginId(), Optional.empty());
					displayPurchases(purchaseHistory);
				});
		searchLayout.add(searchLabel, searchText, searchButton);
		
		setupPurchasesGrid();
		setupReceiptLayout();		
		
		add(searchLayout, purchasesGrid, receiptLayout);
	}

	private void setupReceiptLayout() {
		receiptLayout.setVisible(false);
		receiptLayout.setSizeFull();
	
		HorizontalLayout localDataLayout = new HorizontalLayout();
		localDataLayout.setSizeFull();
		
		Label localDataLabel = new Label("Local Verification Data:");
		
		Div localDataText = new Div();
		ReadOnlyHasValue<String> localData = new ReadOnlyHasValue<>(text -> localDataText.setText(text));
		receiptBinder.forField(localData).bind(PurchaseReceipt::getLocalVerificationData, null);
		
		localDataLayout.add(localDataLabel, localDataText);
		
		HorizontalLayout serverDataLayout = new HorizontalLayout();
		serverDataLayout.setSizeFull();
		
		Label serverDataLabel = new Label("Server Verification Data:");
		
		Div serverDataText = new Div();
		ReadOnlyHasValue<String> serverData = new ReadOnlyHasValue<>(text -> serverDataText.setText(text));
		receiptBinder.forField(serverData).bind(PurchaseReceipt::getServerVerificationData, null);
		
		serverDataLayout.add(serverDataLabel, serverDataText);
		
		purchaseReceiptVerStatusGrid.addColumn(new ComponentRenderer<>(e -> {
			Checkbox checkbox = new Checkbox(e.isValid());
			checkbox.setReadOnly(true);		
			checkbox.setEnabled(false);
			return checkbox;
		})).setAutoWidth(true).setHeader("Valid");
		
		purchaseReceiptVerStatusGrid.addColumn(new ComponentRenderer<>(e -> {
			Checkbox checkbox = new Checkbox(e.isRetry());
			checkbox.setReadOnly(true);		
			checkbox.setEnabled(false);
			return checkbox;
		})).setAutoWidth(true).setHeader("Retry");
		
		purchaseReceiptVerStatusGrid.addColumn(vs -> vs.getErrorMessage()).setAutoWidth(true).setHeader("Err Msg");
		purchaseReceiptVerStatusGrid.addColumn(vs -> DATE_FORMAT.format(new Date(vs.getVerificationDateTime()))).setHeader("Verification Date");		
		receiptLayout.add(localDataLayout, serverDataLayout, purchaseReceiptVerStatusGrid);
	}
	
	private void setupPurchasesGrid() {
		purchasesGrid.addColumn(Purchase::getOrderID).setHeader("Order ID");
		purchasesGrid.addColumn(Purchase::getSource).setHeader("Source");
		purchasesGrid.addColumn(p -> DATE_FORMAT.format(new Date(p.getPurchaseDateTime()))).setHeader("Date");
		purchasesGrid.addColumn(p -> p.getStatus().toString()).setHeader("Status");
		purchasesGrid.addColumn(new ComponentRenderer<>(selectedPurchase -> {	
			Button receiptBtn = new Button("Inspect Receipt", evt ->  {
				displayPurchaseReceipt(selectedPurchase.getReceipt());
			});			
			return receiptBtn;
		})).setHeader("Receipt");
		purchasesGrid.setVisible(false);
		purchasesGrid.setSizeFull();
		purchasesGrid.addSelectionListener(evt -> {
			if(evt.getFirstSelectedItem().isPresent()) {
				Purchase selectedPurchase = evt.getFirstSelectedItem().get();
				if(selectedPurchase != gridOldSelectedPurchase) {
					receiptLayout.setVisible(false);
					purchaseReceiptVerStatusGrid.setItems(Collections.emptyList());
				}
			}			
		});
	}
	
	private void displayPurchaseReceipt(PurchaseReceipt receipt) {
		receiptLayout.setVisible(true);
		receiptBinder.readBean(receipt);
		purchaseReceiptVerStatusGrid.setItems(receipt.getVerificationStatus()).addSortOrder(v -> v.getVerificationDateTime(), SortDirection.DESCENDING);		
	}

	private void displayPurchases(PurchaseHistory purchaseHistory) {
		if(purchaseHistory == null || purchaseHistory.getPurchases().isEmpty()) {
			Notification.show("No Purchases found for the provided player").addThemeVariants(NotificationVariant.LUMO_CONTRAST);
			return;
		}
		purchasesGrid.setItems(purchaseHistory.getPurchases()).addSortOrder(v -> v.getPurchaseDateTime(), SortDirection.DESCENDING);
		purchasesGrid.setVisible(true);		
	}

}
