package com.playertour.backend.vaadin.views.search;
/**
 * Copyright (c) 2012 - 2021 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;

import org.gecko.vaadin.whiteboard.annotations.VaadinComponent;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.annotations.ReferenceScope;
import org.osgi.service.component.annotations.ServiceScope;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.playertour.backend.api.Address;
import com.playertour.backend.api.CourseType;
import com.playertour.backend.api.GolfCourseSearchResult;
import com.playertour.backend.api.GolfCourseSearchResults;
import com.playertour.backend.api.Location;
import com.playertour.backend.api.PlayerApiFactory;
import com.playertour.backend.apis.course.CourseGPSService;
import com.playertour.backend.apis.mmt.common.UnknownTransformationException;
import com.playertour.backend.apis.player.APIGolfCourseService;
import com.playertour.backend.golfcourse.model.golfcourse.CourseGPSVector;
import com.playertour.backend.vaadin.views.main.MainView;
import com.vaadin.flow.component.ComponentEventListener;
import com.vaadin.flow.component.button.Button;
import com.vaadin.flow.component.dependency.CssImport;
import com.vaadin.flow.component.dependency.NpmPackage;
import com.vaadin.flow.component.grid.Grid;
import com.vaadin.flow.component.grid.ItemClickEvent;
import com.vaadin.flow.component.html.Anchor;
import com.vaadin.flow.component.html.Label;
import com.vaadin.flow.component.notification.Notification;
import com.vaadin.flow.component.notification.NotificationVariant;
import com.vaadin.flow.component.orderedlayout.HorizontalLayout;
import com.vaadin.flow.component.orderedlayout.VerticalLayout;
import com.vaadin.flow.component.textfield.TextField;
import com.vaadin.flow.data.renderer.TemplateRenderer;
import com.vaadin.flow.router.PageTitle;
import com.vaadin.flow.router.Route;

/**
 * 
 * @author ilenia, mhs
 * @since Apr 8, 2021
 */
@Route(value = "search", layout = MainView.class)
@PageTitle("Search")
@NpmPackage(value = "leaflet", version = "^1.6.0")
@Component(name = "SearchView", service = SearchView.class, scope = ServiceScope.PROTOTYPE)
@CssImport(themeFor = "vaadin-grid", value = "./themes/wbapp/views/search-view.css")
@VaadinComponent()
public class SearchView extends VerticalLayout {

	/** serialVersionUID */
	private static final long serialVersionUID = -6309340625378010985L;
	private LeafletMap map = new LeafletMap();
	
	private static final ObjectMapper OBJECT_MAPPER = new ObjectMapper();

	@Reference(scope = ReferenceScope.PROTOTYPE_REQUIRED)
	APIGolfCourseService apiCourseService;
	
	@Reference(scope = ReferenceScope.PROTOTYPE_REQUIRED)
	CourseGPSService courseGPSService;

	Grid<DisplayedCourse> grid = new Grid<DisplayedCourse>();
	
	@Activate
	public void renderView() {

		HorizontalLayout hl1 = new HorizontalLayout();

		Label searchLabel = new Label("Search Courses by name:");
		hl1.add(searchLabel);

		TextField searchText = new TextField();
		hl1.add(searchText);

		GolfCourseSearchResults result = PlayerApiFactory.eINSTANCE.createGolfCourseSearchResults();
		Button searchButton = new Button("Search", 
				event ->  {
					result.getResults().clear();
					String query = searchText.getValue();
					try {
						List<GolfCourseSearchResult> courses = apiCourseService.findValidCourses(query, Integer.MAX_VALUE);
						result.getResults().addAll(courses);
						displayResults(result);						
						Notification.show("Found " + courses.size() + " results!");
					} catch (UnknownTransformationException e) {
						Notification.show("Error searching for Golf Courses!");
					}
		});
		hl1.add(searchButton);
		add(hl1);
		
		grid.addColumn(DisplayedCourse::getName).setHeader("Name");
		grid.addColumn(DisplayedCourse::getAddress).setHeader("Address");
		grid.addComponentColumn(v-> new Anchor(v.url, v.url)).setHeader("URL");
		grid.addColumn(TemplateRenderer.<DisplayedCourse> of("<span title='[[item.description]]'>[[item.value]]</span>")
				.withProperty("value", DisplayedCourse::getCourseType)
				.withProperty("description", p -> p.hasOverlappingHoles ? "Course has overlapping holes" : "")).setHeader("# Holes").setKey("holes-num-column");
		grid.getColumnByKey("holes-num-column").setClassNameGenerator(vs -> vs.hasOverlappingHoles() ? "overlapping-holes" : null);
		grid.setWidthFull();
		grid.addItemClickListener(new ComponentEventListener<ItemClickEvent<DisplayedCourse>>() {
			
			/** serialVersionUID */
			private static final long serialVersionUID = -5607636796407870632L;

			@Override
			public void onComponentEvent(ItemClickEvent<DisplayedCourse> event) {
				DisplayedCourse item = event.getItem();	
				
				 setSizeFull();
			     setPadding(true);
			     map.setSizeFull();
			     map.setView(item.getLat(), item.getLon(), 16);
			     add(map);
			     
			     CourseGPSVector gpsVector = courseGPSService.getCourseGPSVector(item.getId(), "Background", "Sand", "Water", "ClubHouse", 
			    		 "Perimeter", "Green", "TeeBox", "Fairway", "Bunker");

			     if (gpsVector != null) {
					String jsonStr;
					try {
						jsonStr = OBJECT_MAPPER.writeValueAsString(gpsVector);
						map.showGPSInfo(jsonStr);
					} catch (JsonProcessingException e) {
						Notification.show("Something went wrong when plotting the GPS information on the map for the course " + item.getName()).addThemeVariants(NotificationVariant.LUMO_ERROR);;
					}
				}
				else {
					Notification.show("Something went wrong when retrieving the GPS information for the course " + item.getName()).addThemeVariants(NotificationVariant.LUMO_ERROR);;
				}
			}
		});
		displayResults(null);
		add(grid);
	}

	private void displayResults(GolfCourseSearchResults result) {		
		if(result == null) {
			grid.setItems(Collections.emptyList());
			return;
		}
		List<DisplayedCourse> displayedCourses = new LinkedList<>();
		result.getResults().stream().forEach(r -> {
			DisplayedCourse displayedCourse = 
					new DisplayedCourse(r.getCourse().getId(), r.getCourse().getName(), 
							r.getCourse().getLocation(), 
							r.getCourse().getCourseUrl() != null ? r.getCourse().getCourseUrl() : "",
							r.getCourse().getAddress(), r.getCourse().getType(), r.getCourse().isHasOverlappingHoles());
			displayedCourses.add(displayedCourse);			
		});
		grid.setItems(displayedCourses);			
	}

	
	class DisplayedCourse {

		private String name;
		private Double lat;
		private Double lon;
		private String id;
		private String url;
		private String address;
		private String courseType;
		private boolean hasOverlappingHoles;
		
		public DisplayedCourse(String id, String name, Location location, String url, Address address, CourseType courseType, boolean hasOverlappingHoles) {
			this.id = id;
			this.name = name;
			this.lat = location.getLat();
			this.lon = location.getLng();
			this.url = url;
			buildAddressString(address);
			buildCourseTypeString(courseType);
			this.hasOverlappingHoles = hasOverlappingHoles;
		}
		
		private void buildCourseTypeString(CourseType courseType) {
			switch(courseType) {
			case EIGHTEEN: 
				this.courseType = "18"; 
				break;
			case FOUR:
				this.courseType = "4"; 
				break;
			case NINE:
				this.courseType = "9"; 
				break;
			case SIX:
				this.courseType = "6"; 
				break;
			case THREE:
				this.courseType = "3"; 
				break;
			case TWELVE:
				this.courseType = "12"; 
				break;
			case OTHER: default:
				this.courseType = "Other"; 
				break;			
			}
		}
		
		private void buildAddressString(Address address) {
			if(address == null) {
				this.address = "";
			}
			else {
				StringBuilder sb = new StringBuilder();
				sb.append(address.getRoute() != null ? address.getRoute() + "," : "");
				sb.append(address.getZipCode() != null ? address.getZipCode() + "," : "");
				sb.append(address.getCity() != null ? address.getCity() + "," : "");
				sb.append(address.getState() != null ? address.getState() + "," : "");
				sb.append(address.getCountry() != null ? address.getCountry() : "");				
				this.address = sb.toString();	
			}			
		}

		public String getId() {
			return id;
		}

		public String getName() {
			return name;
		}
		
		public String getUrl() {
			return url;
		}
		
		public String getAddress() {
			return address;
		}

		public Double getLat() {
			return lat;
		}

		public Double getLon() {
			return lon;
		}
		
		public String getCourseType() {
			return courseType;
		}
		
		public boolean hasOverlappingHoles() {
			return hasOverlappingHoles;
		}
	}



}
