
package com.playertour.backend.vaadin.views.update;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.LinkedList;
import java.util.List;
import java.util.TreeSet;
import java.util.concurrent.Callable;
import java.util.concurrent.Executors;
import java.util.stream.Collectors;

import org.gecko.vaadin.whiteboard.annotations.VaadinComponent;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.annotations.ReferenceScope;
import org.osgi.service.component.annotations.ServiceScope;
import org.osgi.util.promise.PromiseFactory;

import com.playertour.backend.apis.country.CountryConverterService;
import com.playertour.backend.apis.course.ConversionSummaryByCountry;
import com.playertour.backend.apis.course.CourseConverterService;
import com.playertour.backend.apis.igolf.IGolfCountryService;
import com.playertour.backend.apis.igolf.IGolfCountryUpdateService;
import com.playertour.backend.apis.igolf.IGolfCourseUpdateService;
import com.playertour.backend.apis.igolf.IGolfStateService;
import com.playertour.backend.apis.igolf.IGolfStateUpdateService;
import com.playertour.backend.apis.igolf.IGolfUpdateConstants;
import com.playertour.backend.apis.igolf.exceptions.IGolfUpdateException;
import com.playertour.backend.igolf.model.igolf.CountryInfo;
import com.playertour.backend.igolf.model.igolf.StateInfo;
import com.playertour.backend.vaadin.helper.UiUpdateThread;
import com.playertour.backend.vaadin.helper.VaadinViewProgressMonitor;
import com.playertour.backend.vaadin.views.main.MainView;
import com.vaadin.flow.component.UI;
import com.vaadin.flow.component.button.Button;
import com.vaadin.flow.component.checkbox.Checkbox;
import com.vaadin.flow.component.combobox.ComboBox;
import com.vaadin.flow.component.combobox.ComboBox.ItemFilter;
import com.vaadin.flow.component.dependency.CssImport;
import com.vaadin.flow.component.grid.Grid;
import com.vaadin.flow.component.html.Label;
import com.vaadin.flow.component.notification.Notification;
import com.vaadin.flow.component.notification.Notification.Position;
import com.vaadin.flow.component.notification.NotificationVariant;
import com.vaadin.flow.component.orderedlayout.HorizontalLayout;
import com.vaadin.flow.component.orderedlayout.VerticalLayout;
import com.vaadin.flow.component.progressbar.ProgressBar;
import com.vaadin.flow.component.progressbar.ProgressBarVariant;
import com.vaadin.flow.component.radiobutton.RadioButtonGroup;
import com.vaadin.flow.component.radiobutton.RadioGroupVariant;
import com.vaadin.flow.router.PageTitle;
import com.vaadin.flow.router.Route;


/**
 * 
 * @author ilenia
 * @since May 27, 2021
 */

@Route(value = "update", layout = MainView.class)
@PageTitle("Update")
@Component(name = "UpdateView", service = UpdateView.class, scope = ServiceScope.PROTOTYPE)
@VaadinComponent()
@CssImport(themeFor = "vaadin-grid", value = "./themes/wbapp/views/update-view.css")
public class UpdateView extends VerticalLayout {

	/** serialVersionUID */
	private static final long serialVersionUID = 1113047962572067358L;

	@Reference(scope = ReferenceScope.PROTOTYPE_REQUIRED)
	private IGolfCourseUpdateService iGolfCourseUpdateService;

	@Reference(scope = ReferenceScope.PROTOTYPE_REQUIRED)
	private IGolfCountryUpdateService iGolfCountryUpdateService;

	@Reference(scope = ReferenceScope.PROTOTYPE_REQUIRED)
	private IGolfStateUpdateService iGolfStateUpdateService;

	@Reference(target = "(component.name=IGolfDBCountryService)", scope = ReferenceScope.PROTOTYPE_REQUIRED)
	private IGolfCountryService iGolfDBCountryService;
	
	@Reference(target = "(component.name=IGolfDBStateService)", scope = ReferenceScope.PROTOTYPE_REQUIRED)
	private IGolfStateService iGolfDBStateService;

	@Reference(scope = ReferenceScope.PROTOTYPE_REQUIRED)
	private CourseConverterService courseConverterService;

	@Reference(scope = ReferenceScope.PROTOTYPE_REQUIRED)
	private CountryConverterService countryConverterService;

	Comparator<DisplayedSummary> SUMMARY_GRID_COMPARATOR = (ds1, ds2) -> {
		if((ds1.numIGolfCourses - ds1.numPlayertourCourses) > (ds2.numIGolfCourses - ds2.numPlayertourCourses)) {
			return -1;
		}
		else if((ds1.numIGolfCourses - ds1.numPlayertourCourses) == (ds2.numIGolfCourses - ds2.numPlayertourCourses)) {
			return 0;
		}
		return 1;
	};

	private PromiseFactory promiseFactory = new PromiseFactory(Executors.newSingleThreadExecutor());

	List<CountryInfo> countryInfo;
	List<CountryInfo> comboBoxCountryContent;
	List<CountryInfo> comboBoxContinentContent;
	List<StateInfo> comboBoxStateContent;
	
	ItemFilter<StateInfo> stateFilter = (si, filterString) -> si.getStateFull().toLowerCase().contains(filterString.toLowerCase());
	ItemFilter<CountryInfo> countryFilter = (ci, filterString) -> ci.getCountryFull().toLowerCase().contains(filterString.toLowerCase());
	ItemFilter<CountryInfo> continentFilter = (ci, filterString) -> UpdateViewConstants.IGOLF_UI_CONTINENT_NAME_MAP.get(ci.getId_continent()).toLowerCase().contains(filterString.toLowerCase());
	ItemFilter<String> strFilter = (str, filterString) -> str.toLowerCase().contains(filterString.toLowerCase());

	RadioButtonGroup<String> summaryOptionsRadioGroup = new RadioButtonGroup<String>();
	VerticalLayout summaryLayout = new VerticalLayout();
	ComboBox<CountryInfo> summaryCountryComboBox = new ComboBox<CountryInfo>();
	Grid<DisplayedSummary> summaryGrid = new Grid<>();
	
	VerticalLayout updatePlayertourCoursesLayout = new VerticalLayout();
	RadioButtonGroup<String> updatePlayertourCoursesRadioGroup = new RadioButtonGroup<String>();
	Button updatePlayertourCoursesBtn = new Button(UpdateViewConstants.UPDATE_VIEW_BTN_LABEL);
	ComboBox<CountryInfo> updatePlayertourCoursesByCountryComboBox = new ComboBox<CountryInfo>();
	ComboBox<StateInfo> updatePlayertourCoursesByStateComboBox = new ComboBox<StateInfo>();
	
	VerticalLayout updatePlayertourCountriesLayout = new VerticalLayout();
	RadioButtonGroup<String> updatePlayertourCountriesRadioGroup = new RadioButtonGroup<String>();
	Button updatePlayertourCountriesBtn = new Button(UpdateViewConstants.UPDATE_VIEW_BTN_LABEL);
	ComboBox<CountryInfo> updatePlayertourCountriesComboBox = new ComboBox<CountryInfo>();	
	
	Checkbox forceConversionCheckbox = new Checkbox(UpdateViewConstants.UPDATE_VIEW_FORCE_CONVERSION_LABEL);
	
	VerticalLayout updateIGolfCoursesLayout = new VerticalLayout();
	RadioButtonGroup<String>  updateIGolfCoursesRadioGroup = new RadioButtonGroup<String>();
	Button updateIGolfCoursesBtn = new Button(UpdateViewConstants.UPDATE_VIEW_BTN_LABEL);
	ComboBox<CountryInfo> updateIGolfCoursesByCountryComboBox = new ComboBox<>();
	ComboBox<StateInfo> updateIGolfCoursesByStateComboBox = new ComboBox<>();	
	
	VerticalLayout updateIGolfCountriesLayout = new VerticalLayout();
	RadioButtonGroup<String>  updateIGolfCountriesRadioGroup = new RadioButtonGroup<String>();
	Button updateIGolfCountriesBtn = new Button(UpdateViewConstants.UPDATE_VIEW_BTN_LABEL);
	ComboBox<String> updateIGolfCountriesComboBox = new ComboBox<>();
	
	VerticalLayout updateIGolfStatesLayout = new VerticalLayout();
	RadioButtonGroup<String>  updateIGolfStatesRadioGroup = new RadioButtonGroup<String>();
	Button updateIGolfStatesBtn = new Button(UpdateViewConstants.UPDATE_VIEW_BTN_LABEL);
	ComboBox<CountryInfo> updateIGolfStatesComboBox = new ComboBox<>();


	@Activate
	public void renderView() {
		addClassName("update-view"); //this should be the same as the css file name for that view
		setSizeFull();

		promiseFactory.submit(() -> {
			updateCountryInfoList();		
			return true;
		});
		

		//		Options to update iGolf db or playertour db
		VerticalLayout updateOptionsLayout = new VerticalLayout();
		updateOptionsLayout.setHeight("10%");
		RadioButtonGroup<String>  updateOptionsRadioGroup = new RadioButtonGroup<String>();
		updateOptionsRadioGroup.setItems(UpdateViewConstants.UPDATE_OPTION_IGOLF_DB, UpdateViewConstants.UPDATE_OPTION_PLAYERTOUR_DB, UpdateViewConstants.UPDATE_OPTION_SHOW_SUMMARY);
		updateOptionsRadioGroup.addThemeVariants(RadioGroupVariant.LUMO_VERTICAL);	

		updateOptionsLayout.add(updateOptionsRadioGroup);
		add(updateOptionsLayout);

		//		Options to update iGolf db
		HorizontalLayout updateIGolfLayout = new HorizontalLayout();
		updateIGolfLayout.setWidthFull();
		updateIGolfLayout.setHeight("60%");
		updateIGolfLayout.setVisible(false);

		setupIGolfCoursesLayout();
		setupIGolfCountriesLayout();
		setupIGolfStatesLayout();
		
		updateIGolfLayout.add(updateIGolfCoursesLayout, updateIGolfCountriesLayout, updateIGolfStatesLayout);
		add(updateIGolfLayout);

		//      options to update playertour db
		HorizontalLayout updatePlayertourLayout = new HorizontalLayout();
		updatePlayertourLayout.setWidthFull();
		updatePlayertourLayout.setHeight("40%");
		updatePlayertourLayout.setVisible(false);	
		
		VerticalLayout forceConversionLayout = new VerticalLayout();
		forceConversionLayout.setVisible(false);		
		
		forceConversionLayout.add(forceConversionCheckbox);
		forceConversionCheckbox.addValueChangeListener(evt -> {
			if(evt.getValue() == true) {
				displayNotification("By forcing the conversion, courses which had been previously converted and validate will loose the validation results. "
						+ "Please, take care of triggering another validation!", Position.MIDDLE, NotificationVariant.LUMO_CONTRAST);
			}
		});

		//		---------------------------------------------------
		setupPlayertourCoursesLayout();		
		setupPlayertourCountriesLayout();		
		setupSummaryLayout();
//		---------------------------------------

		updateOptionsRadioGroup.addValueChangeListener(evt -> {
			if(evt.getValue() == null) {
				return;
			}
			switch(evt.getValue()) {
			case UpdateViewConstants.UPDATE_OPTION_IGOLF_DB:
				displayNotification("This operation is going to connect to iGolf API, using the limited amount of credits we have. Do it with caution!", Position.MIDDLE, NotificationVariant.LUMO_CONTRAST);
				
				updateIGolfLayout.setVisible(true);
				updatePlayertourLayout.setVisible(false);
				forceConversionLayout.setVisible(false);

				resetPlayertourUpdateLayout();				
				resetSummaryLayout();
				return;
			case UpdateViewConstants.UPDATE_OPTION_PLAYERTOUR_DB: default:
				updateIGolfLayout.setVisible(false);
				updatePlayertourLayout.setVisible(true);
				forceConversionLayout.setVisible(true);

				resetIGolfUpdateLayout();
				resetSummaryLayout();
				return;
				
			case UpdateViewConstants.UPDATE_OPTION_SHOW_SUMMARY:
				
				updateIGolfLayout.setVisible(false);
				updatePlayertourLayout.setVisible(false);
				forceConversionLayout.setVisible(false);
				
				resetPlayertourUpdateLayout();				
				resetIGolfUpdateLayout();
				summaryLayout.setVisible(true);				
				return;
			}
		});
		updatePlayertourLayout.add(updatePlayertourCoursesLayout, updatePlayertourCountriesLayout);
		add(forceConversionLayout, updatePlayertourLayout, summaryLayout);		
	}

	private void resetIGolfUpdateLayout() {
		updateIGolfCoursesBtn.setEnabled(false);
		updateIGolfCountriesBtn.setEnabled(false);
		updateIGolfStatesBtn.setEnabled(false);

		updateIGolfCoursesRadioGroup.setValue(null);
		updateIGolfCountriesRadioGroup.setValue(null);
		updateIGolfStatesRadioGroup.setValue(null);

		updateIGolfCoursesByCountryComboBox.setValue(null);
		updateIGolfCoursesByStateComboBox.setValue(null);
		updateIGolfCountriesComboBox.setValue(null);
		updateIGolfStatesComboBox.setValue(null);
		updateIGolfCoursesByCountryComboBox.setVisible(false);
		updateIGolfCoursesByStateComboBox.setVisible(false);
		updateIGolfCountriesComboBox.setVisible(false);
		updateIGolfStatesComboBox.setVisible(false);
	}
	
	private void setupIGolfCountriesLayout() {
		
		updateIGolfCountriesLayout.setWidth("33%");
		updateIGolfCountriesRadioGroup.setLabel("Update IGolf Country List");
		updateIGolfCountriesRadioGroup.setItems(UpdateViewConstants.UPDATE_OPTION_BY_CONTINENT, UpdateViewConstants.UPDATE_OPTION_ALL);
		updateIGolfCountriesRadioGroup.addThemeVariants(RadioGroupVariant.LUMO_VERTICAL);

		updateIGolfCountriesBtn.setEnabled(false);

		updateIGolfCountriesComboBox.setHeight("10%");
		updateIGolfCountriesComboBox.setWidth("80%");
		updateIGolfCountriesComboBox.setVisible(false);
		updateIGolfCountriesRadioGroup.addValueChangeListener(evt -> {
			if(evt.getValue() == null) {
				return;
			}
			switch(evt.getValue()) {
			case UpdateViewConstants.UPDATE_OPTION_BY_CONTINENT:
				updateIGolfCountriesBtn.setEnabled(false);
				updateIGolfCountriesComboBox.setVisible(true);
				updateIGolfCountriesComboBox.setPlaceholder(UpdateViewConstants.UPDATE_VIEW_SELECT_CONTINENT_LABEL);
				updateIGolfCountriesComboBox.setItems(strFilter, UpdateViewConstants.IGOLF_UI_CONTINENT_NAME_MAP.values());
				break;
			case UpdateViewConstants.UPDATE_OPTION_ALL: default:
				updateIGolfCountriesComboBox.setVisible(false);
				updateIGolfCountriesComboBox.setItems(Collections.emptyList());
				updateIGolfCountriesBtn.setEnabled(true);
				break;
			}			
		});

		updateIGolfCountriesComboBox.addValueChangeListener(evt -> {
			if(UpdateViewConstants.IGOLF_UI_CONTINENT_NAME_MAP.values().contains(evt.getValue())) {
				updateIGolfCountriesBtn.setEnabled(true);
			} else {
				updateIGolfCountriesBtn.setEnabled(false);
			}
		});

		HorizontalLayout iGolfCountriesBarLayout = new HorizontalLayout();
		ProgressBar iGolfCountriesProgressBar = new ProgressBar(0., 100.);
		Label iGolfCountriesBarLabel = new Label();		
		setupProgressBarLayout(iGolfCountriesBarLayout, iGolfCountriesProgressBar, iGolfCountriesBarLabel);

		VaadinViewProgressMonitor iGolfCountriesProgressMonitor = 
				new VaadinViewProgressMonitor(iGolfCountriesProgressBar, iGolfCountriesBarLabel, UI.getCurrent());
		iGolfCountriesProgressMonitor.setMax(100.);

		updateIGolfCountriesBtn.addClickListener(evt -> {
			String radioBtnValue = updateIGolfCountriesRadioGroup.getValue();
			String comboBoxValue = updateIGolfCountriesComboBox.getValue();
			iGolfCountriesBarLayout.setVisible(true);
			Callable<Void> countryUpdateCall = null;
			Callable<Void> endCall = () -> {updateCountryInfoList(); return null;};
			switch(radioBtnValue) {
			case UpdateViewConstants.UPDATE_OPTION_ALL:
				countryUpdateCall = () -> {
						iGolfCountryUpdateService.grabAndSaveIGolfCountriesByContinentId(IGolfUpdateConstants.UPDATE_COUNTRIES_ALL, iGolfCountriesProgressMonitor);
						return null;					
				};
				break;
			case UpdateViewConstants.UPDATE_OPTION_BY_CONTINENT:
				countryUpdateCall = () -> {
						iGolfCountryUpdateService.grabAndSaveIGolfCountriesByContinentId(UpdateViewConstants.UI_IGOLF_CONTINENT_NAME_MAP.get(comboBoxValue), iGolfCountriesProgressMonitor);
						return null;					
				};
				break;
			}

			try {
				launchThread(countryUpdateCall, endCall, UI.getCurrent(), iGolfCountriesProgressBar, iGolfCountriesProgressMonitor, updateIGolfCountriesBtn, iGolfCountriesBarLabel);
			} catch(Exception e) {
				displayNotification("Error occurred when updating Countries " + radioBtnValue, Position.MIDDLE, NotificationVariant.LUMO_ERROR);
				e.printStackTrace();
			}			
		});
		updateIGolfCountriesLayout.add(updateIGolfCountriesRadioGroup, updateIGolfCountriesComboBox, updateIGolfCountriesBtn, iGolfCountriesBarLayout);
	}


	private void setupIGolfStatesLayout() {
		updateIGolfStatesLayout.setWidth("33%");
		updateIGolfStatesRadioGroup.setLabel("Update IGolf State List");
		updateIGolfStatesRadioGroup.setItems(UpdateViewConstants.UPDATE_OPTION_BY_COUNTRY, UpdateViewConstants.UPDATE_OPTION_ALL);
		updateIGolfStatesRadioGroup.addThemeVariants(RadioGroupVariant.LUMO_VERTICAL);
		updateIGolfStatesRadioGroup.setHelperText("To collect data for countries not in the list, update country list first!");

		updateIGolfStatesBtn.setEnabled(false);

		updateIGolfStatesComboBox.setHeight("10%");
		updateIGolfStatesComboBox.setWidth("80%");
		updateIGolfStatesComboBox.setVisible(false);
		updateIGolfStatesRadioGroup.addValueChangeListener(evt -> {
			if(evt.getValue() == null) {
				return;
			}
			switch(evt.getValue()) {
			case UpdateViewConstants.UPDATE_OPTION_BY_COUNTRY:
				updateIGolfStatesBtn.setEnabled(false);
				updateIGolfStatesComboBox.setVisible(true);
				updateIGolfStatesComboBox.setPlaceholder(UpdateViewConstants.UPDATE_VIEW_SELECT_COUNTRY_LABEL);
				updateIGolfStatesComboBox.setItems(countryFilter, comboBoxCountryContent);
				updateIGolfStatesComboBox.setItemLabelGenerator(ci -> ci.getCountryFull());
				break;
			case UpdateViewConstants.UPDATE_OPTION_ALL: default:
				updateIGolfStatesComboBox.setVisible(false);
				updateIGolfStatesComboBox.setItems(Collections.emptyList());
				updateIGolfStatesBtn.setEnabled(true);
				break;
			}			
		});

		updateIGolfStatesComboBox.addValueChangeListener(evt -> {
			if(comboBoxCountryContent.contains(evt.getValue())) {
				updateIGolfStatesBtn.setEnabled(true);
			} else {
				updateIGolfStatesBtn.setEnabled(false);
			}
		});

		HorizontalLayout iGolfStatesBarLayout = new HorizontalLayout();	
		ProgressBar iGolfStatesProgressBar = new ProgressBar(0., 100.);	
		Label iGolfStatesBarLabel = new Label();
	
		setupProgressBarLayout(iGolfStatesBarLayout, iGolfStatesProgressBar, iGolfStatesBarLabel);

		VaadinViewProgressMonitor iGolfStatesProgressMonitor = 
				new VaadinViewProgressMonitor(iGolfStatesProgressBar, iGolfStatesBarLabel, UI.getCurrent());
		iGolfStatesProgressMonitor.setMax(100.);

		updateIGolfStatesBtn.addClickListener(evt -> {
			String radioBtnValue = updateIGolfStatesRadioGroup.getValue();
			CountryInfo comboBoxValue = updateIGolfStatesComboBox.getValue();
			iGolfStatesBarLayout.setVisible(true);
			Callable<Void> stateUpdateCall = null;
			switch(radioBtnValue) {
			case UpdateViewConstants.UPDATE_OPTION_ALL:
				stateUpdateCall = () -> {
						iGolfStateUpdateService
						.grabAndSaveIGolfStatesByCountryIds(countryInfo.stream().map(ci->ci.getId_country().toString()).collect(Collectors.toList()), iGolfStatesProgressMonitor);
						return null;						
				};
				break;
			case UpdateViewConstants.UPDATE_OPTION_BY_COUNTRY:
				stateUpdateCall = () -> {
						iGolfStateUpdateService.grabAndSaveIGolfStatesByCountryId(comboBoxValue.getId_country().toString(), iGolfStatesProgressMonitor);
						return null;						
				};
				break;
			}

			try {
				launchThread(stateUpdateCall, () -> null, UI.getCurrent(), iGolfStatesProgressBar, iGolfStatesProgressMonitor, updateIGolfStatesBtn, iGolfStatesBarLabel);
			} catch(Exception e) {
				displayNotification("Error occurred when updating States " + radioBtnValue, Position.MIDDLE, NotificationVariant.LUMO_ERROR);
				e.printStackTrace();
			}

		});
		updateIGolfStatesLayout.add(updateIGolfStatesRadioGroup, updateIGolfStatesComboBox, updateIGolfStatesBtn, iGolfStatesBarLayout);
	}


	private void setupIGolfCoursesLayout() {
		updateIGolfCoursesLayout.setWidth("33%");
		updateIGolfCoursesLayout.setHeightFull();
		updateIGolfCoursesRadioGroup.setLabel("Update IGolf Courses");
		updateIGolfCoursesRadioGroup.setHelperText("To collect data for continents/countries not in the list, update country list first!");
		updateIGolfCoursesRadioGroup.setItems(UpdateViewConstants.UPDATE_OPTION_BY_CONTINENT, UpdateViewConstants.UPDATE_OPTION_BY_COUNTRY, 
				UpdateViewConstants.UPDATE_OPTION_BY_STATE, UpdateViewConstants.UPDATE_OPTION_ALL);
		updateIGolfCoursesRadioGroup.addThemeVariants(RadioGroupVariant.LUMO_VERTICAL);	

		updateIGolfCoursesBtn.setEnabled(false);

		updateIGolfCoursesByCountryComboBox.setHeight("10%");
		updateIGolfCoursesByCountryComboBox.setWidth("80%");
		updateIGolfCoursesByCountryComboBox.setVisible(false);
		
		updateIGolfCoursesByStateComboBox.setHeight("10%");
		updateIGolfCoursesByStateComboBox.setWidth("80%");
		updateIGolfCoursesByStateComboBox.setVisible(false);
		
		updateIGolfCoursesRadioGroup.addValueChangeListener(evt -> {
			if(evt.getValue() == null) {
				return;
			}
			switch(evt.getValue()) {
			case UpdateViewConstants.UPDATE_OPTION_BY_CONTINENT:
				updateIGolfCoursesByStateComboBox.setVisible(false);
				updateIGolfCoursesByStateComboBox.setItems(Collections.emptyList());
				updateIGolfCoursesBtn.setEnabled(false);
				updateIGolfCoursesByCountryComboBox.setVisible(true);
				updateIGolfCoursesByCountryComboBox.setPlaceholder(UpdateViewConstants.UPDATE_VIEW_SELECT_CONTINENT_LABEL);
				updateIGolfCoursesByCountryComboBox.setItems(continentFilter, comboBoxContinentContent);
				updateIGolfCoursesByCountryComboBox.setItemLabelGenerator(ci -> UpdateViewConstants.IGOLF_UI_CONTINENT_NAME_MAP.get(ci.getId_continent()));
				return;
			case UpdateViewConstants.UPDATE_OPTION_BY_COUNTRY: 
				updateIGolfCoursesByStateComboBox.setVisible(false);
				updateIGolfCoursesByStateComboBox.setItems(Collections.emptyList());
				updateIGolfCoursesBtn.setEnabled(false);
				updateIGolfCoursesByCountryComboBox.setVisible(true);
				updateIGolfCoursesByCountryComboBox.setPlaceholder(UpdateViewConstants.UPDATE_VIEW_SELECT_COUNTRY_LABEL);
				updateIGolfCoursesByCountryComboBox.setItems(countryFilter, comboBoxCountryContent);
				updateIGolfCoursesByCountryComboBox.setItemLabelGenerator(ci -> ci.getCountryFull());
				return;
			case UpdateViewConstants.UPDATE_OPTION_BY_STATE: 
				updateIGolfCoursesBtn.setEnabled(false);
				updateIGolfCoursesByCountryComboBox.setVisible(true);
				updateIGolfCoursesByCountryComboBox.setPlaceholder(UpdateViewConstants.UPDATE_VIEW_SELECT_COUNTRY_LABEL);
				updateIGolfCoursesByCountryComboBox.setItems(countryFilter, comboBoxCountryContent);
				updateIGolfCoursesByCountryComboBox.setItemLabelGenerator(ci -> ci.getCountryFull());
				return;
			case UpdateViewConstants.UPDATE_OPTION_ALL: default:
				updateIGolfCoursesByStateComboBox.setVisible(false);
				updateIGolfCoursesByStateComboBox.setItems(Collections.emptyList());
				updateIGolfCoursesByCountryComboBox.setVisible(false);
				updateIGolfCoursesByCountryComboBox.setItems(Collections.emptyList());
				updateIGolfCoursesBtn.setEnabled(true);
			}			
		});

		updateIGolfCoursesByCountryComboBox.addValueChangeListener(evt -> {
			if(comboBoxContinentContent.contains(evt.getValue()) || comboBoxCountryContent.contains(evt.getValue())) {
				if(UpdateViewConstants.UPDATE_OPTION_BY_STATE.equals(updateIGolfCoursesRadioGroup.getValue())) {
					updateStateInfoList(evt.getValue().getId_country().toString());
					updateIGolfCoursesByStateComboBox.setVisible(true);
					updateIGolfCoursesByStateComboBox.setPlaceholder(UpdateViewConstants.UPDATE_VIEW_SELECT_STATE_LABEL);
					updateIGolfCoursesByStateComboBox.setItems(stateFilter, comboBoxStateContent);
					updateIGolfCoursesByStateComboBox.setItemLabelGenerator(si -> si.getStateFull());
				}
				else {
					updateIGolfCoursesBtn.setEnabled(true);
				}
			} else {
				updateIGolfCoursesBtn.setEnabled(false);
			}
		});
		
		updateIGolfCoursesByStateComboBox.addValueChangeListener(evt -> {
			if(comboBoxStateContent.contains(evt.getValue())) {
				updateIGolfCoursesBtn.setEnabled(true);
			} else {
				updateIGolfCoursesBtn.setEnabled(false);
			}
		});

		HorizontalLayout iGolfCoursesBarLayout = new HorizontalLayout();
		ProgressBar iGolfCoursesProgressBar = new ProgressBar(0., 100.);
		Label iGolfCoursesBarLabel = new Label();
		setupProgressBarLayout(iGolfCoursesBarLayout, iGolfCoursesProgressBar, iGolfCoursesBarLabel);

		VaadinViewProgressMonitor iGolfCoursesProgressMonitor = 
				new VaadinViewProgressMonitor(iGolfCoursesProgressBar, iGolfCoursesBarLabel, UI.getCurrent());

		updateIGolfCoursesBtn.addClickListener(evt -> {
			String radioBtnValue = updateIGolfCoursesRadioGroup.getValue();
			CountryInfo contryInfo = updateIGolfCoursesByCountryComboBox.getValue();
			iGolfCoursesBarLayout.setVisible(true);
			Callable<Void> courseUpdateCall = null;
			
			switch(radioBtnValue) {
			case UpdateViewConstants.UPDATE_OPTION_ALL:
				courseUpdateCall = () -> {
					iGolfCourseUpdateService.grabAndSaveIGolfCoursesByContinentId(IGolfUpdateConstants.UPDATE_COUNTRIES_ALL, iGolfCoursesProgressMonitor);
					return null;
				};
				break;
			case UpdateViewConstants.UPDATE_OPTION_BY_CONTINENT:
				courseUpdateCall = () -> {
						iGolfCourseUpdateService.grabAndSaveIGolfCoursesByContinentId(UpdateViewConstants.IGOLF_UI_CONTINENT_NAME_MAP.get(contryInfo.getId_continent()), iGolfCoursesProgressMonitor);
						return null;				
				};
				break;
			case UpdateViewConstants.UPDATE_OPTION_BY_COUNTRY:
				courseUpdateCall = () -> {
						iGolfCourseUpdateService.grabAndSaveIGolfCoursesByCountryId(contryInfo.getId_country().toString(), iGolfCoursesProgressMonitor);
						return null;				
				};
				break;
			case UpdateViewConstants.UPDATE_OPTION_BY_STATE:
				StateInfo stateInfo = updateIGolfCoursesByStateComboBox.getValue();
				courseUpdateCall = () -> {
						iGolfCourseUpdateService.grabAndSaveIGolfCoursesByStateId(stateInfo.getId_state().toString(), iGolfCoursesProgressMonitor);
						return null;				
				};
				break;
			}
			try {				
				launchThread(courseUpdateCall, () -> null, UI.getCurrent(), iGolfCoursesProgressBar, iGolfCoursesProgressMonitor, updateIGolfCoursesBtn, iGolfCoursesBarLabel);
			} catch(Exception e) {
				displayNotification("Error occurred when updating Courses " + radioBtnValue, Position.MIDDLE, NotificationVariant.LUMO_ERROR);
				e.printStackTrace();
			}
		});

		updateIGolfCoursesLayout.add(updateIGolfCoursesRadioGroup, updateIGolfCoursesByCountryComboBox, updateIGolfCoursesByStateComboBox, updateIGolfCoursesBtn, iGolfCoursesBarLayout);		
	}

	private void launchThread(Callable<Void> updateCall, Callable<Void> endCall, UI currentUI,
			ProgressBar progressBar, VaadinViewProgressMonitor progressMonitor,
			Button progressBarBtn, Label progressBarLabel) {
		
		currentUI.setPollInterval(500);				
		UiUpdateThread updateThread = 
				new UiUpdateThread(updateCall, currentUI, progressBar, progressMonitor, 
						progressBarBtn, progressBarLabel);
		updateThread.setEndCall(endCall);
		updateThread.start();
	}


	private void resetPlayertourUpdateLayout() {
		updatePlayertourCoursesBtn.setEnabled(false);
		updatePlayertourCountriesBtn.setEnabled(false);

		updatePlayertourCoursesRadioGroup.setValue(null);
		updatePlayertourCountriesRadioGroup.setValue(null);

		updatePlayertourCoursesByCountryComboBox.setValue(null);
		updatePlayertourCountriesComboBox.setValue(null);		
		updatePlayertourCoursesByCountryComboBox.setVisible(false);
		updatePlayertourCountriesComboBox.setVisible(false);	
	}


	private void setupPlayertourCoursesLayout() {
		
		updatePlayertourCoursesLayout.setSizeFull();
		
		updatePlayertourCoursesRadioGroup.setLabel("Update Playertour Courses:");
		updatePlayertourCoursesRadioGroup.setItems(UpdateViewConstants.UPDATE_OPTION_BY_CONTINENT, UpdateViewConstants.UPDATE_OPTION_BY_COUNTRY, 
				UpdateViewConstants.UPDATE_OPTION_BY_STATE, UpdateViewConstants.UPDATE_OPTION_ALL);
		updatePlayertourCoursesRadioGroup.addThemeVariants(RadioGroupVariant.LUMO_VERTICAL);	
		updatePlayertourCoursesRadioGroup.setHelperText("To collect data for countries not in the list, update iGolf DB first!");

		updatePlayertourCoursesBtn.setEnabled(false);

		updatePlayertourCoursesByCountryComboBox.setHeightFull();
		updatePlayertourCoursesByCountryComboBox.setWidth("40%");
		updatePlayertourCoursesByCountryComboBox.setVisible(false);
		
		updatePlayertourCoursesByStateComboBox.setHeightFull();
		updatePlayertourCoursesByStateComboBox.setWidth("40%");
		updatePlayertourCoursesByStateComboBox.setVisible(false);

		updatePlayertourCoursesRadioGroup.addValueChangeListener(evt -> {
			if(evt.getValue() == null) {
				return;
			}
			switch(evt.getValue()) {
			case UpdateViewConstants.UPDATE_OPTION_BY_CONTINENT:
				updatePlayertourCoursesByStateComboBox.setVisible(false);
				updatePlayertourCoursesBtn.setEnabled(false);
				updatePlayertourCoursesByCountryComboBox.setVisible(true);
				updatePlayertourCoursesByCountryComboBox.setPlaceholder(UpdateViewConstants.UPDATE_VIEW_SELECT_CONTINENT_LABEL);
				updatePlayertourCoursesByCountryComboBox.setItems(continentFilter, comboBoxContinentContent);
				updatePlayertourCoursesByCountryComboBox.setItemLabelGenerator(ci -> UpdateViewConstants.IGOLF_UI_CONTINENT_NAME_MAP.get(ci.getId_continent()));
				return;
			case UpdateViewConstants.UPDATE_OPTION_BY_COUNTRY:
				updatePlayertourCoursesByStateComboBox.setVisible(false);
				updatePlayertourCoursesBtn.setEnabled(false);
				updatePlayertourCoursesByCountryComboBox.setVisible(true);
				updatePlayertourCoursesByCountryComboBox.setPlaceholder(UpdateViewConstants.UPDATE_VIEW_SELECT_COUNTRY_LABEL);
				updatePlayertourCoursesByCountryComboBox.setItems(countryFilter, comboBoxCountryContent);
				updatePlayertourCoursesByCountryComboBox.setItemLabelGenerator(ci -> ci.getCountryFull());
				return;
			case UpdateViewConstants.UPDATE_OPTION_BY_STATE:
				updatePlayertourCoursesBtn.setEnabled(false);
				updatePlayertourCoursesByCountryComboBox.setVisible(true);
				updatePlayertourCoursesByCountryComboBox.setPlaceholder(UpdateViewConstants.UPDATE_VIEW_SELECT_COUNTRY_LABEL);
				updatePlayertourCoursesByCountryComboBox.setItems(countryFilter, comboBoxCountryContent);
				updatePlayertourCoursesByCountryComboBox.setItemLabelGenerator(ci -> ci.getCountryFull());
				return;
			case UpdateViewConstants.UPDATE_OPTION_ALL: default:
				updatePlayertourCoursesByStateComboBox.setVisible(false);
				updatePlayertourCoursesByCountryComboBox.setVisible(false);
				updatePlayertourCoursesBtn.setEnabled(true);
			}
		});		

		updatePlayertourCoursesByCountryComboBox.addValueChangeListener(evt -> {
			if(comboBoxCountryContent.contains(evt.getValue()) || comboBoxContinentContent.contains(evt.getValue())) {
				if(UpdateViewConstants.UPDATE_OPTION_BY_STATE.equals(updatePlayertourCoursesRadioGroup.getValue())) {
					updateStateInfoList(evt.getValue().getId_country().toString());
					updatePlayertourCoursesByStateComboBox.setVisible(true);
					updatePlayertourCoursesByStateComboBox.setPlaceholder(UpdateViewConstants.UPDATE_VIEW_SELECT_STATE_LABEL);
					updatePlayertourCoursesByStateComboBox.setItems(stateFilter, comboBoxStateContent);
					updatePlayertourCoursesByStateComboBox.setItemLabelGenerator(si -> si.getStateFull());
				}
				else {
					updatePlayertourCoursesBtn.setEnabled(true);
				}				
			} else {
				updatePlayertourCoursesBtn.setEnabled(false);
			}
		});
		
		updatePlayertourCoursesByStateComboBox.addValueChangeListener(evt -> {
			if(comboBoxStateContent.contains(evt.getValue())) {
				updatePlayertourCoursesBtn.setEnabled(true);
			} else {
				updatePlayertourCoursesBtn.setEnabled(false);
			}
		});

		HorizontalLayout playertourCoursesBarLayout = new HorizontalLayout();
		ProgressBar playertourCoursesProgressBar = new ProgressBar(0., 100.);
		Label playertourCoursesBarLabel = new Label();
		setupProgressBarLayout(playertourCoursesBarLayout, playertourCoursesProgressBar, playertourCoursesBarLabel);

		VaadinViewProgressMonitor playertourCoursesUpdateProgressMonitor = 
				new VaadinViewProgressMonitor(playertourCoursesProgressBar, playertourCoursesBarLabel, UI.getCurrent());

		updatePlayertourCoursesBtn.addClickListener(evt -> {
			playertourCoursesBarLayout.setVisible(true);
			CountryInfo comboBoxValue = updatePlayertourCoursesByCountryComboBox.getValue();
			String radioBtnValue = updatePlayertourCoursesRadioGroup.getValue();
			Callable<Void> playertourUpdateCall = null;
			courseConverterService.setForceConversion(forceConversionCheckbox.getValue());

			switch(radioBtnValue) {
			case UpdateViewConstants.UPDATE_OPTION_ALL:
				playertourUpdateCall = () -> {
						courseConverterService.convertAndSaveAllGolfCourses(playertourCoursesUpdateProgressMonitor);
						return null;				
				};
				break;
			case UpdateViewConstants.UPDATE_OPTION_BY_CONTINENT:
				playertourUpdateCall = () -> {
						courseConverterService.convertAndSaveGolfCoursesByContinent(comboBoxValue.getId_continent(), playertourCoursesUpdateProgressMonitor);
						return null;	
				};
				break;
			case UpdateViewConstants.UPDATE_OPTION_BY_COUNTRY:
				playertourUpdateCall = () -> {
						courseConverterService.convertAndSaveGolfCoursesByCountry(comboBoxValue.getId_country().toString(), playertourCoursesUpdateProgressMonitor);
						return null;		
				};
				break;
			case UpdateViewConstants.UPDATE_OPTION_BY_STATE:
				StateInfo stateInfo = updatePlayertourCoursesByStateComboBox.getValue();
				playertourUpdateCall = () -> {
						courseConverterService.convertAndSaveGolfCoursesByState(stateInfo.getId_state().toString(), playertourCoursesUpdateProgressMonitor);
						return null;				
				};
				break;
			}

			try {
				launchThread(playertourUpdateCall, () -> null, UI.getCurrent(), playertourCoursesProgressBar, playertourCoursesUpdateProgressMonitor, updatePlayertourCoursesBtn, playertourCoursesBarLabel);

			} catch(Exception e) {
				displayNotification("Error occurred when updating Courses " + radioBtnValue, Position.MIDDLE, NotificationVariant.LUMO_ERROR);
				e.printStackTrace();
			}
		});
		updatePlayertourCoursesLayout.add(updatePlayertourCoursesRadioGroup, updatePlayertourCoursesByCountryComboBox, updatePlayertourCoursesByStateComboBox, updatePlayertourCoursesBtn, playertourCoursesBarLayout);
	}



	private void setupPlayertourCountriesLayout() {
		updatePlayertourCountriesLayout.setSizeFull();

		updatePlayertourCountriesRadioGroup.setLabel("Update Playertour Countries:");
		updatePlayertourCountriesRadioGroup.setItems(UpdateViewConstants.UPDATE_OPTION_BY_CONTINENT, UpdateViewConstants.UPDATE_OPTION_ALL);
		updatePlayertourCountriesRadioGroup.addThemeVariants(RadioGroupVariant.LUMO_VERTICAL);	
		updatePlayertourCountriesRadioGroup.setHelperText("To collect data not in the list, update iGolf DB first!");

		updatePlayertourCountriesBtn.setEnabled(false);

		updatePlayertourCountriesComboBox.setHeightFull();
		updatePlayertourCountriesComboBox.setWidth("40%");
		updatePlayertourCountriesComboBox.setVisible(false);

		updatePlayertourCountriesComboBox.addValueChangeListener(evt -> {
			if(comboBoxContinentContent.contains(evt.getValue())) {
				updatePlayertourCountriesBtn.setEnabled(true);
			} else {
				updatePlayertourCountriesBtn.setEnabled(false);
			}
		});

		updatePlayertourCountriesRadioGroup.addValueChangeListener(evt -> {
			if(evt.getValue() == null) {
				return;
			}
			switch(evt.getValue()) {
			case UpdateViewConstants.UPDATE_OPTION_BY_CONTINENT:
				updatePlayertourCountriesBtn.setEnabled(false);
				updatePlayertourCountriesComboBox.setVisible(true);
				updatePlayertourCountriesComboBox.setPlaceholder(UpdateViewConstants.UPDATE_VIEW_SELECT_CONTINENT_LABEL);
				updatePlayertourCountriesComboBox.setItems(continentFilter, comboBoxContinentContent);
				updatePlayertourCountriesComboBox.setItemLabelGenerator(ci -> UpdateViewConstants.IGOLF_UI_CONTINENT_NAME_MAP.get(ci.getId_continent()));
				return;
			case UpdateViewConstants.UPDATE_OPTION_ALL: default:
				updatePlayertourCountriesComboBox.setVisible(false);
				updatePlayertourCountriesBtn.setEnabled(true);
			}
		});		

		HorizontalLayout playertourCountriesBarLayout = new HorizontalLayout();
		ProgressBar playertourCountriesProgressBar = new ProgressBar(0., 100.);
		Label playertourCountriesBarLabel = new Label();
		setupProgressBarLayout(playertourCountriesBarLayout, playertourCountriesProgressBar, playertourCountriesBarLabel);
		VaadinViewProgressMonitor playertourCountriesUpdateProgressMonitor = 
				new VaadinViewProgressMonitor(playertourCountriesProgressBar, playertourCountriesBarLabel, UI.getCurrent());

		updatePlayertourCountriesBtn.addClickListener(evt -> {
			playertourCountriesBarLayout.setVisible(true);
			CountryInfo comboBoxValue = updatePlayertourCountriesComboBox.getValue();
			String radioBtnValue = updatePlayertourCountriesRadioGroup.getValue();
			Callable<Void> playertourUpdateCall = null;
			countryConverterService.setForceConversion(forceConversionCheckbox.getValue());
			switch(radioBtnValue) {
			case UpdateViewConstants.UPDATE_OPTION_ALL:
				playertourUpdateCall = () -> {
						countryConverterService.convertAndSaveAllCountries(playertourCountriesUpdateProgressMonitor);
						return null;		
				};
				break;
			case UpdateViewConstants.UPDATE_OPTION_BY_CONTINENT:
				playertourUpdateCall = () -> {
						countryConverterService.convertAndSaveCountriesByContinent(comboBoxValue.getId_continent(), playertourCountriesUpdateProgressMonitor);
						return null;		
				};
				break;
			}

			try {
				launchThread(playertourUpdateCall, () -> null, UI.getCurrent(), playertourCountriesProgressBar, playertourCountriesUpdateProgressMonitor, updatePlayertourCountriesBtn, playertourCountriesBarLabel);
			} catch(Exception e) {
				displayNotification("Error occurred when updating Countries " + radioBtnValue, Position.MIDDLE, NotificationVariant.LUMO_ERROR);
				e.printStackTrace();
			}
		});	
		updatePlayertourCountriesLayout.add(updatePlayertourCountriesRadioGroup, updatePlayertourCountriesComboBox, updatePlayertourCountriesBtn, playertourCountriesBarLayout);
	}

	private void setupProgressBarLayout(HorizontalLayout progressBarLayout, ProgressBar progressBar, Label progressBarLabel) {
		
		progressBarLayout.setSizeFull();
		progressBarLayout.setVisible(false);

		progressBar.setIndeterminate(true);
		progressBar.setVisible(true);
		progressBar.setWidthFull();
		progressBar.setHeight("40%");
		progressBar.addThemeVariants(ProgressBarVariant.LUMO_CONTRAST);

		progressBarLabel.setSizeFull();
		progressBarLabel.setVisible(true);
		progressBarLayout.add(progressBar, progressBarLabel);			
	}



	private void resetSummaryLayout() {
		summaryOptionsRadioGroup.setValue(null);
		summaryLayout.setVisible(false);
		summaryGrid.setItems(Collections.emptyList());
		summaryGrid.setVisible(false);
		summaryCountryComboBox.setVisible(false);		
	}

	
	private void setupSummaryLayout() {
		summaryLayout.setSizeFull();
		summaryLayout.setVisible(false);
		
		
		summaryOptionsRadioGroup.setItems(UpdateViewConstants.UPDATE_OPTION_ALL, UpdateViewConstants.UPDATE_OPTION_BY_COUNTRY);
		summaryOptionsRadioGroup.addThemeVariants(RadioGroupVariant.LUMO_VERTICAL);	
		summaryOptionsRadioGroup.setHeight("10%");
		
		summaryCountryComboBox.setHeight("10%");
		summaryCountryComboBox.setVisible(false);
		summaryCountryComboBox.setItemLabelGenerator(ci -> ci.getCountryFull());
		summaryCountryComboBox.setLabel(UpdateViewConstants.UPDATE_VIEW_SELECT_COUNTRY_LABEL);
		
		summaryGrid.setSizeFull();
		summaryGrid.setVisible(false);
		summaryGrid.addColumn(DisplayedSummary::getCountryName).setHeader("Country").setKey(UpdateViewConstants.UPDATE_VIEW_GRID_COUNTRY_COL_KEY).setVisible(false);
		summaryGrid.addColumn(DisplayedSummary::getStateName).setHeader("State").setKey(UpdateViewConstants.UPDATE_VIEW_GRID_STATE_COL_KEY).setVisible(false);
		summaryGrid.addColumn(DisplayedSummary::getNumIGolfCourses).setHeader("# Courses in igolf_raw db");
		summaryGrid.addColumn(DisplayedSummary::getNumPlayertourCourses).setHeader("# Courses in playertour db");
		summaryGrid.setClassNameGenerator(ds -> ds.getNumIGolfCourses() > ds.getNumPlayertourCourses() ? "to-be-converted" : null);
		
		
		summaryOptionsRadioGroup.addValueChangeListener(evt -> {	
			if(evt.getValue() == null) {
				return;
			}
			summaryGrid.setVisible(false);
			summaryGrid.setItems(Collections.emptyList());
			summaryCountryComboBox.setVisible(false);
			switch(evt.getValue()) {
			case UpdateViewConstants.UPDATE_OPTION_ALL:
				List<DisplayedSummary> summaries = new LinkedList<>();
				comboBoxCountryContent.stream().forEach(ci -> {
					ConversionSummaryByCountry conversionSummaryByCountry = courseConverterService.getConversionSummaryByCountryId(ci.getId_country().toString(), false);
					if(conversionSummaryByCountry.getNumIGolfCourses() !=  0 || conversionSummaryByCountry.getNumIGolfCourses() != 0) {
						DisplayedSummary summary = new DisplayedSummary(ci.getCountryFull(), conversionSummaryByCountry.getNumIGolfCourses(), conversionSummaryByCountry.getNumPlayertourCourses());
						summaries.add(summary);
					}					
				});				
				summaries.sort(SUMMARY_GRID_COMPARATOR.thenComparing(DisplayedSummary::getCountryName));
				summaryGrid.setItems(summaries);
				summaryGrid.getColumnByKey(UpdateViewConstants.UPDATE_VIEW_GRID_COUNTRY_COL_KEY).setVisible(true);
				summaryGrid.getColumnByKey(UpdateViewConstants.UPDATE_VIEW_GRID_STATE_COL_KEY).setVisible(false);
				summaryGrid.setVisible(true);
				break;
			case UpdateViewConstants.UPDATE_OPTION_BY_COUNTRY:
				summaryCountryComboBox.setItems(countryFilter, comboBoxCountryContent);
				summaryCountryComboBox.setVisible(true);
				break;
			}
		});
		
		summaryCountryComboBox.addValueChangeListener(evt -> {
			if(evt.getValue() != null) {
				CountryInfo ci = evt.getValue();
				List<DisplayedSummary> summaries = new LinkedList<>();
				ConversionSummaryByCountry conversionSummaryByCountry = courseConverterService.getConversionSummaryByCountryId(ci.getId_country().toString(), true);
				conversionSummaryByCountry.getConversionSummariesByState().forEach((k,v) -> {
					if(v.getNumIGolfCourses() != 0 || v.getNumPlayertourCourses() != 0) {
						DisplayedSummary summary = new DisplayedSummary(ci.getCountryFull(), v.getStateName(), v.getNumIGolfCourses(), v.getNumPlayertourCourses());
						summaries.add(summary);
					}					
				});
				summaries.sort(SUMMARY_GRID_COMPARATOR.thenComparing(DisplayedSummary::getStateName));
				summaryGrid.getColumnByKey(UpdateViewConstants.UPDATE_VIEW_GRID_COUNTRY_COL_KEY).setVisible(false);
				summaryGrid.getColumnByKey(UpdateViewConstants.UPDATE_VIEW_GRID_STATE_COL_KEY).setVisible(true);
				summaryGrid.setItems(summaries);
				summaryGrid.setVisible(true);
			}
		});
		summaryLayout.add(summaryOptionsRadioGroup, summaryCountryComboBox, summaryGrid);
		
	}



	private void updateStateInfoList(String countryId) {
		try {
			comboBoxStateContent = iGolfDBStateService.getStateInfoByCountry(countryId);
			if(comboBoxStateContent.isEmpty()) {
				displayNotification("There are no States for Country " + countryId + ". Please, use the update by country or try to update the state list first!", Position.BOTTOM_START, NotificationVariant.LUMO_CONTRAST);
			} 
			comboBoxStateContent.sort(Comparator.comparing(StateInfo::getStateFull));
		} catch (IGolfUpdateException e) {
			displayNotification("Error occurred when updating StateInfo list " + e.getMessage(), Position.BOTTOM_START, NotificationVariant.LUMO_ERROR);
			e.printStackTrace();
		}
		
	}

	private void updateCountryInfoList() {
		try {
			comboBoxCountryContent = iGolfDBCountryService.getAllCountryInfo();
			comboBoxContinentContent = comboBoxCountryContent.stream()
					.collect(Collectors.collectingAndThen(Collectors.toCollection(() -> new TreeSet<>(Comparator.comparing(CountryInfo::getId_continent))),
							ArrayList::new));

		} catch (IGolfUpdateException e) {
			displayNotification("Error occurred when updating CountryInfo list " + e.getMessage(), Position.BOTTOM_START, NotificationVariant.LUMO_ERROR);
			e.printStackTrace();
		}		
	}	
	
	private void displayNotification(String msg, Position position, NotificationVariant variant) {
		Notification.show(msg, 1000, position).addThemeVariants(variant);
	}
}
