import * as L from 'leaflet';
import { customElement, html, LitElement, PropertyValues } from 'lit-element';
import { nothing } from 'lit-html';

import { MapUtil } from "./MapUtil";
let myMapUtil = new MapUtil();

const openStreetMapLayer = 'https://server.arcgisonline.com/ArcGIS/rest/services/World_Imagery/MapServer/tile/{z}/{y}/{x}.png';
const openStreetMapAttribution = `&copy; <a href="https://www.esri.com/">Esri</a> contributors`;

@customElement('leaflet-map')
export class LeafletMap extends LitElement {
  private map!: L.Map;
  private geoJSONLayer!: any | undefined = undefined;

  createRenderRoot() {
    // Do not use a shadow root
    return this;
  }

  render() {
    return html`${nothing}`;
  }

  firstUpdated(_changedProperties: PropertyValues) {
    super.firstUpdated(_changedProperties);

    this.map = L.map(this);
    let tileLayer = L.tileLayer(openStreetMapLayer, { attribution: openStreetMapAttribution, maxZoom: 22 });
    tileLayer.addTo(this.map);
  }

  async setView(latitude: number, longitude: number, zoomLevel: number) {
    await this.updateComplete; // Make sure map has been initialized
    this.map.setView([latitude, longitude], zoomLevel);
  }
  
  
  async drawGPSInfo(jsonStr: string) {
  	await this.updateComplete; // Make sure map has been initialized
  	let data = JSON.parse(jsonStr);
	let geoObjs = myMapUtil.createGeoObjs(data);
	if(this.geoJSONLayer !== undefined) {
		console.log("I am removing the previous geoJson");
		this.map.removeLayer(this.geoJSONLayer);
	}
	try {
	this.geoJSONLayer = L.geoJSON(geoObjs, {
	  style: function(feature) {
	  	if(feature.properties.label.startsWith("Water") || feature.properties.label.startsWith("Ocean") || feature.properties.label.startsWith("Lake")) {
	  		return {color: "#00FFFF", fillColor: "#00FFFF", fillOpacity: "1."};
	  	}
	  	else if(feature.properties.label.startsWith("Clubhouse") || feature.properties.label.startsWith("Bunker") || feature.properties.label.startsWith("Sand")) {
	  		return {color: "#D2B48C", fillColor: "#D2B48C", fillOpacity: "1."};
	  	}
	  	else if(feature.properties.label.startsWith("Path") || feature.properties.label.startsWith("Bridge")) {
	  		return {color: "#8B4513", fillColor: "#8B4513", fillOpacity: "1."};
	  	}
	  	else if(feature.properties.label.startsWith("Background")) {
	  		return {color: "#8FBC8F", fillOpactity: "0.1"};
	  	}
	  	else if(feature.properties.label.startsWith("Green")) {
	  		return {color: "#9ACD32", fillColor: "#9ACD32", fillOpacity: "1."};
	  	}
	  	else if(feature.properties.label.startsWith("Fairway")) {
	  		return {opacity: "0.2", fillColor: "#708090"};
	  	}
	  	else if(feature.properties.label.startsWith("Centralpath")) {
	  		return {color: "#008B8B", fillColor: "#008B8B", fillOpacity: "1.", opacity: "1."};
	  	}
	  	else if(feature.properties.label.startsWith("Perimeter")) {
	  		return {color: "#6B8E23", fillOpacity: "0.1"};
	  	}
	  	else if(feature.properties.label.startsWith("Teebox")) {
	  		return {color: "#A0522D", fillColor: "#A0522D", fillOpacity: "1."};
	  	}
	  },
	  pointToLayer: function(feature, latlng) {
	  
	  	try {
	      		if(feature.properties.label.startsWith("Creek")) {
	      			return L.circleMarker(latlng, {color: "#A9A9A9", radius: 4});
	      		}
	      		else if(feature.properties.label.startsWith("Greencenter")) {
	      			return L.circleMarker(latlng, {color: "#00000", radius: 4, opacity: 0.5});
	      		}
	      		else if(feature.properties.label.startsWith("Teeboxcenter")) {
	      			return L.circleMarker(latlng, {color: "#00000", radius: 4});
	  		}
	  	} catch(error) {
	  		console.log("Error for " + feature.properties.label);
	  	}
	      
	     
	  },	      
	}).addTo(this.map);
	} catch(error) {
	  console.log("Error!!! " + error);
	}
	
	
  }
}
