/**
 * Copyright (c) 2012 - 2021 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package com.playertour.backend.vaadin.simulator;

import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;

import org.gecko.vaadin.whiteboard.annotations.VaadinComponent;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.annotations.ReferenceScope;
import org.osgi.service.component.annotations.ServiceScope;

import com.playertour.backend.api.Address;
import com.playertour.backend.api.GolfCourseSearchResult;
import com.playertour.backend.api.GolfCourseSearchResults;
import com.playertour.backend.api.PlayerApiFactory;
import com.playertour.backend.apis.gamesim.GolfGameSimulationService;
import com.playertour.backend.apis.mmt.common.UnknownTransformationException;
import com.playertour.backend.apis.player.APIGolfCourseService;
import com.playertour.backend.apis.player.PlayerSearchService;
import com.playertour.backend.vaadin.views.main.MainView;
import com.vaadin.flow.component.AbstractField.ComponentValueChangeEvent;
import com.vaadin.flow.component.button.Button;
import com.vaadin.flow.component.grid.Grid;
import com.vaadin.flow.component.html.Anchor;
import com.vaadin.flow.component.html.Label;
import com.vaadin.flow.component.notification.Notification;
import com.vaadin.flow.component.orderedlayout.FlexComponent;
import com.vaadin.flow.component.orderedlayout.FlexLayout;
import com.vaadin.flow.component.orderedlayout.HorizontalLayout;
import com.vaadin.flow.component.orderedlayout.VerticalLayout;
import com.vaadin.flow.component.radiobutton.RadioButtonGroup;
import com.vaadin.flow.component.radiobutton.RadioGroupVariant;
import com.vaadin.flow.component.textfield.NumberField;
import com.vaadin.flow.component.textfield.TextField;
import com.vaadin.flow.router.PageTitle;
import com.vaadin.flow.router.Route;
import com.playertour.backend.player.model.player.Player;

/**
 * 
 * @author ilenia
 * @since Jul 5, 2021
 */
@Route(value = "simulate", layout = MainView.class)
@PageTitle("Game Simulator")
@Component(name = "GameSimulatorView", service = GameSimulatorView.class, scope = ServiceScope.PROTOTYPE)
@VaadinComponent()
public class GameSimulatorView extends VerticalLayout {

	/** serialVersionUID */
	private static final long serialVersionUID = 5330167616320862269L;
	
	@Reference(scope = ReferenceScope.PROTOTYPE_REQUIRED)
	APIGolfCourseService courseService;

	@Reference
	PlayerSearchService playerSearchService;
	
	@Reference(scope = ReferenceScope.PROTOTYPE_REQUIRED)
	GolfGameSimulationService simulationService;

	private static final String GENERATE_PLAYERS_OPTION = "Generate players";
	private static final String CHOOSE_PLAYERS_OPTION = "Choose among existing players";
	private static final String CHOOSE_RANDOM_COURSES_OPTION = "Choose random golf courses";
	private static final String CHOOSE_COURSES_OPTION = "Choose golf courses";

	private HorizontalLayout playersHL = new HorizontalLayout();
	private HorizontalLayout coursesHL = new HorizontalLayout();

	private VerticalLayout playersOptionsVL = new VerticalLayout();
	private VerticalLayout playersOptionDetailsVL = new VerticalLayout();
	private VerticalLayout playersSearchVL = new VerticalLayout();
	private VerticalLayout coursesOptionsVL = new VerticalLayout();
	private VerticalLayout coursesOptionDetailsVL = new VerticalLayout();
	private VerticalLayout coursesSearchVL = new VerticalLayout();

	private NumberField playerNumToGenerate;
	private NumberField courseNumToChoose;

	private RadioButtonGroup<String> playersRadioGroup;
	private RadioButtonGroup<String> coursesRadioGroup;

	private Grid<DisplayedPlayer> playerGrid = new Grid<DisplayedPlayer>();
	private Set<DisplayedPlayer> selectedPlayers = Collections.emptySet();
	private TextField playerSearchText;

	private Grid<DisplayedCourse> courseGrid = new Grid<DisplayedCourse>();
	private Set<DisplayedCourse> selectedCourses = Collections.emptySet();
	private TextField coursesSearchText;

	private Button simulateBtn;
	private Button clearBtn;


	@Activate
	public void renderView() {

		renderPlayersOptions();
		renderCoursesOptions();
		renderButtons();

	}

	private void renderPlayersOptions() {

		playersRadioGroup = new RadioButtonGroup<String>();
		playersRadioGroup.setLabel("Choose players for simulated game:");
		playersRadioGroup.setItems(GENERATE_PLAYERS_OPTION, CHOOSE_PLAYERS_OPTION);
		playersRadioGroup.addThemeVariants(RadioGroupVariant.LUMO_VERTICAL);
		playersRadioGroup.addValueChangeListener(event-> {
			reactToPlayerOption(event);
		});		
		playersOptionsVL.add(playersRadioGroup);

		playerNumToGenerate = new NumberField("Number of players to be generated:");
		playerNumToGenerate.setMin(1.);
		playerNumToGenerate.setStep(1.);
		playerNumToGenerate.setErrorMessage("Please set an integer number greater than 0!");
		playerNumToGenerate.setVisible(false);
		playerNumToGenerate.setValue(1.);
		playerNumToGenerate.setMinWidth("250px");
		playerNumToGenerate.addValueChangeListener(evt -> {
			if((playerNumToGenerate.isVisible() && !playerNumToGenerate.isInvalid()) && 
					(selectedCourses.size() > 0 || 
							(courseNumToChoose.isVisible() && !courseNumToChoose.isInvalid()))) {
				simulateBtn.setEnabled(true);
			}
			else {
				simulateBtn.setEnabled(false);
			}
		});
		playersOptionDetailsVL.add(playerNumToGenerate);		

		HorizontalLayout playersSearchHL = new HorizontalLayout();
		Label playerSearchLabel = new Label("Search Players by username:");
		playerSearchText = new TextField();
		Button playerSearchBtn = new Button("Search", evt-> {	
			String query = playerSearchText.getValue();
			List<Player> searchResult = playerSearchService.searchPlayersByUsername(query, Integer.MAX_VALUE);
			displayPlayerResults(searchResult);
			
			
		});	
		playerGrid.setSelectionMode(Grid.SelectionMode.MULTI);
		playerGrid.addColumn(DisplayedPlayer::getUsername).setHeader("Username");
		playerGrid.addColumn(DisplayedPlayer::getTeePreference).setHeader("Tee Preference");
		playerGrid.addColumn(DisplayedPlayer::getExperienceLevel).setHeader("Experience Level");
		playerGrid.addSelectionListener(evt -> {
			selectedPlayers = evt.getAllSelectedItems();
			if(selectedPlayers.size() > 0 && 
					(selectedCourses.size() > 0 || 
							(courseNumToChoose.isVisible() && !courseNumToChoose.isInvalid()))) {
				simulateBtn.setEnabled(true);
			}
			else {
				simulateBtn.setEnabled(false);
			}
		});

		playersSearchHL.add(playerSearchLabel, playerSearchText, playerSearchBtn);
		playersSearchVL.add(playersSearchHL, playerGrid);
		playersSearchVL.setVisible(false);
		playersOptionDetailsVL.add(playersSearchVL);
		playersOptionDetailsVL.setVisible(false);
		playersOptionDetailsVL.getElement().getStyle().set("flex-grow", "3");
		playersOptionDetailsVL.getElement().getStyle().remove("width");
		playersOptionDetailsVL.addClassName("detail-options-layout");

		playersOptionsVL.getElement().getStyle().set("flex-grow", "1");
		playersOptionsVL.getElement().getStyle().remove("width");
		playersOptionsVL.addClassName("options-layout");
		playersHL.add(playersOptionsVL, playersOptionDetailsVL);
		playersHL.setWidthFull();
		playersHL.addClassName("players-hl");
		add(playersHL);		
	}

	
	

	private void renderCoursesOptions() {
		coursesRadioGroup = new RadioButtonGroup<String>();
		coursesRadioGroup.setLabel("Choose courses for simulated game:");
		coursesRadioGroup.setItems(CHOOSE_RANDOM_COURSES_OPTION, CHOOSE_COURSES_OPTION);
		coursesRadioGroup.addThemeVariants(RadioGroupVariant.LUMO_VERTICAL);
		coursesRadioGroup.addValueChangeListener(event-> {
			reactToCourseOption(event);
		});		
		coursesOptionsVL.add(coursesRadioGroup);

		courseNumToChoose = new NumberField("Number of courses to be chosen:");
		courseNumToChoose.setMin(1.);
		courseNumToChoose.setStep(1.);
		courseNumToChoose.setErrorMessage("Please set an integer number greater than 0!");
		courseNumToChoose.setVisible(false);
		courseNumToChoose.setValue(0.);
		courseNumToChoose.setEnabled(false);
		courseNumToChoose.setMinWidth("250px");
		courseNumToChoose.addValueChangeListener(evt -> {
			if((courseNumToChoose.isVisible() && !courseNumToChoose.isInvalid()) && 
					(selectedPlayers.size() > 0 || 
							(playerNumToGenerate.isVisible() && !playerNumToGenerate.isInvalid()))) {
				simulateBtn.setEnabled(true);
			}
			else {
				simulateBtn.setEnabled(false);
			}
		});
		coursesOptionDetailsVL.add(courseNumToChoose);		

		HorizontalLayout coursesSearchHL = new HorizontalLayout();
		Label coursesSearchLabel = new Label("Search Courses by name:");
		coursesSearchText = new TextField();
		Button coursesSearchBtn = new Button("Search", evt-> {				
			String query = coursesSearchText.getValue();
			try {
				GolfCourseSearchResults coursesSearchResult = PlayerApiFactory.eINSTANCE.createGolfCourseSearchResults();
				List<GolfCourseSearchResult> courses = courseService.findValidCourses(query, Integer.MAX_VALUE);
				coursesSearchResult.getResults().addAll(courses);
				displayCoursesResults(coursesSearchResult);						
				Notification.show("Found " + courses.size() + " results!");
			} catch (UnknownTransformationException e) {
				Notification.show("Error searching for Golf Courses!");
			}
		});	
		courseGrid.setSelectionMode(Grid.SelectionMode.MULTI);
		courseGrid.addColumn(DisplayedCourse::getName).setHeader("Username");
		courseGrid.addColumn(DisplayedCourse::getAddress).setHeader("Address");
		courseGrid.addComponentColumn(v-> new Anchor(v.url, v.url)).setHeader("URL");
		courseGrid.addSelectionListener(evt -> {
			selectedCourses = evt.getAllSelectedItems();
			if(selectedCourses.size() > 0 && 
					(selectedPlayers.size() > 0 || 
							(playerNumToGenerate.isVisible() && !playerNumToGenerate.isInvalid()))) {
				simulateBtn.setEnabled(true);
			}
			else {
				simulateBtn.setEnabled(false);
			}
		});

		coursesSearchHL.add(coursesSearchLabel, coursesSearchText, coursesSearchBtn);
		coursesSearchVL.add(coursesSearchHL, courseGrid);
		coursesSearchVL.setVisible(false);
		coursesOptionDetailsVL.add(coursesSearchVL);
		coursesOptionDetailsVL.setVisible(false);
		coursesOptionDetailsVL.getElement().getStyle().set("flex-grow", "3");
		coursesOptionDetailsVL.getElement().getStyle().remove("width");
		coursesOptionDetailsVL.addClassName("detail-options-layout");

		coursesOptionsVL.getElement().getStyle().set("flex-grow", "1");
		coursesOptionsVL.getElement().getStyle().remove("width");
		coursesOptionsVL.addClassName("options-layout");
		coursesHL.add(coursesOptionsVL, coursesOptionDetailsVL);
		coursesHL.setWidthFull();
		coursesHL.addClassName("courses-hl");
		add(coursesHL);				
	}

	
	private void displayPlayerResults(List<Player> searchResult) {

		List<DisplayedPlayer> displayedPlayers = new LinkedList<>();
		searchResult.stream().forEach(r-> {
			DisplayedPlayer displayedPlayer = new DisplayedPlayer(r.getLoginId(), 
					r.getProfile().getLoginName(), 
					r.getProfile().getTeePreference().getLiteral(),
					r.getProfile().getExperienceLevel().getLiteral());
			displayedPlayers.add(displayedPlayer);
		});
		playerGrid.setItems(displayedPlayers);		
	}
	
	private void displayCoursesResults(GolfCourseSearchResults coursesSearchResult) {
		if(coursesSearchResult == null) {
			courseGrid.setItems(Collections.emptyList());
			return;
		}
		List<DisplayedCourse> displayedCourses = new LinkedList<>();
		coursesSearchResult.getResults().stream().forEach(r -> {
			DisplayedCourse displayedCourse = 
					new DisplayedCourse(r.getCourse().getId(), r.getCourse().getName(),
							r.getCourse().getCourseUrl(), r.getCourse().getAddress());
			displayedCourses.add(displayedCourse);			
		});
		courseGrid.setItems(displayedCourses);			
	}

	private void renderButtons() {
		HorizontalLayout btnHL = new HorizontalLayout();
		simulateBtn = new Button("Simulate", evt-> {
			simulateGame();
		});
		simulateBtn.setEnabled(false);

		clearBtn = new Button("Clear", evt-> {
			clearPage();
		});
		
		FlexLayout wrapper = new FlexLayout(simulateBtn, clearBtn);
		btnHL.expand(wrapper);
		wrapper.setJustifyContentMode(FlexComponent.JustifyContentMode.AROUND);
		btnHL.setPadding(true);
		btnHL.setWidth("30%");
		btnHL.add(wrapper);
		add(btnHL);
	}

	private void reactToPlayerOption(ComponentValueChangeEvent<RadioButtonGroup<String>, String> event) {
		if(event.getValue() == null) return;
		switch(event.getValue()) {
		case GENERATE_PLAYERS_OPTION:
			playersOptionDetailsVL.setVisible(true);
			playerNumToGenerate.setVisible(true);
			playerNumToGenerate.setValue(1.);
			if(selectedCourses.size() > 0 || 
					(courseNumToChoose.isVisible() && !courseNumToChoose.isInvalid())) {
				simulateBtn.setEnabled(true);
			}
			else {
				simulateBtn.setEnabled(false);
			}
			playersSearchVL.setVisible(false);
			selectedPlayers = Collections.emptySet();
			break;
		case CHOOSE_PLAYERS_OPTION:
			playersOptionDetailsVL.setVisible(true);
			playerNumToGenerate.setVisible(false);
			playerNumToGenerate.setValue(1.);
			playersSearchVL.setVisible(true);
			simulateBtn.setEnabled(false);
			playerGrid.setItems(Collections.emptyList());
			if(playerSearchText != null) {
				playerSearchText.setValue("");
			}
			break;
		}		
	}

	private void reactToCourseOption(ComponentValueChangeEvent<RadioButtonGroup<String>, String> event) {
		if(event.getValue() == null) return;
		switch(event.getValue()) {
		case CHOOSE_RANDOM_COURSES_OPTION:
			coursesOptionDetailsVL.setVisible(true);
			courseNumToChoose.setVisible(true);
			courseNumToChoose.setValue(0.);
			courseNumToChoose.setEnabled(false);
			if(selectedPlayers.size() > 0 || 
					(playerNumToGenerate.isVisible() && !playerNumToGenerate.isInvalid())) {
				simulateBtn.setEnabled(true);
			}
			else {
				simulateBtn.setEnabled(false);
			}
			coursesSearchVL.setVisible(false);
			selectedCourses = Collections.emptySet();
			break;
		case CHOOSE_COURSES_OPTION:
			coursesOptionDetailsVL.setVisible(true);
			courseNumToChoose.setVisible(false);
			courseNumToChoose.setValue(0.);
			courseNumToChoose.setEnabled(false);
			coursesSearchVL.setVisible(true);
			simulateBtn.setEnabled(false);
			courseGrid.setItems(Collections.emptyList());
			if(coursesSearchText != null) {
				coursesSearchText.setValue("");
			}
			break;
		}		
	}

	private void simulateGame() {
		Notification.show("Ready to Simulate!");
		boolean generatePlayers = (playerNumToGenerate != null && playerNumToGenerate.isVisible() && !playerNumToGenerate.isInvalid());
		int numberOfPlayersToGenerate = (int) (generatePlayers ? playerNumToGenerate.getValue() : 0);
		List<String> existingPlayersLoginIds = new LinkedList<>();
		if(!generatePlayers) {			
			for(DisplayedPlayer selectedPlayer : selectedPlayers) {
				existingPlayersLoginIds.add(selectedPlayer.getId());
			}
		}
		
		boolean chooseCoursesRandom = (courseNumToChoose != null && courseNumToChoose.isVisible() && !courseNumToChoose.isInvalid());
		@SuppressWarnings("unused")
		int numOfCoursesToChooseRandomly = (int) (chooseCoursesRandom ? courseNumToChoose.getValue() : 0);
		
		List<String> golfCoursesIds = new LinkedList<>();		
		if(!chooseCoursesRandom) {
			for(DisplayedCourse selectedCourse : selectedCourses) {
				golfCoursesIds.add(selectedCourse.getId());
			}
		} else {
//			TODO select golf courses randomly and place ids into golfCoursesIds
		}
		simulationService.playGolfGame(golfCoursesIds, true, generatePlayers, numberOfPlayersToGenerate, 
				false, 0, 0, existingPlayersLoginIds.toArray(new String[0]));
		
		Notification.show("Simulation completed!");
	}

	private void clearPage() {
		playersRadioGroup.setValue(null);
		playersOptionDetailsVL.setVisible(false);
		playerNumToGenerate.setVisible(false);
		playerNumToGenerate.setValue(1.);
		playersSearchVL.setVisible(false);
		selectedPlayers = Collections.emptySet();
		playerGrid.setItems(Collections.emptyList());
		if(playerSearchText != null) {
			playerSearchText.setValue("");
		}

		coursesRadioGroup.setValue(null);
		coursesOptionDetailsVL.setVisible(false);
		courseNumToChoose.setVisible(false);
		courseNumToChoose.setValue(0.);
		courseNumToChoose.setEnabled(false);
		coursesSearchVL.setVisible(false);
		selectedCourses = Collections.emptySet();
		courseGrid.setItems(Collections.emptyList());
		if(coursesSearchText != null) {
			coursesSearchText.setValue("");
		}
	}

	class DisplayedPlayer {
		private String id;
		private String username;
		private String teePreference;
		private String experienceLevel;

		public DisplayedPlayer(String id, String username, String teePreference, String experienceLevel) {
			this.id = id;
			this.username = username;
			this.teePreference = teePreference;
			this.experienceLevel = experienceLevel;
		}
		
		public String getId() {
			return this.id;
		}

		public String getUsername() {
			return username;
		}

		public String getTeePreference() {
			return teePreference;
		}

		public String getExperienceLevel() {
			return experienceLevel;
		}
	}

	class DisplayedCourse {

		private String name;
		private String id;
		private String url;
		private String address;

		public DisplayedCourse(String id, String name, String url, Address address) {
			this.id = id;
			this.name = name;
			this.url = url;
			StringBuilder sb = new StringBuilder();
			sb.append(address.getRoute() != null ? address.getRoute() : "");
			sb.append(", ");
			sb.append(address.getZipCode() != null ? address.getZipCode() : "");
			sb.append(", ");
			sb.append(address.getCity() != null ? address.getCity() : "");
			sb.append(", ");
			sb.append(address.getState() != null ? address.getState() : "");
			sb.append(", ");
			sb.append(address.getCountry() != null ? address.getCountry() : "");

			this.address = sb.toString();
		}

		public String getId() {
			return id;
		}

		public String getName() {
			return name;
		}

		public String getUrl() {
			return url;
		}

		public String getAddress() {
			return address;
		}
	}

}
