/**
 * Copyright (c) 2012 - 2021 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package com.playertour.backend.vaadin.views.manual.course;

import java.io.File;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import org.gecko.vaadin.whiteboard.annotations.VaadinComponent;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.annotations.ReferenceScope;
import org.osgi.service.component.annotations.ServiceScope;

import com.playertour.backend.apis.course.CourseGPSService;
import com.playertour.backend.apis.course.CourseService;
import com.playertour.backend.golfcourse.model.golfcourse.CourseDetails;
import com.playertour.backend.golfcourse.model.golfcourse.CourseGPSVector;
import com.playertour.backend.golfcourse.model.golfcourse.CourseScorecardInfo;
import com.playertour.backend.golfcourse.model.golfcourse.CourseScorecards;
import com.playertour.backend.golfcourse.model.golfcourse.GenderType;
import com.playertour.backend.golfcourse.model.golfcourse.GolfCourse;
import com.playertour.backend.golfcourse.model.golfcourse.GolfCourseFactory;
import com.playertour.backend.golfcourse.model.golfcourse.Location;
import com.playertour.backend.golfcourse.model.golfcourse.Tee;
import com.playertour.backend.vaadin.views.main.MainView;
import com.vaadin.flow.component.button.Button;
import com.vaadin.flow.component.html.Div;
import com.vaadin.flow.component.html.Label;
import com.vaadin.flow.component.notification.Notification;
import com.vaadin.flow.component.orderedlayout.HorizontalLayout;
import com.vaadin.flow.component.orderedlayout.VerticalLayout;
import com.vaadin.flow.component.radiobutton.RadioButtonGroup;
import com.vaadin.flow.component.radiobutton.RadioGroupVariant;
import com.vaadin.flow.component.select.Select;
import com.vaadin.flow.component.tabs.Tab;
import com.vaadin.flow.component.tabs.Tabs;
import com.vaadin.flow.component.textfield.NumberField;
import com.vaadin.flow.component.textfield.TextField;
import com.vaadin.flow.component.upload.Upload;
import com.vaadin.flow.component.upload.receivers.FileBuffer;
import com.vaadin.flow.component.upload.receivers.FileData;
import com.vaadin.flow.router.PageTitle;
import com.vaadin.flow.router.Route;

import elemental.json.Json;


/**
 * 
 * @author ilenia
 * @since Jul 22, 2021
 */
@Route(value = "manual-course-creation", layout = MainView.class)
@PageTitle("Manual Course Creation")
@Component(name = "ManualCourseView", service = ManualCourseView.class, scope = ServiceScope.PROTOTYPE)
@VaadinComponent()
public class ManualCourseView extends VerticalLayout {

	/** serialVersionUID */
	private static final long serialVersionUID = 4064382428200409365L;
	
	@Reference(scope = ReferenceScope.PROTOTYPE_REQUIRED)
	CourseGPSService gpsService;
	
	@Reference(scope = ReferenceScope.PROTOTYPE_REQUIRED)
	CourseService courseService;
	
	
	private CourseDetails courseDetails;
	private Location courseLocation;
	private CourseGPSVector courseGPSVector;
	private List<Tee> courseTees;
	private CourseScorecards courseScorecards;
	
	private TextField courseNameField;
	private NumberField latField, lonField;
	private RadioButtonGroup<String> holeOptionsRadioGroup;
	
	private Upload upload;
	private File kmlFile;
	
	private RadioButtonGroup<String> distanceUnitsOptionsRadioGroup;
	private Boolean isDistUnitMeters;
	private Button addTeeBtn = new Button("Add Tee");
	private Label teeLabel = new Label();
	private VerticalLayout teeFormLayout = new VerticalLayout();
	
	private Button addScorecardBtn = new Button("Add Scorecard");
	private Label scorecardLabel = new Label();
	private VerticalLayout scorecardFormLayout = new VerticalLayout();
	
	private Button submitBtnTab1, submitBtnTab2, submitBtnTab3, submitBtnTab4;
	private Button clearBtnTab1, clearBtnTab2, clearBtnTab3, clearBtnTab4;	
	
	@Activate
	public void renderView() {
		
		clearForm();
		
		Tab tab1 = new Tab("Course Details");
		HorizontalLayout hl1 = new HorizontalLayout();
		courseNameField = new TextField("Golf Course Name:");
		courseNameField.setWidthFull();
		courseNameField.addValueChangeListener(event -> {
			courseDetails.setCourseName(event.getValue());
			setSumbitVisibility();
		});
		
		holeOptionsRadioGroup = new RadioButtonGroup<>();
		holeOptionsRadioGroup.setLabel("Select the number of holes:");
		holeOptionsRadioGroup.setItems("9", "18");
		holeOptionsRadioGroup.addValueChangeListener(event -> {
			if("9".equals(event.getValue())) {
				courseDetails.setHoleNum(9);
			}
			else if("18".equals(event.getValue())) {
				courseDetails.setHoleNum(18);
			}
			else {
				courseDetails.setHoleNum(0);
			}
			setSumbitVisibility();
		});
		holeOptionsRadioGroup.addThemeVariants(RadioGroupVariant.LUMO_VERTICAL);
		
		latField = new NumberField("Latitude");
		latField.setWidthFull();
		latField.setStep(0.000000000000001);
		latField.setMin(-90.);
		latField.setMax(90.);	
		latField.setValue(0.);
		latField.setErrorMessage("Allowed values between -90 and +90!");
		latField.addValueChangeListener(event -> {
			courseLocation.setLatitude(latField.getValue());
			courseDetails.setLocation(courseLocation);
			setSumbitVisibility();
		});

		lonField = new NumberField("Longitude");
		lonField.setWidthFull();
		lonField.setStep(0.000000000000001);
		lonField.setMin(-180.);
		lonField.setMax(180.);
		lonField.setValue(0.);
		lonField.setErrorMessage("Allowed values between -180 and +180!");
		lonField.addValueChangeListener(event -> {
			courseLocation.setLongitude(lonField.getValue());
			courseDetails.setLocation(courseLocation);
			setSumbitVisibility();
		});
		hl1.add(courseNameField, latField, lonField);
		hl1.setWidthFull();
		
		Div page1 = new Div();
		page1.add(hl1, holeOptionsRadioGroup);
		page1.setWidthFull();
		page1.setHeightFull();
		
		Tab tab2 = new Tab("GPS Vector");
		VerticalLayout vl2 = new VerticalLayout();		
		FileBuffer buffer = new FileBuffer();
		upload = new Upload(buffer);
		upload.setAcceptedFileTypes(".kml");
		upload.addSucceededListener(event -> {
			Notification.show("File successfully uploaded!").getElement().getThemeList().add("success");
			FileData fileData = buffer.getFileData();
			kmlFile = fileData.getFile();
			setSumbitVisibility();
		});

		upload.getElement().addEventListener("file-remove", event -> {
			kmlFile = null;
			courseGPSVector = null;
			setSumbitVisibility();
		});
		
		upload.setWidthFull();
		vl2.add(upload);
		vl2.setWidthFull();
		Div page2 = new Div();
		page2.add(vl2);
		page2.setVisible(false);
		
		Tab tab3 = new Tab("Tees");
		VerticalLayout vl3 = new VerticalLayout();
		distanceUnitsOptionsRadioGroup = new RadioButtonGroup<>();
		distanceUnitsOptionsRadioGroup.setLabel("Choose units of measure for distances:");
		distanceUnitsOptionsRadioGroup.setItems("Meters", "Yards");
		distanceUnitsOptionsRadioGroup.addValueChangeListener(event -> {
			if("Meters".equals(event.getValue())) {
				isDistUnitMeters = true;
				addTeeBtn.setVisible(true);
			}
			else if("Yards".equals(event.getValue())) {
				isDistUnitMeters = false;
				addTeeBtn.setVisible(true);
			}
			else {
				isDistUnitMeters = null;
				addTeeBtn.setVisible(false);
			}
			setSumbitVisibility();
		});
		
		teeFormLayout.setVisible(false);
		addTeeBtn.setVisible(false);
		addTeeBtn.addClickListener(event -> {
			fillTeeForm();
			teeFormLayout.setVisible(true);
		});
		HorizontalLayout hl3 = new HorizontalLayout();
		hl3.add(addTeeBtn, teeLabel);
		vl3.add(distanceUnitsOptionsRadioGroup, hl3, teeFormLayout);
		Div page3 = new Div();
		page3.add(vl3);
		page3.setVisible(false);

		Tab tab4 = new Tab("Scorecards");
		VerticalLayout vl4 = new VerticalLayout();		
		scorecardFormLayout.setVisible(false);
		addScorecardBtn.addClickListener(event -> {
			fillScorecardForm();
			scorecardFormLayout.setVisible(true);
		});
		HorizontalLayout hl4 = new HorizontalLayout();
		hl4.add(addScorecardBtn, scorecardLabel);
		vl4.add(hl4, scorecardFormLayout);
		Div page4 = new Div();
		page4.add(vl4);
		page4.setVisible(false);	
		
		submitBtnTab1 = new Button("Submit", event -> {
			createCourse();
			clearForm();
		});
		submitBtnTab1.setEnabled(false);
		
		submitBtnTab2 = new Button("Submit", event -> {
			createCourse();
			clearForm();
		});
		submitBtnTab2.setEnabled(false);
		
		submitBtnTab3 = new Button("Submit", event -> {
			createCourse();
			clearForm();
		});
		submitBtnTab3.setEnabled(false);
		
		submitBtnTab4 = new Button("Submit", event -> {
			createCourse();
			clearForm();
		});
		submitBtnTab4.setEnabled(false);
		
		clearBtnTab1 = new Button("Clear", event -> {
			clearTab1();
		});
		clearBtnTab2 = new Button("Clear", event -> {
			clearTab2();
		});
		clearBtnTab3 = new Button("Clear All", event -> {
			clearTab3();
		});
		clearBtnTab3.getElement().setProperty("title", "Remove all previously created Tees");
		
		clearBtnTab4 = new Button("Clear All", event -> {
			clearTab4();
		});
		clearBtnTab4.getElement().setProperty("title", "Remove all previously created Scorecards");	
		
		HorizontalLayout submitLayoutTab1 = new HorizontalLayout();
		HorizontalLayout submitLayoutTab2 = new HorizontalLayout();
		HorizontalLayout submitLayoutTab3 = new HorizontalLayout();
		HorizontalLayout submitLayoutTab4 = new HorizontalLayout();

		submitLayoutTab1.add(submitBtnTab1, clearBtnTab1);
		submitLayoutTab2.add(submitBtnTab2, clearBtnTab2);
		submitLayoutTab3.add(submitBtnTab3, clearBtnTab3);
		submitLayoutTab4.add(submitBtnTab4, clearBtnTab4);
		
		submitLayoutTab1.setJustifyContentMode(JustifyContentMode.END);
		submitLayoutTab2.setJustifyContentMode(JustifyContentMode.END);
		submitLayoutTab3.setJustifyContentMode(JustifyContentMode.END);
		submitLayoutTab4.setJustifyContentMode(JustifyContentMode.END);
		
		page1.add(submitLayoutTab1);
		page2.add(submitLayoutTab2);
		page3.add(submitLayoutTab3);
		page4.add(submitLayoutTab4);
		
		Map<Tab, com.vaadin.flow.component.Component> tabsToPages = new HashMap<>();
		tabsToPages.put(tab1, page1);
		tabsToPages.put(tab2, page2);
		tabsToPages.put(tab3, page3);
		tabsToPages.put(tab4, page4);
		Tabs tabs = new Tabs(tab1, tab2, tab3, tab4);
		Div pages = new Div(page1, page2, page3, page4);

		tabs.addSelectedChangeListener(event -> {
		    tabsToPages.values().forEach(page -> page.setVisible(false));
		    com.vaadin.flow.component.Component selectedPage = tabsToPages.get(tabs.getSelectedTab());
		    selectedPage.setVisible(true);
		});
		
		tabs.setSelectedTab(tab1);
		setAlignItems(Alignment.STRETCH);
		setJustifyContentMode(JustifyContentMode.EVENLY);
		add(tabs, pages);
	}

	
	private void clearForm() {
		clearTab1();
		clearTab2();
		clearTab3();
		clearTab4();		
	}
	
	private void clearTab1() {
		courseDetails = GolfCourseFactory.eINSTANCE.createCourseDetails();
		courseLocation = GolfCourseFactory.eINSTANCE.createLocation();
		if(courseNameField != null) {
			courseNameField.setValue("");
		}
		if(latField != null) {
			latField.setValue(0.);
		}
		if(lonField != null) {
			lonField.setValue(0.);
		}
		if(holeOptionsRadioGroup != null) {
			holeOptionsRadioGroup.setValue(null);
		}
	}
	
	private void clearTab2() {
		if(upload != null) {
			upload.getElement().setPropertyJson("files", Json.createArray());
			kmlFile = null;
			courseGPSVector = null;
		}
	}
	
	private void clearTab3() {
		if(distanceUnitsOptionsRadioGroup != null) {
			distanceUnitsOptionsRadioGroup.setValue(null);
		}
		courseTees = new LinkedList<Tee>();
		teeLabel.setText("");
	}
	
	private void clearTab4() {
		courseScorecards = GolfCourseFactory.eINSTANCE.createCourseScorecards();
		scorecardLabel.setText("");
	}



	private void fillScorecardForm() {
		scorecardFormLayout.removeAll();
		
		HorizontalLayout hl = new HorizontalLayout();
		Select<String> scorecardGenderSelect = new Select<String>();
		scorecardGenderSelect.setItems("MEN", "WOMEN");
		scorecardGenderSelect.setLabel("Select Scorecard Gender:");
		hl.add(scorecardGenderSelect);
		
		List<Integer> parList = new ArrayList<>(18);
		List<Integer> hcpList = new ArrayList<>(18);
		for(int i = 0; i < 18; i++) {
			parList.add(0);
			hcpList.add(0);
		}
		
		Label parLabel = new Label("Add the par values for each hole:");
		Label hcpLabel = new Label("Add the hcp values for each hole:");
		
		HorizontalLayout parhl1to9 = new HorizontalLayout();
		HorizontalLayout parhl10to18 = new HorizontalLayout();
		HorizontalLayout hcphl1to9 = new HorizontalLayout();
		HorizontalLayout hcphl10to18 = new HorizontalLayout();
		for(int h = 1; h <= 18; h++) {
			NumberField parHole = createHoleDistField("Par Hole ", h, parList);
			NumberField hcpHole = createHoleDistField("Hcp Hole ", h, hcpList);
			if(h <= 9) {
				parhl1to9.add(parHole);
				hcphl1to9.add(hcpHole);
			}
			else {
				parhl10to18.add(parHole);
				hcphl10to18.add(hcpHole);
			}			
		}
		
		Button saveBtn = new Button("Save", event -> {
			CourseScorecardInfo scorecardInfo = GolfCourseFactory.eINSTANCE.createCourseScorecardInfo();
			scorecardInfo.getParHole().addAll(parList);
			scorecardInfo.getHcpHole().addAll(hcpList);
			int parOut = parList.subList(0, 9).stream().mapToInt(i->i).sum();
			int parIn = parList.subList(9, 18).stream().mapToInt(i->i).sum();
			int parTot = parOut + parIn;
			scorecardInfo.setParIn(parIn);
			scorecardInfo.setManualParIn(parIn);
			scorecardInfo.setParOut(parOut);
			scorecardInfo.setManualParOut(parOut);
			scorecardInfo.setParTotal(parTot);
			scorecardInfo.setManualParTotal(parTot);			
			if("MEN".equals(scorecardGenderSelect.getValue())) {
				courseScorecards.getMenScorecard().add(scorecardInfo);
			} 
			else if("WOMEN".equals(scorecardGenderSelect.getValue())) {
				courseScorecards.getWmnScorecard().add(scorecardInfo);
			} 
			Notification.show("Scorecard added succesfully!").getElement().getThemeList().add("success");
			int numScorecards = courseScorecards.getMenScorecard().size() + courseScorecards.getWmnScorecard().size();
			scorecardLabel.setText("You have added " + numScorecards + " Scorecard!");
			setSumbitVisibility();
			scorecardFormLayout.setVisible(false);			
		});
		
		Button clearFormBtnTab4 = new Button("Clear Form", event -> {
			fillScorecardForm();
		});
		
		HorizontalLayout btnLayout = new HorizontalLayout();
		btnLayout.add(saveBtn, clearFormBtnTab4);
		scorecardFormLayout.add(hl, parLabel, parhl1to9, parhl10to18, hcpLabel, hcphl1to9, hcphl10to18, btnLayout);	
	}



	private void fillTeeForm() {
		
		teeFormLayout.removeAll();
		
		List<Integer> distList = new ArrayList<>(18);
		for(int i = 0; i < 18; i++) {
			distList.add(0);
		}
		
		HorizontalLayout hl = new HorizontalLayout();
		
		Select<String> teeGenderSelect = new Select<>();
		teeGenderSelect.setItems("MEN", "WOMEN");
		teeGenderSelect.setLabel("Select Tee Gender:");
		
		Select<String> teeColorSelect = new Select<>();
		teeColorSelect.setItems("YELLOW", "RED");
		teeColorSelect.setLabel("Select Tee Color:");
		
		hl.add(teeGenderSelect, teeColorSelect);
		HorizontalLayout hl1to9 = new HorizontalLayout();
		HorizontalLayout hl10to18 = new HorizontalLayout();
		for(int h = 1; h <= 18; h++) {
			NumberField distHole = createHoleDistField("Dist Hole ", h, distList);
			if(h <= 9) {
				hl1to9.add(distHole);
			}
			else {
				hl10to18.add(distHole);
			}			
		}	
	
		Button saveBtn = new Button("Save", event -> {
			Tee tee = GolfCourseFactory.eINSTANCE.createTee();
			if("MEN".equals(teeGenderSelect.getValue())) {
				tee.setGender(GenderType.MEN);
			}
			else if("WOMEN".equals(teeGenderSelect.getValue())) {
				tee.setGender(GenderType.WOMEN);
			}	
			if("YELLOW".equals(teeColorSelect.getValue())) {
				tee.setColor("Yellow");
				tee.setColorValue("FFFF00");
			}
			else if("RED".equals(teeColorSelect.getValue())) {
				tee.setColor("Red");
				tee.setColorValue("FF0000");
			}	
			int dist1to18 = distList.stream().mapToInt(i->i).sum();
			int dist1to9 = distList.subList(0, 9).stream().mapToInt(i->i).sum();
			int dist10to18 = distList.subList(9, 18).stream().mapToInt(i->i).sum();
			if(isDistUnitMeters) {
				tee.getMetersHole().addAll(distList);				
				tee.setMeters1to18(String.valueOf(dist1to18));				
				tee.setMeters1to9(String.valueOf(dist1to9));
				tee.setMeters10to18(String.valueOf(dist10to18));
			} else {
				tee.getYdsHole().addAll(distList);				
				tee.setYds1to18(String.valueOf(dist1to18));				
				tee.setYds1to9(String.valueOf(dist1to9));
				tee.setYds10to18(String.valueOf(dist10to18));
			}
			courseTees.add(tee);
			Notification.show("Tee added succesfully!").getElement().getThemeList().add("success");
			teeLabel.setText("You have added " + courseTees.size() + " Tees!");
			setSumbitVisibility();
			teeFormLayout.setVisible(false);
		});	
		
		Button clearFormBtnTab3 = new Button("Clear Form", event -> {
			fillTeeForm();
		});
		HorizontalLayout btnLayout = new HorizontalLayout();
		btnLayout.add(saveBtn, clearFormBtnTab3);
		
		teeFormLayout.add(hl, hl1to9, hl10to18, btnLayout);
	}


	private NumberField createHoleDistField(String label, int holeNum, List<Integer> distList) {
		NumberField fieldHole = new NumberField(label + holeNum + ":");
		fieldHole.setStep(1.);
		fieldHole.setMin(0);
		fieldHole.setValue(0.);
		fieldHole.addValueChangeListener(event -> {
			distList.add(holeNum-1, Integer.valueOf((int) Math.round(event.getValue())));
		});
		return fieldHole;
	}

	private void createCourse() {		
		GolfCourse golfCourse = GolfCourseFactory.eINSTANCE.createGolfCourse();
		courseGPSVector = gpsService.createManualGPSVector(kmlFile.getPath(), courseDetails.getHoleNum());
		golfCourse.setCourseDetails(courseDetails);
		golfCourse.setCourseGPSVector(courseGPSVector);
		golfCourse.getTee().addAll(courseTees);
		golfCourse.setScorecards(courseScorecards);
		courseService.saveCourse(golfCourse);
		Notification.show("Golf Course succesfully created!").getElement().getThemeList().add("success");		
	}

	private void setSumbitVisibility() {		
		if(courseDetails.getCourseName() == null || courseDetails.getCourseName().isEmpty() || 
				courseDetails.getHoleNum() == 0 || 
				courseDetails.getLocation() == null || 
				kmlFile == null || 
				isDistUnitMeters == null || 
				courseTees.isEmpty() || 
				(courseScorecards.getMenScorecard().isEmpty() && courseScorecards.getWmnScorecard().isEmpty())) {
			
			submitBtnTab1.setEnabled(false);
			submitBtnTab2.setEnabled(false);
			submitBtnTab3.setEnabled(false);
			submitBtnTab4.setEnabled(false);			
		}
		else {
			submitBtnTab1.setEnabled(true);
			submitBtnTab1.getElement().setProperty("title", "Save the Golf Course!");
			
			submitBtnTab2.setEnabled(true);
			submitBtnTab2.getElement().setProperty("title", "Save the Golf Course!");

			submitBtnTab3.setEnabled(true);
			submitBtnTab3.getElement().setProperty("title", "Save the Golf Course!");

			submitBtnTab4.setEnabled(true);	
			submitBtnTab4.getElement().setProperty("title", "Save the Golf Course!");
		}
	}
}
