package com.playertour.backend.vaadin.views.notification;
/**
 * Copyright (c) 2012 - 2021 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
import java.lang.reflect.InvocationTargetException;
import java.util.concurrent.Executors;

import org.gecko.vaadin.databinding.emf.EMFBinder;
import org.gecko.vaadin.databinding.emf.EMFBinderFactory;
import org.gecko.vaadin.whiteboard.annotations.VaadinComponent;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.annotations.ReferenceScope;
import org.osgi.service.component.annotations.ServiceScope;
import org.osgi.service.log.Logger;
import org.osgi.service.log.LoggerFactory;
import org.osgi.util.promise.Promise;
import org.osgi.util.promise.PromiseFactory;

import com.playertour.backend.api.Notification;
import com.playertour.backend.api.PlayerApiFactory;
import com.playertour.backend.api.PlayerApiPackage;
import com.playertour.backend.notifications.persistent.service.api.PersistentNotificationService;
import com.playertour.backend.notifications.templates.service.api.NotificationTemplateService;
import com.playertour.backend.vaadin.views.main.MainView;
import com.vaadin.flow.component.button.Button;
import com.vaadin.flow.component.html.Label;
import com.vaadin.flow.component.orderedlayout.HorizontalLayout;
import com.vaadin.flow.component.orderedlayout.VerticalLayout;
import com.vaadin.flow.component.radiobutton.RadioButtonGroup;
import com.vaadin.flow.component.radiobutton.RadioGroupVariant;
import com.vaadin.flow.component.textfield.TextArea;
import com.vaadin.flow.router.PageTitle;
import com.vaadin.flow.router.Route;

/**
 * 
 * @author ilenia
 * @since May 11, 2021
 */
@Route(value = "notification", layout = MainView.class)
@PageTitle("Notification")
@Component(name = "NotificationView", service = NotificationView.class, scope = ServiceScope.PROTOTYPE)
@VaadinComponent()
public class NotificationView extends VerticalLayout {

	/** serialVersionUID */
	private static final long serialVersionUID = -6309340625378010985L;
	
	private static final String NEW_NOTIFICATION_BTN = "Create New";
	private static final String TEMPLATE_NOTIFICATION_BTN = "Create from Template";	
	
	private PromiseFactory factory = new PromiseFactory(Executors.newFixedThreadPool(4));
	private Logger logger;
	
	private EMFBinder<Notification> notificationBinder;	
	private RadioButtonGroup<String> notCreationRadioGroup;
	private RadioButtonGroup<String> notTemplateRadioGroup;
	private VerticalLayout notBodyLayout;
	private Label notBodyLabel;

	@Reference(scope = ReferenceScope.PROTOTYPE_REQUIRED)
	PersistentNotificationService notificationService;
	
	@Reference
	NotificationTemplateService notificationTemplateService;
	
	@Reference
	private EMFBinderFactory binderFactory;
	
	@Reference(service = LoggerFactory.class)
	void setLogger(Logger logger) {
		this.logger = logger;
	}
	
	
	@Activate
	public void renderView() {

		VerticalLayout vl1 = new VerticalLayout();
		
		notificationBinder = binderFactory.createBinder(PlayerApiPackage.Literals.NOTIFICATION);

		notCreationRadioGroup = new RadioButtonGroup<>();
		notCreationRadioGroup.setLabel("Select the preferred option:");
		notCreationRadioGroup.setItems(NEW_NOTIFICATION_BTN, TEMPLATE_NOTIFICATION_BTN);
		notCreationRadioGroup.addThemeVariants(RadioGroupVariant.LUMO_VERTICAL);
		
		notCreationRadioGroup.addValueChangeListener(event -> { 			
			showNotificationOptions(event.getValue());			
		});
		
		vl1.add(notCreationRadioGroup);
		add(vl1);

		notBodyLayout = new VerticalLayout();
		notBodyLayout.setVisible(false);
		
		notBodyLabel = new Label();
		
		TextArea msg = new TextArea();
		msg.setLabel("Notification Message");
		msg.setWidthFull();
		notificationBinder.bind(msg, PlayerApiPackage.Literals.NOTIFICATION__MESSAGE);
		
		HorizontalLayout hl3 = new HorizontalLayout();
		Button sendBtn = new Button("Send");
		sendBtn.addClickListener(event -> {
			
			Promise<Boolean> result = factory.submit(() -> {
				notificationService.saveNotificationForAllPlayers(notificationBinder.getBean());
				return true;
			}).onSuccess(t -> logger.info("Notification has been sent to all Players!")).onFailure(t -> t.printStackTrace());
			try {
				result.getValue();
				com.vaadin.flow.component.notification.Notification.show("Notification has been sent to all Players!");
			} catch (InvocationTargetException | InterruptedException e) {
				com.vaadin.flow.component.notification.Notification.show("Error occurred when sending the Notification!");
			}
			
			
		});
		
		Button clearBtn = new Button("Clear");
		clearBtn.addClickListener(event -> {
			clearNotificationBody();
		});
		hl3.add(sendBtn, clearBtn);
		
		notBodyLayout.add(notBodyLabel, msg, hl3);
		add(notBodyLayout);
	}
	
	
	private void clearNotificationBody() {
		if(NEW_NOTIFICATION_BTN.equals(notCreationRadioGroup.getValue())) {
			showNotificationTemplate(NEW_NOTIFICATION_BTN);
		}
		else {
			showNotificationTemplate(notTemplateRadioGroup.getValue());
		}
	}

	private void showNotificationOptions(String value) {
		
		switch(value) {
		case NEW_NOTIFICATION_BTN:
			if(notTemplateRadioGroup != null) {
				notTemplateRadioGroup.setVisible(false);
			}
			showNotificationTemplate(value);
			break;
		case TEMPLATE_NOTIFICATION_BTN:
			notBodyLayout.setVisible(false);
			showNotificationTemplateOptions();
			break;
		}
	}

	
	private void showNotificationTemplateOptions() {
		if(notTemplateRadioGroup == null) {
			notTemplateRadioGroup = new RadioButtonGroup<>();
			notTemplateRadioGroup.setLabel("Select a Notification Template:");
			notTemplateRadioGroup.setItems(NotificationTemplateService.SCHEDULED_MAINTENANCE_TEMPLATE,
					NotificationTemplateService.TOURNAMENT_ANNOUNCEMENT_TEMPLATE);
			notTemplateRadioGroup.addThemeVariants(RadioGroupVariant.LUMO_VERTICAL);
			notCreationRadioGroup.add(notTemplateRadioGroup);
			notTemplateRadioGroup.addValueChangeListener(event -> { 
				
				showNotificationTemplate(event.getValue());
				
			});
		}
		else {
			notTemplateRadioGroup.setVisible(true);
		}
		
	}

	
	private void showNotificationTemplate(String value) {
		
		Notification notification = null;
		
		switch(value) {
		case NEW_NOTIFICATION_BTN:
			notification = PlayerApiFactory.eINSTANCE.createNotification();
			break;
		case NotificationTemplateService.SCHEDULED_MAINTENANCE_TEMPLATE:
		case NotificationTemplateService.TOURNAMENT_ANNOUNCEMENT_TEMPLATE:			
			notification = notificationTemplateService.getNotificationByTemplate(value);
			break;
		
		}
		if(notification != null) {
			notBodyLabel.setText(value);
			notificationBinder.readBean(notification);
			notificationBinder.setBean(notification);
			notBodyLayout.setVisible(true);
		}
	}
}
