package com.playertour.backend.vaadin.views.users;
import java.lang.reflect.InvocationTargetException;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

import org.eclipse.emf.ecore.EObject;
/**
 * Copyright (c) 2012 - 2021 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
import org.gecko.vaadin.whiteboard.annotations.VaadinComponent;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.annotations.ReferenceScope;
import org.osgi.service.component.annotations.ServiceScope;
import org.osgi.util.pushstream.PushStream;

import com.playertour.backend.apis.player.PlayerSearchService;
import com.playertour.backend.apis.player.PlayerService;
import com.playertour.backend.meritpoints.service.api.MeritPointsService;
import com.playertour.backend.player.model.player.Player;
import com.playertour.backend.vaadin.views.main.MainView;
import com.vaadin.flow.component.button.Button;
import com.vaadin.flow.component.button.ButtonVariant;
import com.vaadin.flow.component.grid.Grid;
import com.vaadin.flow.component.grid.Grid.Column;
import com.vaadin.flow.component.grid.editor.Editor;
import com.vaadin.flow.component.html.Label;
import com.vaadin.flow.component.icon.VaadinIcon;
import com.vaadin.flow.component.notification.Notification;
import com.vaadin.flow.component.notification.NotificationVariant;
import com.vaadin.flow.component.orderedlayout.HorizontalLayout;
import com.vaadin.flow.component.orderedlayout.VerticalLayout;
import com.vaadin.flow.component.radiobutton.RadioButtonGroup;
import com.vaadin.flow.component.radiobutton.RadioGroupVariant;
import com.vaadin.flow.component.textfield.IntegerField;
import com.vaadin.flow.component.textfield.TextField;
import com.vaadin.flow.data.binder.Binder;
import com.vaadin.flow.data.renderer.ComponentRenderer;
import com.vaadin.flow.router.PageTitle;
import com.vaadin.flow.router.Route;

/**
 * 
 * @author ilenia, mhs
 * @since Apr 8, 2021
 */
@Route(value = "users", layout = MainView.class)
@PageTitle("Users")
@Component(name = "UserView", service = UserView.class, scope = ServiceScope.PROTOTYPE)
@VaadinComponent()
public class UserView extends VerticalLayout {

	@Reference
	PlayerSearchService playerSearchService;
	
	@Reference(scope = ReferenceScope.PROTOTYPE_REQUIRED)
	PlayerService playerService;
	
	@Reference(scope = ReferenceScope.PROTOTYPE_REQUIRED)
	MeritPointsService meritPointsService;
	
	/** serialVersionUID */
	private static final long serialVersionUID = -6505076369742209512L;
	
	private static final String GRID_COLUMN_HEADER_USERNAME = "Username";
	private static final String GRID_COLUMN_HEADER_COUNTRY = "Country";
	private static final String GRID_COLUMN_HEADER_MERIT_POINTS = "Merit Points";
	private static final String GRID_COLUMN_HEADER_EDIT = "Edit";
	private static final String GRID_COLUMN_KEY_USERNAME = "user-username-column";
	private static final String GRID_COLUMN_KEY_COUNTRY = "user-country-column";
	private static final String GRID_COLUMN_KEY_MERIT_POINTS = "user-merit-points-column";
	private static final String GRID_COLUMN_KEY_EDIT = "user-edit-column";
	
	private static final String MERIT_POINT_DEPOSIT_DESCRIPTION = "Merit points added by admin";
	private static final String MERIT_POINT_WITHDRAW_DESCRIPTION = "Merit points withdrawn by admin";
	
	private static final String VIEW_ALL_USERS_OPTION = "View all users";
	private static final String SEARCH_USER = "Search users";
	
	
	private Grid<DisplayedUser> usersGrid = new Grid<>();

	@Activate
	public void renderView() {
		
		setClassName("users-view");		
		setSizeFull();

		VerticalLayout mainLayout = new VerticalLayout();
		mainLayout.setSizeFull();

		RadioButtonGroup<String>  usersOptionsRadioGroup = new RadioButtonGroup<>();
		usersOptionsRadioGroup.setLabel("Select the preferred option:");
		usersOptionsRadioGroup.setItems(VIEW_ALL_USERS_OPTION, SEARCH_USER);
		usersOptionsRadioGroup.addThemeVariants(RadioGroupVariant.LUMO_VERTICAL);
		
		HorizontalLayout searchLayout = new HorizontalLayout();
		searchLayout.setVisible(false);
		
		Label searchLabel = new Label("Search Users by username:");
		TextField searchText = new TextField();
		Button searchButton = new Button("Search", evt -> {
			try {
				List<Player> players = playerSearchService.searchPlayersByUsername(searchText.getValue(), Integer.MAX_VALUE);
				Notification.show(players.size() + " match(es) found").addThemeVariants(NotificationVariant.LUMO_SUCCESS);
				displayPlayers(players);
			} catch(Exception e) {
				Notification.show("Exception when searching for matching players in db!").addThemeVariants(NotificationVariant.LUMO_ERROR);
			}
			
		});
		searchLayout.add(searchLabel, searchText, searchButton);
		
		setupGrid();		
		
		usersOptionsRadioGroup.addValueChangeListener(evt -> {
			usersGrid.setVisible(false);
			switch(evt.getValue()) {
			case VIEW_ALL_USERS_OPTION:
				searchLayout.setVisible(false);
				PushStream<EObject> playersPushStream = playerService.getPlayers();
				if(playersPushStream == null) {
					Notification.show("Null PushStream when searching for all players in db!").addThemeVariants(NotificationVariant.LUMO_ERROR);
				}
				else {
					displayPlayers(playersPushStream);
				}
				return;
			case SEARCH_USER:
				searchText.setValue("");
				searchLayout.setVisible(true);
				return;
			default:
				return;
			}
			
		});		
		mainLayout.add(usersOptionsRadioGroup, searchLayout, usersGrid);
		add(mainLayout);
	}

	private void setupGrid() {
		
		usersGrid.setVisible(false);
		usersGrid.setSizeFull();
		usersGrid.addColumn(DisplayedUser::getUsername).setAutoWidth(true)
			.setHeader(GRID_COLUMN_HEADER_USERNAME).setKey(GRID_COLUMN_KEY_USERNAME);
		usersGrid.addColumn(DisplayedUser::getCountry).setAutoWidth(true)
		.setHeader(GRID_COLUMN_HEADER_COUNTRY).setKey(GRID_COLUMN_KEY_COUNTRY);
		usersGrid.addColumn(DisplayedUser::getMeritPoints).setAutoWidth(true)
		.setHeader(GRID_COLUMN_HEADER_MERIT_POINTS).setKey(GRID_COLUMN_KEY_MERIT_POINTS);

		setGridEditor();
	}
	
	private void setGridEditor() {
		Editor<DisplayedUser> gridEditor = usersGrid.getEditor();
		Column<DisplayedUser> editColumn = usersGrid.addColumn(new ComponentRenderer<>(user -> {
			Button editButton = new Button("Edit");
			editButton.addClickListener(e -> {
				if (gridEditor.isOpen()) {
					gridEditor.cancel();
				}
				usersGrid.getEditor().editItem(user);
			});
			return editButton;
		})).setAutoWidth(true).setHeader(GRID_COLUMN_HEADER_EDIT).setKey(GRID_COLUMN_KEY_EDIT);

		Binder<DisplayedUser> binder = new Binder<>(DisplayedUser.class);
		gridEditor.setBinder(binder);
		gridEditor.setBuffered(true);

		IntegerField meritPointsField = new IntegerField();
		meritPointsField.setWidthFull();
		binder.forField(meritPointsField)
		.withValidator(value -> value != null && value >= 0, "The user merit points cannot be a negative number!")
		.bind(DisplayedUser::getMeritPoints, DisplayedUser::setMeritPoints);
		usersGrid.getColumnByKey(GRID_COLUMN_KEY_MERIT_POINTS).setEditorComponent(meritPointsField);	

		Button saveButton = new Button("Save", e -> gridEditor.save());
		Button cancelButton = new Button(VaadinIcon.CLOSE.create(),
				e -> gridEditor.cancel());
		cancelButton.addThemeVariants(ButtonVariant.LUMO_ICON,
				ButtonVariant.LUMO_ERROR);
		HorizontalLayout actions = new HorizontalLayout(saveButton,
				cancelButton);
		actions.setPadding(false);
		editColumn.setEditorComponent(actions);
		
		gridEditor.addSaveListener(event -> {
			DisplayedUser item = event.getItem();
			Integer meritPointsBefore = item.getMeritPointsBeforeEdit();
			Integer meritPointsNow = item.getMeritPoints();
			int diff = meritPointsNow - meritPointsBefore;
			if(diff > 0) {
				meritPointsService.deposit(item.getLoginId(), new BigDecimal(BigInteger.valueOf(diff)), MERIT_POINT_DEPOSIT_DESCRIPTION);
			}
			else if(diff < 0) {
				meritPointsService.withdraw(item.getLoginId(), new BigDecimal(BigInteger.valueOf(-diff)), MERIT_POINT_WITHDRAW_DESCRIPTION);
			}
			gridEditor.refresh();
		});
	}
	
	private void displayPlayers(List<Player> players) {
		usersGrid.setItems(Collections.emptyList());
		List<DisplayedUser> displayedUsers = players.stream()
				.map(p -> new DisplayedUser(p.getLoginId(), p.getProfile().getLoginName(), p.getProfile().getCountry(), meritPointsService.getBalance(p.getLoginId())))
				.collect(Collectors.toList());
		usersGrid.setItems(displayedUsers);		
		usersGrid.setVisible(true);
	}
	
	private void displayPlayers(PushStream<EObject> playerPushStream) {
		usersGrid.setItems(Collections.emptyList());
		try {
			List<DisplayedUser> displayedUsers = playerPushStream.map(eo -> (Player) eo)
					.map(p -> new DisplayedUser(p.getLoginId(), p.getProfile().getLoginName(), p.getProfile().getCountry(), meritPointsService.getBalance(p.getLoginId())))
					.collect(Collectors.toList()).getValue();
			usersGrid.setItems(displayedUsers);	
			usersGrid.setVisible(true);
		} catch (InvocationTargetException | InterruptedException e) {
			Notification.show("Exception while reading existing players in db!").addThemeVariants(NotificationVariant.LUMO_ERROR);
		}
	}

}
