package com.playertour.backend.vaadin.views.validation;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.util.Collections;
import java.util.Comparator;
import java.util.LinkedList;
import java.util.List;
import java.util.concurrent.Callable;

import org.gecko.vaadin.whiteboard.annotations.VaadinComponent;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.annotations.ReferenceScope;
import org.osgi.service.component.annotations.ServiceScope;

import com.playertour.backend.apis.course.CourseValidationReportService;
import com.playertour.backend.apis.course.CourseValidationService;
import com.playertour.backend.apis.course.ValidationSummaryByCountry;
import com.playertour.backend.apis.igolf.IGolfCountryService;
import com.playertour.backend.apis.igolf.IGolfStateService;
import com.playertour.backend.apis.igolf.exceptions.IGolfUpdateException;
import com.playertour.backend.apis.progress.monitor.ProgressMonitor;
import com.playertour.backend.igolf.model.igolf.CountryInfo;
import com.playertour.backend.igolf.model.igolf.StateInfo;
import com.playertour.backend.vaadin.helper.UiUpdateThread;
import com.playertour.backend.vaadin.helper.VaadinViewProgressMonitor;
import com.playertour.backend.vaadin.views.main.MainView;
import com.vaadin.flow.component.UI;
import com.vaadin.flow.component.button.Button;
import com.vaadin.flow.component.checkbox.Checkbox;
import com.vaadin.flow.component.combobox.ComboBox;
import com.vaadin.flow.component.combobox.ComboBox.ItemFilter;
import com.vaadin.flow.component.dependency.CssImport;
import com.vaadin.flow.component.grid.Grid;
import com.vaadin.flow.component.html.Anchor;
import com.vaadin.flow.component.html.Label;
import com.vaadin.flow.component.notification.Notification;
import com.vaadin.flow.component.notification.NotificationVariant;
import com.vaadin.flow.component.orderedlayout.HorizontalLayout;
import com.vaadin.flow.component.orderedlayout.VerticalLayout;
import com.vaadin.flow.component.progressbar.ProgressBar;
import com.vaadin.flow.component.progressbar.ProgressBarVariant;
import com.vaadin.flow.component.radiobutton.RadioButtonGroup;
import com.vaadin.flow.component.radiobutton.RadioGroupVariant;
import com.vaadin.flow.router.PageTitle;
import com.vaadin.flow.router.Route;
import com.vaadin.flow.server.Command;
import com.vaadin.flow.server.StreamResource;


/**
 *
 * @author ilenia
 * @since Jun 23, 2021
 */
@Route(value = "validate", layout = MainView.class)
@PageTitle("Validate")
@Component(name = "ValidationView", service = ValidationView.class, scope = ServiceScope.PROTOTYPE)
@VaadinComponent()
@CssImport(themeFor = "vaadin-grid", value = "./themes/wbapp/views/validation-view.css")
public class ValidationView extends VerticalLayout {

	@Reference(target = "(component.name=IGolfDBCountryService)", scope = ReferenceScope.PROTOTYPE_REQUIRED)
	private IGolfCountryService iGolfDBCountryService;
	
	@Reference(target = "(component.name=IGolfDBStateService)", scope = ReferenceScope.PROTOTYPE_REQUIRED)
	private IGolfStateService iGolfDBStateService;
	
	@Reference(scope = ReferenceScope.PROTOTYPE_REQUIRED)
	CourseValidationReportService courseValidationReportService;
	
	@Reference(scope = ReferenceScope.PROTOTYPE_REQUIRED)
	CourseValidationService courseValidationService;

	/** serialVersionUID */
	private static final long serialVersionUID = -8524058762022841725L;

	private static final String LAST_REPORT_BTN = "View last validation report";
	private static final String NEW_REPORT_BTN = "Revalidate and create a new validation report";
	private static final String VALIDATION_SUMMARY_BTN = "View validation summary";
	private static final String OPTION_REPORT_ALL = "All courses";
	private static final String OPTION_REPORT_BY_COUNTRY = "By country";
	private static final String OPTION_REPORT_BY_STATE = "By state";

	private RadioButtonGroup<String> reportOptionsGroup = new RadioButtonGroup<String>();
	private Anchor lastReportAllAnchor = new Anchor();
	private Anchor lastReportByCountryAnchor = new Anchor();
	private Anchor lastReportByStateAnchor = new Anchor();
	private Anchor newReportAllAnchor = new Anchor();
	private Anchor newReportByCountryAnchor = new Anchor();
	private Anchor newReportByStateAnchor = new Anchor();
	private Button validateBtn = new Button("Validate");

	List<CountryInfo> comboBoxCountryContent;
	List<StateInfo> comboBoxStateContent;

	@Activate
	public void renderView() {
		setClassName("validation-view");		
		setSizeFull();

		VerticalLayout layout = new VerticalLayout();
		layout.setSizeFull();

		RadioButtonGroup<String>  validateOptionsRadioGroup = new RadioButtonGroup<>();
		validateOptionsRadioGroup.setLabel("Select the preferred option:");
		validateOptionsRadioGroup.setItems(NEW_REPORT_BTN, LAST_REPORT_BTN, VALIDATION_SUMMARY_BTN);
		validateOptionsRadioGroup.addThemeVariants(RadioGroupVariant.LUMO_VERTICAL);

		VerticalLayout reportOptionsLayout = new VerticalLayout();
		reportOptionsLayout.setSizeFull();
		reportOptionsLayout.setVisible(false);
		
		Checkbox forceValidationCheckbox = new Checkbox("Force New Validation");
		forceValidationCheckbox.setVisible(false);

		reportOptionsGroup.addThemeVariants(RadioGroupVariant.LUMO_VERTICAL);

		ComboBox<CountryInfo> countriesComboBox = new ComboBox<CountryInfo>();
		countriesComboBox.setHeightFull();
		countriesComboBox.setWidth("40%");
		countriesComboBox.setVisible(false);
		countriesComboBox.setPlaceholder("Select Country");
		countriesComboBox.setItemLabelGenerator(ci -> ci.getCountryFull());
		updateCountryInfoList();
		ItemFilter<CountryInfo> countryFilter = (ci, filterString) -> ci.getCountryFull().toLowerCase().contains(filterString.toLowerCase());
		countriesComboBox.setItems(countryFilter, comboBoxCountryContent);
		
		ComboBox<StateInfo> statesComboBox = new ComboBox<StateInfo>();
		statesComboBox.setHeightFull();
		statesComboBox.setWidth("40%");
		statesComboBox.setVisible(false);
		statesComboBox.setPlaceholder("Select State");
		statesComboBox.setItemLabelGenerator(ci -> ci.getStateFull());
		ItemFilter<StateInfo> stateFilter = (ci, filterString) -> ci.getStateFull().toLowerCase().contains(filterString.toLowerCase());
		
		Grid<DisplayedSummary> validationSummaryGrid = new Grid<>();
		validationSummaryGrid.setVisible(false);
		validationSummaryGrid.setSizeFull();
		validationSummaryGrid.addColumn(DisplayedSummary::getCountryName).setHeader("Country").setKey("country-column").setVisible(false);
		validationSummaryGrid.addColumn(DisplayedSummary::getStateName).setHeader("State").setKey("state-column").setVisible(false);
		validationSummaryGrid.addColumn(DisplayedSummary::getNumValidatedCourses).setHeader("# Validated Courses");
		validationSummaryGrid.addColumn(DisplayedSummary::getNumNotValidatedCourses).setHeader("# NOT Validated Courses");
		validationSummaryGrid.addColumn(DisplayedSummary::getNumInvalid).setHeader("# Invalid Courses");
		validationSummaryGrid.addColumn(DisplayedSummary::getNumTotallyInvalid).setHeader("# Totally Invalid Courses");
		validationSummaryGrid.setClassNameGenerator(vs -> vs.getNumNotValidatedCourses() > 0 ? "to-be-validated" : null);


		validateOptionsRadioGroup.addValueChangeListener(evt -> {
			removeAnchors();
			reportOptionsGroup.setValue(null);
			reportOptionsLayout.setVisible(true);
			validationSummaryGrid.setVisible(false);
			validationSummaryGrid.setItems(Collections.emptyList());
			validateBtn.setVisible(false);
			countriesComboBox.setVisible(false);
			countriesComboBox.setValue(null);
			statesComboBox.setVisible(false);
			statesComboBox.setValue(null);
			if(VALIDATION_SUMMARY_BTN.equals(evt.getValue())) {
				reportOptionsGroup.setItems(OPTION_REPORT_ALL, OPTION_REPORT_BY_COUNTRY);
			}
			else {
				reportOptionsGroup.setItems(OPTION_REPORT_ALL, OPTION_REPORT_BY_COUNTRY, OPTION_REPORT_BY_STATE);
			}
			if(NEW_REPORT_BTN.equals(evt.getValue())) {
				forceValidationCheckbox.setVisible(true);
			}
			else {
				forceValidationCheckbox.setVisible(false);
			}
		});

		lastReportAllAnchor.getElement().setAttribute("download", true);
		lastReportAllAnchor.setTarget("_blank"); //to open it in a new tab
		lastReportAllAnchor.setVisible(false);
		reportOptionsGroup.add(lastReportAllAnchor);

		lastReportByCountryAnchor.getElement().setAttribute("download", true);
		lastReportByCountryAnchor.setTarget("_blank"); //to open it in a new tab
		lastReportByCountryAnchor.setVisible(false);
		reportOptionsGroup.add(lastReportByCountryAnchor);
		
		lastReportByStateAnchor.getElement().setAttribute("download", true);
		lastReportByStateAnchor.setTarget("_blank"); //to open it in a new tab
		lastReportByStateAnchor.setVisible(false);
		reportOptionsGroup.add(lastReportByStateAnchor);

		newReportAllAnchor.getElement().setAttribute("download", true);
		newReportAllAnchor.setTarget("_blank"); //to open it in a new tab
		newReportAllAnchor.setVisible(false);
		reportOptionsGroup.add(newReportAllAnchor);

		newReportByCountryAnchor.getElement().setAttribute("download", true);
		newReportByCountryAnchor.setTarget("_blank"); //to open it in a new tab
		newReportByCountryAnchor.setVisible(false);
		reportOptionsGroup.add(newReportByCountryAnchor);
		
		newReportByStateAnchor.getElement().setAttribute("download", true);
		newReportByStateAnchor.setTarget("_blank"); //to open it in a new tab
		newReportByStateAnchor.setVisible(false);
		reportOptionsGroup.add(newReportByStateAnchor);

		HorizontalLayout comboLayout = new HorizontalLayout();
		comboLayout.setHeight("10%");
		comboLayout.setWidthFull();
		comboLayout.setVisible(false);		
		validateBtn.setVisible(false);
		validateBtn.setEnabled(false);
		comboLayout.add(countriesComboBox, statesComboBox, validateBtn);		
		
		reportOptionsLayout.add(forceValidationCheckbox, reportOptionsGroup, comboLayout, validationSummaryGrid);

		reportOptionsGroup.addValueChangeListener(evt -> {
			removeAnchors();
			if(evt.getValue() == null) {
				comboLayout.setVisible(false);
				return;
			}
			comboLayout.setVisible(true);
			validateBtn.setVisible(true);
			switch(validateOptionsRadioGroup.getValue()) {
			case LAST_REPORT_BTN:
				validateBtn.setText("Search");
				break;
			case NEW_REPORT_BTN:
				validateBtn.setText("Validate");
				break;
			case VALIDATION_SUMMARY_BTN:
				validateBtn.setText("View");
				break;
			}

			switch(evt.getValue()) {
			case OPTION_REPORT_ALL:		
				countriesComboBox.setVisible(false);
				statesComboBox.setVisible(false);
				validateBtn.setEnabled(true);
				break;
			case OPTION_REPORT_BY_COUNTRY:
				countriesComboBox.setVisible(true);
				countriesComboBox.setValue(null);
				statesComboBox.setVisible(false);
				validateBtn.setEnabled(false);
				break;
			case OPTION_REPORT_BY_STATE:
				countriesComboBox.setVisible(true);
				countriesComboBox.setValue(null);				
				validateBtn.setEnabled(false);
				break;
			}
		});

		countriesComboBox.addValueChangeListener(evt -> {
			if(evt.getValue() == null) {
				return;
			}
			if(OPTION_REPORT_BY_STATE.equals(reportOptionsGroup.getValue())) {
				statesComboBox.setVisible(true);
				updateStateInfoList(evt.getValue().getId_country().toString());
				statesComboBox.setItems(stateFilter, comboBoxStateContent);
			}
			else {
				validateBtn.setEnabled(true);
			}
		});
		
		statesComboBox.addValueChangeListener(evt -> {
			if(evt.getValue() == null) {
				return;
			}
			validateBtn.setEnabled(true);
		});

		HorizontalLayout validateBarLayout = new HorizontalLayout();
		validateBarLayout.setWidthFull();
		validateBarLayout.setHeight("20%");
		validateBarLayout.setVisible(false);

		ProgressBar validateProgressBar = new ProgressBar(0., 100.);
		validateProgressBar.setIndeterminate(true);
		validateProgressBar.setVisible(true);
		validateProgressBar.setHeight("20%");
		validateProgressBar.setWidth("40%");
		validateProgressBar.addThemeVariants(ProgressBarVariant.LUMO_CONTRAST);

		Label validateBarLabel = new Label();
		validateBarLabel.setSizeFull();
		validateBarLabel.setVisible(true);
		validateBarLayout.add(validateProgressBar, validateBarLabel);

		VaadinViewProgressMonitor validateProgressMonitor = 
				new VaadinViewProgressMonitor(validateProgressBar, validateBarLabel, UI.getCurrent());

		validateBtn.addClickListener(evt -> {
			removeAnchors();
			switch(validateOptionsRadioGroup.getValue()) {
			case LAST_REPORT_BTN:
				switch(reportOptionsGroup.getValue()) {
				case OPTION_REPORT_ALL:
					//						retrieve last report for all courses
					File lastReport = findLastReport();
					if(lastReport != null) {
						StreamResource res = createReportResource(lastReport);
						lastReportAllAnchor.setText(lastReport.getName());
						lastReportAllAnchor.setHref(res);
						lastReportAllAnchor.setVisible(true);
						reportOptionsGroup.add(lastReportAllAnchor);
					} else {
						Notification.show("No Validation Report available! Please, trigger a new Validation!");
					}
					break;
				case OPTION_REPORT_BY_COUNTRY:
					//						retrieve last report for country id get from combo box
					CountryInfo countryInfo = countriesComboBox.getValue();
					File lastReportByCountry = findLastReportByCountryId(countryInfo.getId_country().toString());
					if(lastReportByCountry != null) {
						StreamResource res = createReportResource(lastReportByCountry);
						lastReportByCountryAnchor.setText(lastReportByCountry.getName());
						lastReportByCountryAnchor.setHref(res);
						lastReportByCountryAnchor.setVisible(true);
						reportOptionsGroup.add(lastReportByCountryAnchor);
					} else {
						Notification.show("No Validation Report available for Country " + countryInfo.getCountryFull() + "! Please, trigger a new Validation!");
					}
					countriesComboBox.setValue(null);
					validateBtn.setEnabled(false);
					break;
				case OPTION_REPORT_BY_STATE:
					StateInfo stateInfo = statesComboBox.getValue();
					File lastReportByState = findLastReportByStateId(stateInfo.getId_state().toString());
					if(lastReportByState != null) {
						StreamResource res = createReportResource(lastReportByState);
						lastReportByStateAnchor.setText(lastReportByState.getName());
						lastReportByStateAnchor.setHref(res);
						lastReportByStateAnchor.setVisible(true);
						reportOptionsGroup.add(lastReportByStateAnchor);
					} else {
						Notification.show("No Validation Report available for State " + stateInfo.getStateFull() + "! Please, trigger a new Validation!");
					}
					countriesComboBox.setValue(null);
					statesComboBox.setValue(null);
					validateBtn.setEnabled(false);
					break;
				}
				break;
			case NEW_REPORT_BTN:
				courseValidationReportService.setForceValidation(forceValidationCheckbox.getValue());
				validateBarLayout.setVisible(true);
				Callable<Void> call = null;
				UiUpdateThread updateThread = null;
				UI ui = UI.getCurrent();
				ui.setPollInterval(500);		
				switch(reportOptionsGroup.getValue()) {
				case OPTION_REPORT_ALL:
					//					trigger new validation for all courses
					call = new Callable<Void>() {
						@Override
						public Void call() throws Exception {				
							File f = performValidation(null, null, validateProgressMonitor);
							ui.access(new Command() {
								/** serialVersionUID */
								private static final long serialVersionUID = 7262570483927202816L;
								@Override
								public void execute() {
									StreamResource r = createReportResource(f);
									newReportAllAnchor.setText(f.getName());
									newReportAllAnchor.setHref(r);
									newReportAllAnchor.setVisible(true);
									reportOptionsGroup.add(newReportAllAnchor);
								}
							});	
							return null;
						}
					};								
					break;
				case OPTION_REPORT_BY_COUNTRY:
					//					trigger new validation by country
					CountryInfo countryInfo = countriesComboBox.getValue();					
					call = new Callable<Void>() {
						@Override
						public Void call() throws Exception {													
							File f = performValidation(countryInfo.getId_country().toString(), null, validateProgressMonitor);							
							ui.access(new Command() {
								/** serialVersionUID */
								private static final long serialVersionUID = 7262570483927202816L;
								@Override
								public void execute() {
									StreamResource r = createReportResource(f);
									newReportByCountryAnchor.setText(f.getName());
									newReportByCountryAnchor.setHref(r);
									newReportByCountryAnchor.setVisible(true);
									reportOptionsGroup.add(newReportByCountryAnchor);
									countriesComboBox.setValue(null);
									validateBtn.setEnabled(false);
								}
							});	
							return null;
						}
					};					
					break;
				case OPTION_REPORT_BY_STATE:
					StateInfo stateInfo = statesComboBox.getValue();
					call = new Callable<Void>() {
						@Override
						public Void call() throws Exception {													
							File f = performValidation(null, stateInfo.getId_state().toString(), validateProgressMonitor);							
							ui.access(new Command() {
								/** serialVersionUID */
								private static final long serialVersionUID = 7262570483927202816L;
								@Override
								public void execute() {
									StreamResource r = createReportResource(f);
									newReportByStateAnchor.setText(f.getName());
									newReportByStateAnchor.setHref(r);
									newReportByStateAnchor.setVisible(true);
									reportOptionsGroup.add(newReportByStateAnchor);
									countriesComboBox.setValue(null);
									statesComboBox.setValue(null);
									validateBtn.setEnabled(false);
								}
							});	
							return null;
						}
					};				
					break;
				}	
				updateThread = new UiUpdateThread(call, ui, validateProgressBar, validateProgressMonitor, validateBtn, validateBarLabel);
				updateThread.start();	
				break;
			case VALIDATION_SUMMARY_BTN:
				List<DisplayedSummary> displayedSummaries = new LinkedList<>();
				switch(reportOptionsGroup.getValue()) {
				case OPTION_REPORT_ALL:					
					comboBoxCountryContent.stream().forEach(ci -> {
						ValidationSummaryByCountry validationSummary = courseValidationService.getValidationSummaryByCountryId(ci.getId_country().toString(), false);
						if(validationSummary != null) {
							DisplayedSummary displayedSummary = new DisplayedSummary();
							displayedSummary.setCountryName(validationSummary.getCountryName());
							displayedSummary.setNumInvalid(validationSummary.getNumInvalid());
							displayedSummary.setNumTotallyInvalid(validationSummary.getNumTotallyInvalid());
							displayedSummary.setNumNotValidatedCourses(validationSummary.getNumNotValidatedCourses());
							displayedSummary.setNumValidatedCourses(validationSummary.getNumValidatedCourses());
							displayedSummaries.add(displayedSummary);
						}						
					});
					validationSummaryGrid.getColumnByKey("state-column").setVisible(false);
					validationSummaryGrid.getColumnByKey("country-column").setVisible(true);
					validationSummaryGrid.setVisible(true);
					displayedSummaries.sort(Comparator.comparing(DisplayedSummary::getNumNotValidatedCourses).reversed().thenComparing(Comparator.comparing(DisplayedSummary::getCountryName)));
					validationSummaryGrid.setItems(displayedSummaries);
					break;
				case OPTION_REPORT_BY_COUNTRY:
					CountryInfo countryInfo = countriesComboBox.getValue();
					ValidationSummaryByCountry validationSummary = courseValidationService.getValidationSummaryByCountryId(countryInfo.getId_country().toString(), true);					
					if(validationSummary != null) {
						validationSummary.getSummariesByState().forEach((k,v) -> {
							DisplayedSummary displayedSummary = new DisplayedSummary();
							displayedSummary.setStateName(v.getStateName());
							displayedSummary.setNumInvalid(v.getNumInvalid());
							displayedSummary.setNumTotallyInvalid(v.getNumTotallyInvalid());
							displayedSummary.setNumNotValidatedCourses(v.getNumNotValidatedCourses());
							displayedSummary.setNumValidatedCourses(v.getNumValidatedCourses());
							displayedSummaries.add(displayedSummary);
						});
						validationSummaryGrid.getColumnByKey("state-column").setVisible(true);
						validationSummaryGrid.getColumnByKey("country-column").setVisible(false);
						displayedSummaries.sort(Comparator.comparing(DisplayedSummary::getNumNotValidatedCourses).reversed().thenComparing(Comparator.comparing(DisplayedSummary::getStateName)));
						validationSummaryGrid.setItems(displayedSummaries);
						validationSummaryGrid.setVisible(true);
					}
					else {
						Notification.show("No courses for country " + countryInfo.getCountryFull() + " in db!").addThemeVariants(NotificationVariant.LUMO_ERROR);
					}					
					break;
				}
				break;
			}			
		});

		validateBarLayout.add(validateProgressBar, validateBarLabel);
		reportOptionsLayout.add(validateBarLayout);
		layout.add(validateOptionsRadioGroup, reportOptionsLayout);
		add(layout);
	}
	
	private void updateStateInfoList(String countryId) {
		try {
			comboBoxStateContent = iGolfDBStateService.getStateInfoByCountry(countryId);
			if(comboBoxStateContent.isEmpty()) {
				Notification.show("There are no States for Country " + countryId + ". Please, use the update by country or try to update the state list first!");
			}
		} catch (IGolfUpdateException e) {
			Notification.show("Error occurred when updating StateInfo list " + e.getMessage()).addThemeVariants(NotificationVariant.LUMO_ERROR);
			e.printStackTrace();
		}		
	}

	private void updateCountryInfoList() {
		try {
			comboBoxCountryContent = iGolfDBCountryService.getAllCountryInfo();
		} catch (IGolfUpdateException e) {
			Notification.show("Error updating country list from db!").addThemeVariants(NotificationVariant.LUMO_ERROR);
		}		
	}

	private File findLastReport() {
		File lastReport = courseValidationReportService.getLastReport();
		return lastReport;
	}

	private File findLastReportByCountryId(String countryId) {
		File lastReport = courseValidationReportService.getLastReportByCountryId(countryId);
		return lastReport;
	}
	
	private File findLastReportByStateId(String stateId) {
		File lastReport = courseValidationReportService.getLastReportByStateId(stateId);
		return lastReport;
	}

	private StreamResource createReportResource(File report) {
		StreamResource res = new StreamResource(report.getName(), () -> {
			try {
				return new FileInputStream(report);
			} catch (FileNotFoundException e) {
				throw new RuntimeException(e);
			}
		});
		res.setContentType("text/plain");
		return res;
	}


	private void removeAnchors() {
		if(lastReportAllAnchor.isVisible()) {
			reportOptionsGroup.remove(lastReportAllAnchor);
			lastReportAllAnchor.setVisible(false);
		}
		if(lastReportByCountryAnchor.isVisible()) {
			reportOptionsGroup.remove(lastReportByCountryAnchor);
			lastReportByCountryAnchor.setVisible(false);
		}
		if(newReportAllAnchor.isVisible()) {
			reportOptionsGroup.remove(newReportAllAnchor);
			newReportAllAnchor.setVisible(false);
		}
		if(newReportByCountryAnchor.isVisible()) {
			reportOptionsGroup.remove(newReportByCountryAnchor);
			newReportByCountryAnchor.setVisible(false);
		}
	}

	private File performValidation(String countryId, String stateId, ProgressMonitor progressMonitor) {
		File report;
		if(countryId != null) {
			report = courseValidationReportService.validateAndReportByCountryId(countryId, progressMonitor);
		}
		else if(stateId != null) {
			report = courseValidationReportService.validateAndReportByStateId(stateId, progressMonitor);
		}
		else {
			report = courseValidationReportService.validateAndReportAll(progressMonitor);
		}
		return report;
	}
	
	class DisplayedSummary {
		String countryName;
		String stateName;
		Integer numValidatedCourses = 0;
		Integer numNotValidatedCourses = 0;
		Integer numTotallyInvalid = 0;
		Integer numInvalid = 0;
		
		public String getCountryName() {
			return countryName;
		}
		
		public void setCountryName(String countryName) {
			this.countryName = countryName;
		}
		
		public String getStateName() {
			return stateName;
		}
		
		public void setStateName(String stateName) {
			this.stateName = stateName;
		}
		
		public Integer getNumValidatedCourses() {
			return numValidatedCourses;
		}
		
		public void setNumValidatedCourses(Integer numValidatedCourses) {
			this.numValidatedCourses = numValidatedCourses;
		}
		
		public Integer getNumNotValidatedCourses() {
			return numNotValidatedCourses;
		}
		
		public void setNumNotValidatedCourses(Integer numNotValidatedCourses) {
			this.numNotValidatedCourses = numNotValidatedCourses;
		}
		
		public Integer getNumTotallyInvalid() {
			return numTotallyInvalid;
		}
		
		public void setNumTotallyInvalid(Integer numTotallyInvalid) {
			this.numTotallyInvalid = numTotallyInvalid;
		}
		
		public Integer getNumInvalid() {
			return numInvalid;
		}
		
		public void setNumInvalid(Integer numInvalid) {
			this.numInvalid = numInvalid;
		}
	}
}
