/**
 * Copyright (c) 2012 - 2022 Data In Motion and others.
 * All rights reserved.
 *
 * This program and the accompanying materials are made available under the terms of the
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package de.dim.androidpublisher.tests;

import static org.assertj.core.api.Assertions.assertThatExceptionOfType;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.security.GeneralSecurityException;

import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.junit.platform.commons.annotation.Testable;
import org.osgi.framework.BundleContext;
import org.osgi.framework.FrameworkUtil;
import org.osgi.test.common.annotation.InjectBundleContext;
import org.osgi.test.junit5.context.BundleContextExtension;
import org.osgi.test.junit5.service.ServiceExtension;

import com.google.api.client.googleapis.javanet.GoogleNetHttpTransport;
import com.google.api.client.googleapis.json.GoogleJsonResponseException;
import com.google.api.client.json.gson.GsonFactory;
import com.google.api.services.androidpublisher.AndroidPublisher;
import com.google.api.services.androidpublisher.AndroidPublisher.Inappproducts;
import com.google.api.services.androidpublisher.AndroidPublisherScopes;
import com.google.api.services.androidpublisher.model.InAppProduct;
import com.google.api.services.androidpublisher.model.InappproductsListResponse;
import com.google.api.services.androidpublisher.model.ProductPurchase;
import com.google.auth.http.HttpCredentialsAdapter;
import com.google.auth.oauth2.GoogleCredentials;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;

@Testable
@ExtendWith(BundleContextExtension.class)
@ExtendWith(ServiceExtension.class)
public class AndroidPublisherIntegrationTest {
	
    private static final String APPLICATION_NAME = "Player Amateur Tour";
    private static final String PACKAGE_NAME = "com.theplayertour.app";
    private static final String SERVICE_ACCOUNT_JSON = "/data/pc-api-4732445104444293801-392-9bfa0325d51b.json";
    private static final String PRODUCT_ID = "theplayertour_bundle_1_100";

	@InjectBundleContext
	BundleContext bundleContext;
	
	AndroidPublisher service;
	
	@BeforeEach
	public void setup() {
		Preconditions.checkArgument(!Strings.isNullOrEmpty(SERVICE_ACCOUNT_JSON), "SERVICE_ACCOUNT_JSON cannot be null or empty!");
		Preconditions.checkArgument(!Strings.isNullOrEmpty(PACKAGE_NAME), "PACKAGE_NAME cannot be null or empty!");

        try {
			service = initialize(APPLICATION_NAME);
			
			System.out.println("Application name: " + service.getApplicationName());
			System.out.println("Base URL: " + service.getBaseUrl());
			System.out.println("Root URL: " + service.getRootUrl());
			System.out.println("Service Path: " + service.getServicePath());
			
        } catch (IOException | GeneralSecurityException ex) {
        	ex.printStackTrace();
            System.err.println("Exception was thrown while initializing API service");
        }
	}
	
	@Test
	public void testListInAppProducts() throws IOException {
		Inappproducts inappproducts = service.inappproducts();

		InappproductsListResponse inAppProductsListResponse = inappproducts.list(PACKAGE_NAME).execute();

		java.util.List<InAppProduct> inAppProductsList = inAppProductsListResponse.getInappproduct();
		for (InAppProduct inAppProduct : inAppProductsList) {
			System.out.println(inAppProduct.toPrettyString());
		}
	}
	
	@Test
	public void verifyPurchaseValidToken() {
		String validToken = "mkfjlopddgomngibmndjonbo.AO-J1Oy6MZTsnCLeEGHSXXC5TiQDy6OhASo6IBc3mwWo-NBEBf4zk42_XtltiMe51BJO5HKe1OBRObLt87yQ-q81RcqMefxVCuN8w14I35APQguXd1esi7E";
		
		try {
			
			ProductPurchase productPurchase = service.purchases().products().get(PACKAGE_NAME, PRODUCT_ID, validToken).execute();
			
			/*
			productPurchase.getAcknowledgementState();
			productPurchase.getConsumptionState();
			productPurchase.getDeveloperPayload();
			productPurchase.getOrderId();
			productPurchase.getProductId();
			productPurchase.getPurchaseState();
			productPurchase.getPurchaseTimeMillis();
			productPurchase.getPurchaseToken();
			productPurchase.getPurchaseType();
			productPurchase.getQuantity();
			productPurchase.getRegionCode();
			productPurchase.toPrettyString();
			*/
			
			System.out.println(productPurchase.toPrettyString());
			
		} catch (GoogleJsonResponseException ex) {
			
			System.err.println("Status code: " + ex.getStatusCode());
			System.err.println("Status message: " + ex.getStatusMessage());
			
			System.err.println("Reason: " + ex.getDetails().getErrors().get(0).getReason()); // Reason: invalid
			System.err.println("Message: " + ex.getDetails().getMessage());
			
			ex.printStackTrace();
			
		} catch (IOException ex) {
			ex.printStackTrace();
		}
	}
	
	@Test
	public void verifyPurchaseInvalidToken() {
		String invalidToken = "mkfjlopddgomngibmndjonb0.AO-J1Oy6MZTsnCLeEGHSXXC5TiQDy6OhASo6IBc3mwWo-NBEBf4zk42_XtltiMe51BJO5HKe1OBRObLt87yQ-q81RcqMefxVCuN8w14I35APQguXd1esi7E";
		
		assertThatExceptionOfType(GoogleJsonResponseException.class).isThrownBy(() -> {			
			service.purchases().products().get(PACKAGE_NAME, PRODUCT_ID, invalidToken).execute();
		});
	}	
	
	private AndroidPublisher initialize(String applicationName)
			throws FileNotFoundException, IOException, GeneralSecurityException {
		Preconditions.checkArgument(!Strings.isNullOrEmpty(applicationName),
				"applicationName cannot be null or empty!");

		GoogleCredentials credentials = GoogleCredentials
				.fromStream(FrameworkUtil.getBundle(getClass()).getEntry(SERVICE_ACCOUNT_JSON).openStream())
				.createScoped(AndroidPublisherScopes.ANDROIDPUBLISHER);

		return new AndroidPublisher.Builder(GoogleNetHttpTransport.newTrustedTransport(),
				GsonFactory.getDefaultInstance(), new HttpCredentialsAdapter(credentials))
						.setApplicationName(applicationName).build();
	}
}
