/**
 * Copyright (c) 2012 - 2019 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package de.dim.trafficos.device.api;

import java.util.Date;
import java.util.List;

import org.osgi.annotation.versioning.ProviderType;

import de.dim.trafficos.model.device.DataEntry;
import de.dim.trafficos.model.device.Device;
import de.dim.trafficos.model.device.DeviceConfiguration;
import de.dim.trafficos.model.device.DeviceInfo;
import de.dim.trafficos.model.device.LifeCycleDeviceType;

/**
 * Service to create or update a Device which should be triggered from the REST 
 * 
 * @author Mark Hoffmann
 * @since 30.05.2019
 */
@ProviderType
public interface DevicesService {
	
	public static final String TOPIC_DEVICE_STATUS = "device/status";
	
	/**
	 * @return the list of all active devices, or an empty list, in case no active device was found.
	 */
	public List<Device> getActiveDevices();
	
	/**
	 * @return the list of all active devices, or an empty list, in case no active device was found.
	 */
	public List<Device> getAllDevices();
	
	/**
	 * Returns the LifeCycleDeviceType associated with the Device of provided ID
	 * @param deviceId the ID of the Device whose Status we want to retrieve
	 * @return the LifeCycleDeviceType associated with the Device of provided ID
	 */
	public LifeCycleDeviceType getStatus(String deviceId);
	
	/**
	 * Retrieves from the DB the Device with the given ID. It returns null if the given ID is null or if no Device with
	 * such ID was found.
	 * 
	 * @param deviceId the ID of the device we want to retrieve
	 * @return the Device with the provided ID or null, if none was found or the ID was null
	 */
	public Device getDeviceById(String deviceId);
		
	/**
	 * Updates the given Device. It returns null if the provided Device has a null ID. 
	 * If no Device was found with the provided Device id, then the Device will be saved as a new Device,
	 * otherwise the Device with the same ID will be updated. Once updating a Device the corresponding DeviceService
	 * instance is updated.
	 * 
	 * @param device the updated Device to be saved
	 * @return the updated Device
	 */
	public Device updateDevice(Device device);
	
	/**
	 * Removes the Device with the given ID, if it exists and notifies the EventAdmin, so it can trigger
	 * the removal of the corresponding DeviceService instance.
	 * 
	 * @param deviceId the ID of the Device to be removed
	 */
	public void removeDevice(String deviceId);
	
	/**
	 * Delegates to the <code>DeviceService::updateDeviceConfiguration<code> associated with the Device of provided id
	 * 
	 * @param deviceId the id of the Device whose Configuration has to be updated
	 * @param configuration the new DeviceConfiguration 
	 * @return the updated DeviceConfiguration, or null if the corresponding
	 * DeviceService is not registered
	 */
	public DeviceConfiguration updateDeviceConfiguration(String deviceId, DeviceConfiguration configuration);
	
	/**
	 * Delegates to the <code>DeviceService::updateDeviceInformation<code> associated with the Device of provided id
	 * 
	 * @param deviceId the id of the Device whose DeviceInfo has to be updated
	 * @param configuration the new DeviceInfo 
	 * @return the updated DeviceInfo, or null if the corresponding
	 * DeviceService is not registered
	 */
	public DeviceInfo updateDeviceInformation(String deviceId, DeviceInfo info);
	
	/**
	 * Delegates to the <code>DeviceService::getCurrentConfiguration<code> associated with the Device of provided id
	 * 
	 * @param deviceId the id of the Device whose Configuration we want to know
	 * @return the current DeviceConfiguration associated with the Device of provided id, or null if the corresponding
	 * DeviceService is not registered
	 */
	public DeviceConfiguration getCurrentConfiguration(String deviceId);
	
	/**
	 * Delegates to the <code>DeviceService::getConfigurations<code> associated with the Device of provided id
	 * 
	 * @param deviceId the id of the Device whose Configurations we want to know
	 * @return the list of Configurations associated with the Device of provided id, or null if the corresponding
	 * DeviceService is not registered
	 */
	public List<DeviceConfiguration> getConfigurations(String deviceId);
	
	/**
	 * Delegates to the <code>DeviceService::isRunning<code> associated with the Device of provided id
	 * 
	 * @param deviceId the id of the Device
	 * @return <code>true<code> if the Device is running, <code>false<code> if the associated DeviceService is not 
	 * registered, or if the Device is not running
	 */
	public boolean isRunning(String deviceId);
	
	/**
	 * Delegates to the <code>DeviceService::startDevice<code> associated with the Device of provided id
	 * 
	 * @param deviceId the id of the Device
	 * @throws DeviceException
	 */
	public boolean startDevice(String deviceId) throws DeviceException;
	
	/**
	 * Delegates to the <code>DeviceService::stopDevice<code> associated with the Device of provided id
	 * 
	 * @param deviceId the id of the Device
	 * @throws DeviceException
	 */
	public boolean stopDevice(String deviceId) throws DeviceException;
	
	/**
	 * Delegates to the <code>DeviceService::initialize<code> 
	 * @param deviceId the Device id
	 * @param forceReload a boolean indicating whether a reload is required
	 * @param forceRestart a boolean indicating whether a restart is required
	 * @throws DeviceException
	 */
	public void initializeDevice(String deviceId, boolean forceReload, boolean forceRestart) throws DeviceException;
	
	/**
	 * Delegates to the <code>DeviceService::getDataEntries<code> associated with the Device of provided id
	 * 
	 * @param deviceId the id of the Device
	 * @param startDate the start date
	 * @param endDate the end date
	 * @return the list of DataEntry associated with the Device and for the provided period of time, an empty list
	 * in case no DataEntry has been found, or the dates are inconsistent and null in case the corresponding 
	 * DeviceService is not registered or 
	 */
	public List<DataEntry> getDataEntries(String deviceId, Date startDate, Date endDate);
	
	/**
	 * @param deviceId the Device id
	 * @param timestamp the current Date
	 * @return the DataEntry for the given Device ID and Date, or null, 
	 * if no DataEntry was stored for the provided Device and time, or in case the deviceId is null
	 */
	public DataEntry getCurrentDataEntry(String deviceId, Date timestamp);
	
}
