/**
 * Copyright (c) 2012 - 2019 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package de.dim.trafficos.device.api;

import java.util.Date;
import java.util.List;

import org.osgi.annotation.versioning.ProviderType;

import de.dim.trafficos.model.device.DataEntry;
import de.dim.trafficos.model.device.Device;
import de.dim.trafficos.model.device.DeviceConfiguration;
import de.dim.trafficos.model.device.DeviceInfo;
import de.dim.trafficos.model.device.LifeCycleDeviceType;

/**
 * 
 * @author ilenia
 * @since Jun 26, 2019
 */
@ProviderType
public interface DeviceService {
	
	public static final String TOPIC_DEVICE = "device/%s";
	public static final String TOPIC_DEVICE_UPDATE = "device/topic/update";
	public static final String TOPIC_DEVICE_REMOVE = "device/topic/remove";
	public static final String TOPIC_DEVICE_LIFECYCLE = "device/topic/lifecycle";
	public static final String PROP_DEVICE_ID = "deviceId";
	public static final String PROP_DEVICE_STATUS = "deviceStatus";
	
	public static final String TOPIC_DATA_ENTRY = "dataEntry/%s";
	public static final String TOPIC_ALL_DATA_ENTRY = "allDataEntry/%s";
	public static final String TOPIC_CHANGE_DATA_ENTRY = "changedDataEntry/%s";
	public static final String TOPIC_DEVICE_ALL_ENTRIES = "device/topic/all/entries";
	public static final String PROP_DATA_ENTRY = "device.dataEntry";
	public static final String PROP_CYCLE_COUNTER = "device.cycleCounter";
	
	
	/**
	 * @return the LifeCycleDeviceType of the Device
	 */
	public LifeCycleDeviceType getStatus();	
	
	/**
	 * @return the Device associated with the DeviceService instance
	 */
	public Device getDevice();
	
	/**
	 * @return the id of the Device associated with the DeviceService instance
	 */
	public String getDeviceId();
	
	/**
	 * Initializes the device manually.
	 * @param forceReload forces to reload all data, independent from current running state
	 * @param forceRestart forces to restart a simulation, if it currently running
	 * @throws DeviceException
	 */
	public void initialize(boolean forceReload, boolean forceRestart) throws DeviceException;
		
	/**
	 * Updates the Device associated with the DeviceService instance with the given Device.
	 * If the provided Device is null or has a null update, no update is performed and the method returns the old Device.
	 * Otherwise the update is performed, the Device is saved to the DB and the updated Device is returned. 
	 * 
	 * @param device the updated Device to be saved
	 * @return the updated Device
	 */
	public Device updateDevice(Device device);
	
	/**
	 * Updates the DeviceConfiguration associated with the current device. The new DeviceConfiguration is set as the 
	 * current Configuration of the Device, and the Device is saved to the DB together with the updated Configuration.
	 * 
	 * @param configuration the new Configuration 
	 * @return the DeviceConfiguration after the update
	 */
	public DeviceConfiguration updateDeviceConfiguration(DeviceConfiguration configuration);
	
	/**
	 * Updates the DeviceInfo associated with the current device. The new DeviceInfo is set as the 
	 * current DeviceInfo of the Device, and the Device is saved to the DB together with the updated DeviceInfo.
	 * 
	 * @param info the new DeviceInfo 
	 * @return the DeviceInfo after the update
	 */
	public DeviceInfo updateDeviceInformation(DeviceInfo info);
	
	/**
	 * @return the current configuration of the Device associated with the DeviceService instance
	 */
	public DeviceConfiguration getConfiguration();
	
	/**
	 * @return the list of DeviceConfigurations of the Device associated with the DeviceService instance
	 */
	public List<DeviceConfiguration> getConfigurations();
	
	/**
	 * @return <code>true<code> if the Device associated with the DeviceService instance is running, <code>false<code>
	 * otherwise
	 */
	public boolean isRunning();
	
	/**
	 * Starts the Device associated with the DeviceService instance, if it is not running yet, and if it was possible 
	 * to start. A Device can be started if it has a DeviceConfiguration with an Intersection in it.
	 * @throws DeviceException
	 */
	public boolean startDevice() throws DeviceException;
	
	/**
	 * Stops the Device associated with the DeviceService instance, in case it was running.
	 * @throws DeviceException
	 */
	public boolean stopDevice() throws DeviceException;
	
	/**
	 * Retrieves the list of DataEntry within the provided time range for the Device associated with the DeviceService
	 * instance. If no DataEntry for the provided period is found, or if the start and end provided dates are inconsistent
	 * an empty list is returned. 
	 * 
	 * @param startDate the start date
	 * @param endDate the end date 
	 * @return the list of DataEntry of the Device for the given period, or an empty list
	 */
	public List<DataEntry> getDataEntries(Date startDate, Date endDate);
	
	/**
	 * @param timestamp the Date
	 * @return the DataEntry for the provided Date, or null, if there is none
	 */
	public DataEntry getCurrentDataEntry(Date timestamp);

}
