/**
 * Copyright (c) 2012 - 2019 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package de.dim.trafficos.device.impl;

import java.io.IOException;
import java.util.Collections;
import java.util.Date;
import java.util.Dictionary;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicReference;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;

import org.eclipse.emf.ecore.util.EcoreUtil;
import org.gecko.emf.repository.EMFRepository;
import org.gecko.emf.repository.query.IQuery;
import org.gecko.emf.repository.query.IQueryBuilder;
import org.gecko.emf.repository.query.QueryRepository;
import org.osgi.service.cm.Configuration;
import org.osgi.service.cm.ConfigurationAdmin;
import org.osgi.service.cm.ConfigurationException;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.ConfigurationPolicy;
import org.osgi.service.component.annotations.Deactivate;
import org.osgi.service.component.annotations.Modified;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.annotations.ReferenceScope;
import org.osgi.service.event.Event;
import org.osgi.service.event.EventAdmin;
import org.osgi.service.event.EventConstants;

import de.dim.trafficos.device.api.DeviceException;
import de.dim.trafficos.device.api.DeviceService;
import de.dim.trafficos.device.api.DevicesService;
import de.dim.trafficos.model.device.CacheDataEntry;
import de.dim.trafficos.model.device.DataEntry;
import de.dim.trafficos.model.device.Device;
import de.dim.trafficos.model.device.DeviceConfiguration;
import de.dim.trafficos.model.device.DeviceInfo;
import de.dim.trafficos.model.device.Intersection;
import de.dim.trafficos.model.device.LifeCycleDeviceType;
import de.dim.trafficos.model.device.Program;
import de.dim.trafficos.model.device.SignalTable;
import de.dim.trafficos.model.device.TOSDevicePackage;
import de.dim.trafficos.simulator.api.DeviceSimulator;
import de.dim.trafficos.simulator.api.DeviceStatusCallback;

/**
 * DeviceService implementation for a specific Device
 * 
 * @author ilenia
 * @since Jun 26, 2019
 */
@Component(service = DeviceService.class, configurationPolicy = ConfigurationPolicy.REQUIRE, configurationPid = "TOSDevice")
public class DeviceServiceImpl implements DeviceService , DeviceStatusCallback{

	@Reference(scope=ReferenceScope.PROTOTYPE)
	private EMFRepository repository;

	@Reference
	private EventAdmin eventAdmin;
	
	@Reference
	private ConfigurationAdmin configAdmin;

	private static final Logger logger = Logger.getLogger(DeviceServiceImpl.class.getName());
	private volatile AtomicReference<Device> deviceRef = new AtomicReference<Device>();
	private volatile List<DeviceConfiguration> configurations = new LinkedList<DeviceConfiguration>();
	private volatile AtomicReference<DeviceSimulator> deviceSimulatorRef = new AtomicReference<DeviceSimulator>();
	private volatile LifeCycleDeviceType lifecycle = LifeCycleDeviceType.NONE;
	private DeviceConfig config;
	private Map<Object, Object> dbOptions = new HashMap<Object, Object>();

	@interface DeviceConfig {
		String deviceId();
		boolean forceRestart() default false;
	}

	@Activate
	public void activate(DeviceConfig config) throws ConfigurationException {
		this.config = config;
		try {
			initialize(false, false);
		} catch (DeviceException e) {
			if (e.getCause() instanceof ConfigurationException) {
				throw (ConfigurationException)e.getCause();
			}
			logger.log(Level.SEVERE, String.format("[%s] Received error activating this device. Switching to lifecycle EXCEPTIONAL", config.deviceId()), e);
			lifecycle = LifeCycleDeviceType.EXCEPTIONAL;
			fireLifeCycleEvent();
		}
	}

	@Modified
	public void modify(DeviceConfig config) throws ConfigurationException {
		this.config = config;
		try {
			initialize(true, config.forceRestart());
		} catch (DeviceException e) {
			if (e.getCause() instanceof ConfigurationException) {
				throw (ConfigurationException)e.getCause();
			}
			logger.log(Level.SEVERE, String.format("[%s] Received error updating this device. Switching to lifecycle EXCEPTIONAL", config.deviceId()), e);
			lifecycle = LifeCycleDeviceType.EXCEPTIONAL;
			fireLifeCycleEvent();
		}
	}

	@Deactivate
	public void deactivate() {
		if(isRunning()) {
			try {
				stopDevice();
			} catch (DeviceException e) {
				logger.severe(String.format("[%s] Error stopping the device", config.deviceId()));
			}
		}
	}

	/* 
	 * (non-Javadoc)
	 * @see de.dim.trafficos.device.api.DeviceService#initialize(boolean, boolean)
	 */
	public void initialize(boolean forceReload, boolean forceRestart) throws DeviceException {
		if (LifeCycleDeviceType.LOADING.equals(lifecycle)) {
			throw new DeviceException("Error, initialization is already in progress");
		}
		LifeCycleDeviceType oldState = lifecycle;
		lifecycle = LifeCycleDeviceType.LOADING;
		fireLifeCycleEvent();
		if (config.deviceId() == null || config.deviceId().isEmpty()) {
			ConfigurationException cause = new ConfigurationException("deviceId", "DeviceId is a mandatory property");
			throw new DeviceException("Missing configuration property", cause);
		}
		Device oldDevice = deviceRef.get();
		Device newDevice = reloadAndCompareDevice(oldDevice);
		LifeCycleDeviceType newState = oldState;
		if (newDevice == null) {
			newState = LifeCycleDeviceType.EXCEPTIONAL;
		} else {
			if (!newDevice.equals(oldDevice)) {
				newState = compareConfiguration(oldDevice, newDevice);
			}
		}
		lifecycle = checkLifeCycle(oldState, newState);
		if(deviceRef.get() != null) {
			deviceRef.get().setLifeCycleType(lifecycle);
		}
		fireLifeCycleEvent();
	}

	/* 
	 * (non-Javadoc)
	 * @see de.dim.trafficos.device.api.DeviceService#getStatus()
	 */
	@Override
	public LifeCycleDeviceType getStatus() {
		return lifecycle;
	}

	/* 
	 * (non-Javadoc)
	 * @see de.dim.trafficos.device.api.DeviceService#getDevice()
	 */
	@Override
	public Device getDevice() {
		Device device = deviceRef.get();
		if(device != null) {
			return device;
		}
		return null;
	}

	/* 
	 * (non-Javadoc)
	 * @see de.dim.trafficos.device.api.DeviceService#getDeviceId()
	 */
	@Override
	public String getDeviceId() {
		return config.deviceId();
	}

	/* 
	 * (non-Javadoc)
	 * @see de.dim.trafficos.device.api.DeviceService#updateDevice(de.dim.trafficos.model.device.Device)
	 */
	@Override
	public Device updateDevice(Device device) {
		Device old = deviceRef.get();
		if(device == null || device.getId() == null) {
			logger.severe("Cannot update the Device with a null object or with a null id.");	
			return deviceRef.get();
		}
		else if (old == null) {
			deviceRef.set(device);
			repository.save(device);			
			
		} else if (!EcoreUtil.equals(old, device)) {
			deviceRef.compareAndSet(old, device);
			repository.save(device);
		}
		if(device.getConfiguration() != null && device.getConfiguration().getCurrentIntersection() != null) {
			lifecycle = LifeCycleDeviceType.PROVISIONED;
			deviceRef.get().setLifeCycleType(LifeCycleDeviceType.PROVISIONED);
			fireLifeCycleEvent();
		}
		else {
			lifecycle = LifeCycleDeviceType.UNPROVISIONED;
			deviceRef.get().setLifeCycleType(LifeCycleDeviceType.UNPROVISIONED);
			fireLifeCycleEvent();
		}
		return deviceRef.get();
	}

	/* 
	 * (non-Javadoc)
	 * @see de.dim.trafficos.device.api.DeviceService#updateDeviceConfiguration(de.dim.trafficos.model.device.DeviceConfiguration)
	 */
	@Override
	public DeviceConfiguration updateDeviceConfiguration(DeviceConfiguration configuration) {
		Device device = deviceRef.get();
		DeviceConfiguration oldConfig = device.getConfiguration();
		DeviceConfiguration newConfig = null;
		Intersection intersection = null;
		if (oldConfig != null) {
			if (!EcoreUtil.equals(oldConfig, configuration)) {
				newConfig = configuration;

			} 
		} else {
			newConfig = configuration;
		}
		if (newConfig != null) {
			synchronized (device) {
				device.setConfiguration(newConfig);
				if(newConfig.getCurrentIntersection() != null) {
					intersection = newConfig.getCurrentIntersection();
					if(LifeCycleDeviceType.RUNNING.equals(device.getLifeCycleType())) {
						Intersection oldIntersection = oldConfig.getCurrentIntersection();
						if(!EcoreUtil.equals(oldIntersection, intersection)) {
							try {
								stopDevice();
							} catch (DeviceException e) {								
								logger.severe(String.format("[%s] Error stopping the Device after Configuration update", device.getId()));
							}
							device.setLifeCycleType(LifeCycleDeviceType.PROVISIONED);
							lifecycle = LifeCycleDeviceType.PROVISIONED;
							fireLifeCycleEvent();
							try {
								startDevice();
							} catch (DeviceException e) {
								logger.severe(String.format("[%s] Error restarting the Device after Configuration update", device.getId()));
							}
						}
					}
					device.setLifeCycleType(LifeCycleDeviceType.PROVISIONED);
					lifecycle = LifeCycleDeviceType.PROVISIONED;
					fireLifeCycleEvent();
				}
				else {
					if(LifeCycleDeviceType.RUNNING.equals(device.getLifeCycleType())) {
						try {
							stopDevice();
						} catch (DeviceException e) {	
							logger.severe(String.format("[%s] Error stopping the Device after Configuration update", device.getId()));
						}
					}
					device.setLifeCycleType(LifeCycleDeviceType.UNPROVISIONED);		
					lifecycle = LifeCycleDeviceType.UNPROVISIONED;
					fireLifeCycleEvent();
				}
				repository.save(newConfig);
				repository.save(device);				
			}
			configurations.add(newConfig);
			return newConfig;
		} else {
			return oldConfig;
		}
	}

	/* 
	 * (non-Javadoc)
	 * @see de.dim.trafficos.device.api.DeviceService#updateDeviceInformation(de.dim.trafficos.model.device.DeviceInfo)
	 */
	@Override
	public DeviceInfo updateDeviceInformation(DeviceInfo info) {
		Device device = deviceRef.get();
		DeviceInfo oldInfo = device.getDeviceInformation();
		DeviceInfo newInfo = null;
		if (oldInfo != null) {
			if (!EcoreUtil.equals(oldInfo, info)) {
				newInfo = info;
			} 
		} else {
			newInfo = info;
		}
		if (newInfo != null) {
			synchronized (device) {
				device.setDeviceInformation(newInfo);
				repository.save(device);
			}
			return newInfo;
		} else {
			return oldInfo;
		}
	}

	/* 
	 * (non-Javadoc)
	 * @see de.dim.trafficos.device.api.DeviceService#getConfiguration()
	 */
	@Override
	public DeviceConfiguration getConfiguration() {
		Device device = deviceRef.get();
		return device.getConfiguration();
	}

	/* 
	 * (non-Javadoc)
	 * @see de.dim.trafficos.device.api.DeviceService#getConfigurations()
	 */
	@Override
	public List<DeviceConfiguration> getConfigurations() {
		return Collections.unmodifiableList(configurations);
	}

	/* 
	 * (non-Javadoc)
	 * @see de.dim.trafficos.device.api.DeviceService#isRunning()
	 */
	@Override
	public boolean isRunning() {
		return deviceSimulatorRef.get() != null && deviceSimulatorRef.get().isRunning();
	}

	
	/* 
	 * (non-Javadoc)
	 * @see de.dim.trafficos.device.api.DeviceService#startDevice()
	 */
	@Override
	public boolean startDevice() throws DeviceException {
		Device device = deviceRef.get();
		DeviceConfiguration config = device.getConfiguration();
		if(!LifeCycleDeviceType.PROVISIONED.equals(device.getLifeCycleType())) {
			logger.severe(String.format("[%s] Cannot start a Device in status %s", device.getId(), device.getLifeCycleType()));
			return false;
		}
		Intersection intersection = config.getCurrentIntersection();
		DeviceSimulator simulator = new DeviceSimulator(this);
		if (deviceSimulatorRef.compareAndSet(null, simulator)) {	
			if(registerMQTTHandler()) {
				simulator.initializeSimulation(intersection);
				simulator.setNotifyConsumer(this::handleDataEntry);				
				if(simulator.startSimulation()) {
					device.setLifeCycleType(LifeCycleDeviceType.RUNNING);
					lifecycle = LifeCycleDeviceType.RUNNING;
					fireLifeCycleEvent();
					logger.fine(String.format("[%s] Started new simulation", device.getId()));
					return true;
				}
				else {
					logger.warning(String.format("[%s] Simulation was already started", device.getId()));
					device.setLifeCycleType(LifeCycleDeviceType.RUNNING);
					lifecycle = LifeCycleDeviceType.RUNNING;
					fireLifeCycleEvent();
					return false;
				}			
			}
			else {
				logger.warning(String.format("[%s] Error activating MQTTEventHandler. Simulation will not start.", device.getId()));
				return false;
			}
		} else {
			logger.warning(String.format("[%s] There is already a simulation running, nothing to start", device.getId()));
			return false;
		}		
	}

	/* 
	 * (non-Javadoc)
	 * @see de.dim.trafficos.device.api.DeviceService#stopDevice()
	 */
	@Override
	public boolean stopDevice() throws DeviceException {
		Device device = deviceRef.get();
		DeviceSimulator simulator = deviceSimulatorRef.get();
		if (simulator != null) {
			if(simulator.stopSimulation()) {
				simulator.setNotifyConsumer(null);
				deviceSimulatorRef.compareAndSet(simulator, null);
				device.setLifeCycleType(LifeCycleDeviceType.PROVISIONED);
				lifecycle = LifeCycleDeviceType.PROVISIONED;
				fireLifeCycleEvent();
				unregisterMQTTHandler();
				logger.fine(String.format("[%s] Stopped simulation", device.getId()));
				return true;
			}
			else {
				logger.warning(String.format("[%s] Simulation was already interrupted", device.getId()));
				device.setLifeCycleType(LifeCycleDeviceType.PROVISIONED);
				lifecycle = LifeCycleDeviceType.PROVISIONED;
				fireLifeCycleEvent();
				return false;
			}
			
		} else {
			logger.warning(String.format("[%s] There is no simulation to stop", device.getId()));
			return false;
		}
	}

	
	/* 
	 * (non-Javadoc)
	 * @see de.dim.trafficos.device.api.DeviceService#getDataEntries(java.util.Date, java.util.Date)
	 */
	@Override
	public List<DataEntry> getDataEntries(Date startDate, Date endDate) {
		Device device = deviceRef.get();
		String deviceId = device.getId();
		List<DataEntry> dataList = new LinkedList<DataEntry>();		
		if(deviceId == null) {
			logger.severe("Cannot retrieve data associated with a device of null id.");
			return dataList;
		}
		if(dateIsOK(startDate, endDate)) {
			dbOptions.put("COLLECTION_PARTITION_EXTENSION", deviceId);		
			dataList.addAll(getDataEntryInTimeRange(startDate, endDate));
			if(dataList.isEmpty()) {
				logger.warning(String.format("[%s] No data within the provided time range.", config.deviceId()));
			}
		}
		else {
			logger.severe(String.format("[%s] Start and End dates are not consistent.", config.deviceId()));
		}
		return dataList;
	}
	
	/* 
	 * (non-Javadoc)
	 * @see de.dim.trafficos.device.api.DeviceService#getCurrentDataEntry(java.util.Date)
	 */
	@Override
	public DataEntry getCurrentDataEntry(Date timestamp) {
		
		DeviceSimulator simulator = deviceSimulatorRef.get();
		if(simulator == null) {
			logger.warning(String.format("[%s] Simulation has not started yet.", config.deviceId()));
			return null;
		}
		Date startTime = simulator.getStartTime();
		if(startTime == null) {
			logger.warning(String.format("[%s] Start Time for simulation has not been set yet.", config.deviceId()));
		}		
		long diff = (timestamp.getTime()-startTime.getTime())/1000;
		int cycleNum = simulator.getNumCycles();
		int currentCycleSec = (int) (diff/cycleNum);
		Program program = simulator.getRunningProgram();
		SignalTable sigTab = program.getSignalTable();
		List<CacheDataEntry> cacheDataEntry = sigTab.getCacheDataEntry();
		cacheDataEntry = cacheDataEntry.stream().filter(de->(String.valueOf(currentCycleSec).equals(de.getId())))
				.collect(Collectors.toList());
		if(cacheDataEntry.isEmpty()) {
			logger.severe(String.format("[%s] No cached DateEntry for starting TX", config.deviceId()));
			return null;
		}
		if(cacheDataEntry.size() > 1) {
			logger.severe(String.format("[%s] More than one cached DataEntry for starting TX", config.deviceId()));
			return null;
		}
		DataEntry cde = cacheDataEntry.get(0);
		cde.setTimestamp(timestamp);
		cde.setDevice(config.deviceId());
		if(deviceRef.get().getConfiguration() != null) {
			cde.setConfiguration(deviceRef.get().getConfiguration().getId());
		}		
		return cde;
	}

	
	/**
	 * Makes the DataEntry request within the provided time range
	 * 
	 * @param startDate
	 * @param endDate
	 * @return
	 */
	private List<DataEntry> getDataEntryInTimeRange(Date startDate, Date endDate) {

		QueryRepository queryRepository = (QueryRepository) repository;        
		IQueryBuilder rangeQueryBuilder = queryRepository.createQueryBuilder(); //here we are creating the query
		rangeQueryBuilder.column(TOSDevicePackage.Literals.DATA_ENTRY__TIMESTAMP).rangeQuery();
		rangeQueryBuilder.startValue(startDate, true);
		rangeQueryBuilder.endValue(endDate, true);
		IQuery query = rangeQueryBuilder.build();
		List<DataEntry> entries = queryRepository.getEObjectsByQuery(TOSDevicePackage.Literals.DATA_ENTRY, query, 
				dbOptions);
		return entries;
	}
	
	
	/**
	 * @param oldState
	 * @throws DeviceException 
	 */
	private LifeCycleDeviceType checkLifeCycle(LifeCycleDeviceType oldState, LifeCycleDeviceType newState) throws DeviceException {
		if (LifeCycleDeviceType.EXCEPTIONAL.equals(oldState) && oldState.equals(newState)) {
			deactivate();
			return LifeCycleDeviceType.EXCEPTIONAL;
		}
		// nothing happened
		if (newState == null) {
			return oldState;
		}
		switch (newState) {
		case UNPROVISIONED:
			deactivate();
			return LifeCycleDeviceType.UNPROVISIONED;
		case PROVISIONED:
			deactivate();
			if (config.forceRestart() && LifeCycleDeviceType.RUNNING.equals(oldState)) {
				startDevice();
				return LifeCycleDeviceType.RUNNING;
			} else {
				return LifeCycleDeviceType.PROVISIONED;
			}
		default:
			deactivate();
			return LifeCycleDeviceType.EXCEPTIONAL;
		}
	}

	/**
	 * Loads the Device which correspond to the provided properties
	 * @param properties
	 */
	private Device reloadAndCompareDevice(Device oldDevice) throws DeviceException {
		String deviceId = config.deviceId();
		try {
			Device newDevice = repository.getEObject(TOSDevicePackage.Literals.DEVICE, deviceId);
			if(newDevice == null) {
				return null;
			}
			else {
				if (!EcoreUtil.equals(oldDevice, newDevice)) {
					if (deviceRef.compareAndSet(oldDevice, newDevice)) {
						return newDevice;
					}
				}
			}
			return oldDevice;
		} catch (Exception e) {
			if (e instanceof DeviceException) {
				throw e;
			}
			throw new DeviceException(String.format("[%s] An error occured loading a device", deviceId), e);
		}
	}

	/**
	 * Loads the Device which correspond to the provided properties
	 * @param properties
	 */
	private LifeCycleDeviceType compareConfiguration(Device oldDevice, Device newDevice) throws DeviceException {
		try {
			DeviceConfiguration oldConfig = oldDevice == null ? null : oldDevice.getConfiguration();
			DeviceConfiguration newConfig = newDevice.getConfiguration();
			if (!EcoreUtil.equals(oldConfig, newDevice)) {
				if(newConfig != null && newConfig.getCurrentIntersection() != null) {
					newDevice.setLifeCycleType(LifeCycleDeviceType.PROVISIONED);
					configurations.add(newConfig);
					return LifeCycleDeviceType.PROVISIONED;
				}
				else {
					newDevice.setLifeCycleType(LifeCycleDeviceType.UNPROVISIONED);
					if(newConfig != null) {
						configurations.add(newConfig);
					}
					return LifeCycleDeviceType.UNPROVISIONED;
				}
				
			} 
			return null;
		} catch (Exception e) {
			if (e instanceof DeviceException) {
				throw e;
			}
			throw new DeviceException(String.format("[%s] An error occured loading a device", config.deviceId()), e);
	
		}
	}

	/**
	 * Checks consistency between start and end time stamps.
	 * 
	 * @param start the start time stamp
	 * @param end the end time stamp
	 * @return <code>true<code> if the time stamps are consistent, <code>false<code> otherwise.
	 */
	private boolean dateIsOK(Date start, Date end) {
		if(start == null|| end == null) {
			return false;
		}
		if(start.after(end)) {
			return false;
		}
		return true;
	}

	/**
	 * Notifies the EventAdmin when a new DataEntry has been created during a simulation.
	 * @param entry the DataEntry object
	 * @param tx the TX value
	 */
	private void handleDataEntry(DataEntry entry, Integer tx) {
		Device device = deviceRef.get();
		String topic = String.format(TOPIC_DATA_ENTRY, device.getId());
		Map<String, Object> properties = new HashMap<String, Object>();
		entry.setDevice(device.getId());
		properties.put(PROP_DATA_ENTRY, entry);
		properties.put(PROP_CYCLE_COUNTER, tx);
		Event event = new Event(topic, properties);
		eventAdmin.postEvent(event);
	}
	
	/**
	 * Fires a LifeCycleEvent which should be listen from the DevicesService
	 */
	private void fireLifeCycleEvent() {
		String topic = DevicesService.TOPIC_DEVICE_STATUS;
		Map<String, Object> properties = new HashMap<String, Object>();
		properties.put(PROP_DEVICE_ID, config.deviceId());
		properties.put(DeviceService.PROP_DEVICE_STATUS, lifecycle);
		Event event = new Event(topic, properties);
		eventAdmin.postEvent(event);		
	}

	/**
	 * Notifies the ConfigAdmin to register the MQTTEventHandler for this Device
	 * @return <code>true<code> if the Configuration has been updated, <code>false<code> if an error occurred
	 */
	private boolean registerMQTTHandler() {
		
		Dictionary<String, Object> props = new Hashtable<String, Object>();
		props.put(PROP_DEVICE_ID, config.deviceId());
		props.put(EventConstants.EVENT_TOPIC, "dataEntry/"+config.deviceId());
//		props.put("mqttTopic", "signalStates/"+config.deviceId());
		props.put("mqttTopicAllEntries", "allDataEntry/"+config.deviceId());
		props.put("mqttTopicChangedEntry", "changedDataEntry/"+config.deviceId());
		try {
			logger.fine(String.format("[%s] Registering Config for MQTTDataEntryHandler", config.deviceId()));
			Configuration configuration = configAdmin.getFactoryConfiguration("MQTTDataEntryHandler", config.deviceId(), "?");
			configuration.update(props);
			return true;
		} catch (IOException e) {
			logger.severe(String.format("[%s] Error registering Config for MQTTDataEntryHandler", config.deviceId()));
			return false;
		}		
	}
	
	/**
	 * Unregisters the Configuration for the MQTTEventHandler for this Device
	 */
	private void unregisterMQTTHandler() {
		try {
			logger.fine(String.format("[%s] Unregistering Config for MQTTDataEntryHandler", config.deviceId()));
			Configuration configuration = configAdmin.getFactoryConfiguration("MQTTDataEntryHandler", config.deviceId(), "?");
			configuration.delete();
		} catch (IOException e) {
			logger.severe(String.format("[%s] Error unregistering Config for MQTTDataEntryHandler", config.deviceId()));
			return;
		}				
	}

	/* 
	 * (non-Javadoc)
	 * @see de.dim.trafficos.device.api.DeviceStatusCallback#statusChanged(boolean)
	 */
	@Override
	public void statusChanged(boolean running) {
		if(!running) {
			logger.warning(String.format("[%s] Simulation was interrupted." , deviceRef.get().getId()));
			lifecycle = LifeCycleDeviceType.PROVISIONED;
			fireLifeCycleEvent();
			deviceSimulatorRef.set(null);
			unregisterMQTTHandler();
		}
	}
}
