package de.dim.trafficos.device.sample;
/**
 * Copyright (c) 2012 - 2019 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */

import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.Executors;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;

import org.gecko.emf.repository.EMFRepository;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;
import org.osgi.util.promise.Promise;
import org.osgi.util.promise.PromiseFactory;

import de.dim.trafficos.model.device.Device;
import de.dim.trafficos.model.device.DeviceActivationType;
import de.dim.trafficos.model.device.DeviceConfiguration;
import de.dim.trafficos.model.device.DeviceInfo;
import de.dim.trafficos.model.device.DirectionType;
import de.dim.trafficos.model.device.HardwareConfiguration;
import de.dim.trafficos.model.device.Intersection;
import de.dim.trafficos.model.device.OutPinConfiguration;
import de.dim.trafficos.model.device.Output;
import de.dim.trafficos.model.device.OutputConfiguration;
import de.dim.trafficos.model.device.OutputElementType;
import de.dim.trafficos.model.device.Phase;
import de.dim.trafficos.model.device.PhaseGroup;
import de.dim.trafficos.model.device.Position;
import de.dim.trafficos.model.device.Program;
import de.dim.trafficos.model.device.ScheduleModeType;
import de.dim.trafficos.model.device.TOSDeviceFactory;
import de.dim.trafficos.simulator.api.IntersectionConstants;
import de.dim.trafficos.simulator.api.IntersectionService;
import de.dim.trafficos.simulator.api.SignalPlanConstants;
import de.dim.trafficos.simulator.api.SignalPlanService;

/**
 * Test Component to create some sample Devices
 * 
 * @author ilenia
 * @since Jun 28, 2019
 */
@Component(immediate = true)
public class TestDeviceComponent {

	@Reference(target ="(repo_id=timRSA)")
	private EMFRepository repository;

	@Reference
	IntersectionService interService;
	
	@Reference
	SignalPlanService sigService;

	public static final Logger logger = Logger.getLogger(TestDeviceComponent.class.getName());


	@Activate
	public void activate() {
		PromiseFactory pf = new PromiseFactory(Executors.newSingleThreadExecutor());
		Promise<String> p = pf.resolved("test");
		p.delay(5000l).thenAccept((s)->{
			for(int i = 0; i < 1; i++) {
				try {
					logger.info("Creating Device " + i);
					HardwareConfiguration hardware = TOSDeviceFactory.eINSTANCE.createHardwareConfiguration();
					Device device = createDevice("dev_"+String.valueOf(i), hardware);
					DeviceConfiguration config = device.getConfiguration();
					device.setConfiguration(null);
					repository.save(device);
					repository.save(config);
					device.setConfiguration(config);
					repository.save(device);
					repository.save(hardware);
//					repository.save(device.getConfiguration());
					logger.info("Saved Device " + i);
				} catch (Exception e) {
					logger.log(Level.SEVERE, "Error saving device " + i, e);
				}
			}
		});
	}


	/**
	 * @return
	 */
	private Device createDevice(String id, HardwareConfiguration hardware) {
		Device device = TOSDeviceFactory.eINSTANCE.createDevice();
		device.setId(id);
		device.setActivationState(DeviceActivationType.ACTIVE);
		if(id.contains("0")) {
			Position pos = TOSDeviceFactory.eINSTANCE.createPosition();
			pos.setLatitude(50.921036);
			pos.setLongitude(11.581245);
			device.getLocation().add(pos);
		}
		else if(id.contains("1")) {
			Position pos = TOSDeviceFactory.eINSTANCE.createPosition();
			pos.setLatitude(50.901747);
			pos.setLongitude(11.578155);
			device.getLocation().add(pos);
		}
		createDeviceInfo(device);
		createDeviceConfig(device, hardware);
		return device;
	}


	

	/**
	 * @param device
	 */
	private void createDeviceConfig(Device device, HardwareConfiguration hardware) {
		DeviceConfiguration config = TOSDeviceFactory.eINSTANCE.createDeviceConfiguration();
		config.setId("conf_"+device.getId());
		hardware.setDeviceConfigurationId(config.getId());
		hardware.setDeviceId(device.getId());
		hardware.setId(device.getId());
		createIntersection(config, hardware);
		device.setConfiguration(config);		
	}


	/**
	 * Creates the intersection and hardware config
	 * https://pi4j.com/1.2/pins/model-3b-rev1.html
	 * Road 1
	 * SG-1_0 pin 3 - gpio 8 RED
	 * SG-1_0 pin 5 - gpio 9 AMBER
	 * SG-1_0 pin 7 - gpio 7 GREEN
	 * 
	 * SG-1_1 pin 8 - gpio 15 RED
	 * SG-1_1 pin 10 - gpio 16 AMBER
	 * SG-1_1 pin 12 - gpio 1 GREEN
	 * 
	 * SG-P_1_0 pin 16 - gpio 4 RED
	 * SG-P_1_0 pin 18 - gpio 5 GREEN
	 * 
	 * Road 2
	 * 
	 * SG-2_1 pin 22 - gpio 6 RED
	 * SG-2_1 pin 24 - gpio 10 AMBER
	 * SG-2_1 pin 26 - gpio 11 GREEN
	 * 
	 * SG-P_2_0 pin 28 - gpio 31 RED
	 * SG-P_2_0 pin 32 - gpio 26 GREEN
	 * 
	 * Road 3
	 * 
	 * SG-0_1 pin 36 - gpio 6 RED
	 * SG-0_1 pin 38 - gpio 10 AMBER
	 * SG-0_1 pin 40 - gpio 11 GREEN
	 * 
	 * SG-P_0_0 pin 35 - gpio 24 RED
	 * SG-P_0_0 pin 37 - gpio 25 GREEN
	 * 
	 * 
	 * @param config
	 */
	private void createIntersection(DeviceConfiguration config, HardwareConfiguration hardware) {
		Map<Integer, String> options = new HashMap<Integer, String>();
		Intersection intersection = null;
		if(config.getId().contains("0")) {
//			create DIM intersection
//			options.put(0, IntersectionConstants.MAIN_STRAIGHT_LEFT_SEP);
//			options.put(1, IntersectionConstants.SEC_LEFT_RIGHT_MERGE);
//			options.put(2, IntersectionConstants.MAIN_STRAIGHT_RIGHT_MERGE);
			options.put(0, IntersectionConstants.MAIN_STRAIGHT_RIGHT_MERGE);
			options.put(1, IntersectionConstants.MAIN_STRAIGHT_LEFT_SEP);
			options.put(2, IntersectionConstants.SEC_LEFT_RIGHT_MERGE);
			intersection = interService.createIntersection(options);
			
			List<Output> outputs = intersection.getOutput();
			for(Output out : outputs) {
				Position loc = TOSDeviceFactory.eINSTANCE.createPosition();
				OutputConfiguration outConf = TOSDeviceFactory.eINSTANCE.createOutputConfiguration();
				outConf.setOutput(out);
				hardware.getPinConfiguration().add(outConf);
				switch(out.getId()) {
				case "SG-1_0":
					loc.setLatitude(50.920967);
					loc.setLongitude(11.581207);
					loc.setOrientation(DirectionType.STRAIGHT);
					createPinConfigs(outConf, false, new int[] {3, 5, 7}, new int[] {8, 9, 7}, new String[]{"SDA1","SLC1","GPIO04"});
					break;
				case "SG-1_1":
					loc.setLatitude(50.920960);
					loc.setLongitude(11.581207);
					loc.setOrientation(DirectionType.TURN_LEFT);
					createPinConfigs(outConf, false, new int[] {8, 10, 12}, new int[] {15, 16, 1}, new String[]{"TxD","RxD","GPIO18"});
					break;
				case "SG-P_1_0":
					loc.setLatitude(50.920970);
					loc.setLongitude(11.581207);
					createPinConfigs(outConf, true, new int[] {16, 18}, new int[] {4, 5}, new String[]{"GPIO23","GPIO24"});
					break;
				case "SG-2_1":
					loc.setLatitude(50.921179);
					loc.setLongitude(11.580993);
					loc.setOrientation(DirectionType.LEFT_RIGHT);
					createPinConfigs(outConf, false, new int[] {22, 24, 26}, new int[] {6, 10, 11}, new String[]{"GPIO25","CE0","CE1"});
					break;
				case "SG-P_2_0":
					loc.setLatitude(50.921179);
					loc.setLongitude(11.580990);
					createPinConfigs(outConf, true, new int[] {28, 32}, new int[] {31, 26}, new String[]{"SCL0","GPIO12"});
					break;
				case "SG-0_1":
					loc.setLatitude(50.921064);
					loc.setLongitude(11.58117);
					loc.setOrientation(DirectionType.STRAIGHT_RIGHT);
					createPinConfigs(outConf, false, new int[] {36, 38, 40}, new int[] {27, 28, 29}, new String[]{"GPIO16","GPIO20","GPIO21"});
					break;
				case "SG-P_0_0":
					loc.setLatitude(50.921060);
					loc.setLongitude(11.58117);
					createPinConfigs(outConf, true, new int[] {35, 37}, new int[] {24, 25}, new String[]{"GPIO19","GPIO26"});
					break;
				}					
			out.getLocation().add(loc);
			}
		}
		else if(config.getId().contains("1")) {
//			Winzerla intersection
			options.put(0, IntersectionConstants.MAIN_STRAIGHT_TURNS_SEP);
			options.put(1, IntersectionConstants.SEC_STRAIGHT_RIGHT_MERGE);
			options.put(2, IntersectionConstants.MAIN_STRAIGHT_RIGHT_MERGE);
			options.put(3, IntersectionConstants.SEC_STRAIGHT_LEFT_MERGE);
			intersection = interService.createIntersection(options);
			
			List<Output> outputs = intersection.getOutput();
			for(Output out : outputs) {
				Position loc = TOSDeviceFactory.eINSTANCE.createPosition();
				switch(out.getId()) {
				case "SG-0_0":
					loc.setLatitude(50.901848);
					loc.setLongitude(11.578053);
					loc.setOrientation(DirectionType.TURN_RIGHT);
					break;
				case "SG-0_1":
					loc.setLatitude(50.901848);
					loc.setLongitude(11.578055);
					loc.setOrientation(DirectionType.STRAIGHT);
					break;
				case "SG-0_2":
					loc.setLatitude(50.901848);
					loc.setLongitude(11.578057);
					loc.setOrientation(DirectionType.TURN_LEFT);
					break;
				case "SG-P_0_0":
					loc.setLatitude(50.901848);
					loc.setLongitude(11.578059);
					break;
				case "SG-1_1":
					loc.setLatitude(50.90179);
					loc.setLongitude(11.578187);
					loc.setOrientation(DirectionType.STRAIGHT_RIGHT);
					break;
				case "SG-1_2":
					loc.setLatitude(50.90177);
					loc.setLongitude(11.578187);
					loc.setOrientation(DirectionType.TURN_LEFT);
					break;
				case "SG-P_1_0":
					loc.setLatitude(50.90175);
					loc.setLongitude(11.578187);
					break;
				case "SG-2_1":
					loc.setLatitude(50.901696);
					loc.setLongitude(11.578154);
					loc.setOrientation(DirectionType.STRAIGHT_RIGHT);
					break;
				case "SG-2_2":
					loc.setLatitude(50.901696);
					loc.setLongitude(11.578152);
					loc.setOrientation(DirectionType.TURN_LEFT);
					break;
				case "SG-P_2_0":
					loc.setLatitude(50.901696);
					loc.setLongitude(11.578150);
					break;
				case "SG-3_0":
					loc.setLatitude(50.90179);
					loc.setLongitude(11.577800);
					loc.setOrientation(DirectionType.TURN_RIGHT);
					break;
				case "SG-3_1":
					loc.setLatitude(50.90181);
					loc.setLongitude(11.577800);
					loc.setOrientation(DirectionType.STRAIGHT_LEFT);
					break;
				case "SG-P_3_0":
					loc.setLatitude(50.90183);
					loc.setLongitude(11.577800);
					break;
				}					
			out.getLocation().add(loc);
			}
		}		
		createPhases(intersection);
		createProgram(intersection);
		config.getIntersection().add(intersection);
		config.setCurrentIntersection(intersection);
	}


	/**
	 * @param intersection
	 */
	private void createProgram(Intersection intersection) {
		List<PhaseGroup> phGroups = intersection.getPhaseGroup().stream().sorted(Comparator.comparing(PhaseGroup::getPenalty))
				.collect(Collectors.toList());
		PhaseGroup pg = phGroups.get(0);
		List<Phase> groupPh = pg.getPhase();
		if(groupPh.size() == 3) {
			for(Phase p : groupPh) {			
				if(p.getId().equals("PH_0")) {
					p.setWeightMax(70);
					p.setWeightMin(60);
				}
				else if(p.getId().equals("PH_2")) {
					p.setWeightMax(50);
					p.setWeightMin(30);
				}
				else if(p.getId().equals("PH_3")) {
					p.setWeightMax(20);
					p.setWeightMin(10);
				}
			}
		}
		else if(groupPh.size() == 4) {
			for(int i = 0; i < 4; i++) {
				Phase p = groupPh.get(i);
				if(i == 0) {
					p.setWeightMax(70);
					p.setWeightMin(35);
				}
				else if(i == 1) {
					p.setWeightMax(50);
					p.setWeightMin(25);
				}
				else if(i == 2) {
					p.setWeightMax(20);
					p.setWeightMin(25);
				}
				else {
					p.setWeightMax(20);
					p.setWeightMin(15);
				}
			}
		}
		
		Program program = sigService.createFixTimeProgram(pg, UUID.randomUUID().toString(), 100);
		sigService.applyProgram(intersection, program, ScheduleModeType.WORKING_DAY);		
		sigService.addCacheDataValueDurations(program);
	}


	/**
	 * @param intersection
	 */
	private void createPhases(Intersection intersection) {
		sigService.createPhases(intersection, SignalPlanConstants.ALL_PHASES);
		sigService.createPhaseGroups(intersection);		
	}


	/**
	 * @param device
	 */
	private void createDeviceInfo(Device device) {
		DeviceInfo info = TOSDeviceFactory.eINSTANCE.createDeviceInfo();
		info.setDescription("des_"+device.getId());
		info.setHumanReadableName("name_"+device.getId());
		info.setShortName("alias_"+device.getId());
		info.setTechnicalName("tec_"+device.getId());
		device.setDeviceInformation(info);		
	}
	
	private void createPinConfigs(OutputConfiguration config, boolean pedestrian, int[] pin, int[] gpio, String[] comments) {
		config.getPin().add(createPinConfig(pin[0], gpio[0], comments[0], OutputElementType.RED));
		if (pedestrian) {
			config.getPin().add(createPinConfig(pin[1], gpio[1], comments[1], OutputElementType.GREEN));
		} else {
			config.getPin().add(createPinConfig(pin[1], gpio[1], comments[1], OutputElementType.AMBER));
			config.getPin().add(createPinConfig(pin[2], gpio[2], comments[2], OutputElementType.GREEN));
			
		}
	}
	
	private OutPinConfiguration createPinConfig(int pin, int gpio, String comment, OutputElementType type) {
		OutPinConfiguration pinConfig = TOSDeviceFactory.eINSTANCE.createOutPinConfiguration();
		pinConfig.setPin(pin);
		pinConfig.setGpio(gpio);
		pinConfig.setComment(comment);
		pinConfig.setType(type);
		return pinConfig;
	}

}
