/**
 * Copyright (c) 2012 - 2018 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package de.dim.trafficos.device.tests;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;
import static org.mockito.Matchers.any;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Date;
import java.util.Dictionary;
import java.util.Hashtable;
import java.util.LinkedList;
import java.util.List;
import java.util.UUID;
import java.util.logging.Logger;

import org.eclipse.emf.ecore.EObject;
import org.gecko.core.tests.AbstractOSGiTest;
import org.gecko.core.tests.ServiceChecker;
import org.gecko.emf.repository.EMFRepository;
import org.gecko.emf.repository.query.IQuery;
import org.gecko.emf.repository.query.IQueryBuilder;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.Mock;
import org.mockito.Mockito;
import org.mockito.runners.MockitoJUnitRunner;
import org.osgi.framework.Bundle;
import org.osgi.framework.FrameworkUtil;
import org.osgi.framework.InvalidSyntaxException;
import org.osgi.framework.PrototypeServiceFactory;
import org.osgi.framework.ServiceRegistration;
import org.osgi.service.cm.Configuration;
import org.osgi.service.cm.ConfigurationAdmin;
import org.osgi.service.event.EventAdmin;

import de.dim.trafficos.device.api.DeviceException;
import de.dim.trafficos.device.api.DeviceLifeCycleHandler;
import de.dim.trafficos.device.api.DeviceService;
import de.dim.trafficos.device.api.DevicesService;
import de.dim.trafficos.model.device.DataEntry;
import de.dim.trafficos.model.device.DataValue;
import de.dim.trafficos.model.device.Device;
import de.dim.trafficos.model.device.DeviceActivationType;
import de.dim.trafficos.model.device.DeviceConfiguration;
import de.dim.trafficos.model.device.DeviceInfo;
import de.dim.trafficos.model.device.IdNameElement;
import de.dim.trafficos.model.device.Intersection;
import de.dim.trafficos.model.device.LifeCycleDeviceType;
import de.dim.trafficos.model.device.TOSDeviceFactory;
import de.dim.trafficos.model.device.TOSDevicePackage;


/**
 * Tests the implementation of the DevicesService and DeviceService
 * @since 1.0
 */
@RunWith(MockitoJUnitRunner.class)
public class DevicesServiceIntegrationTest extends AbstractOSGiTest {
	
	@Mock
	private QueryRepositoryMock repository;
	@Mock
	private IQueryBuilder builder;
	
	private DevicesService devService;
	private DeviceService deviceService;
	private static final Logger logger = Logger.getLogger(DevicesServiceIntegrationTest.class.getName());
	
	/**
	 * Creates a new instance.
	 * @param bundleContext
	 */
	public DevicesServiceIntegrationTest() {
		super(FrameworkUtil.getBundle(DevicesServiceIntegrationTest.class).getBundleContext());
	}

	/**
	 * Here you can put everything you want to be executed before every test
	 */
	public void doBefore() {
		ConfigurationAdmin configAdmin = getConfigAdmin();
		try {
			Configuration[] configs = configAdmin.listConfigurations("("+DeviceService.PROP_DEVICE_ID+"=*)");
			if(configs != null) {
				for(Configuration c : configs) {
					c.delete();
				}
			}
			
		} catch (IOException | InvalidSyntaxException e) {
			e.printStackTrace();
		}
	}
	
	/**
	 * Here you can put everything you want to be executed after every test
	 */
	public void doAfter() {
		ConfigurationAdmin configAdmin = getConfigAdmin();
		try {
			Configuration[] configs = configAdmin.listConfigurations("("+DeviceService.PROP_DEVICE_ID+"=*)");
			if(configs != null) {
				for(Configuration c : configs) {
					c.delete();
				}
			}
		} catch (IOException | InvalidSyntaxException e) {
			e.printStackTrace();
		}
	}
	
	/**
	 * Here you can put your test
	 */
	@Test
	public void testServiceCreation() {
		
		setupServices();
	}
	
	/**
	 * Tests the getActiveDevices() method
	 */
	@Test
	public void testActiveDevice() {
		
		setupServices();
		
		Device device = createDeviceSample(DeviceActivationType.ACTIVE);
		List<EObject> list = new ArrayList<EObject>();
		list.add(device);		
			
		List<Device> result = devService.getActiveDevices();
		assertNotNull(result);
		assertTrue(result.isEmpty()); //no DeviceService has been registered, so no Device can be found
	}
	
	
	
	/**
	 * Tests the getAllDevices() method of the DevicesService
	 */
	@Test
	public void testAllDevices() {
		setupServices();
		
		Device device = createDeviceSample(DeviceActivationType.INACTIVE);
		List<EObject> list = new ArrayList<EObject>();
		list.add(device);		
		
		List<Device> result = devService.getAllDevices();
		assertNotNull(result);
		assertTrue(result.isEmpty()); //no DeviceService has been registered, so no Device can be found
	}
	
	@Test
	public void testDeviceNullId() {
		
		setupServices();
		Device device = devService.getDeviceById(null);
		assertNull(device);
	}
	
	@Test 
	public void testDeviceWrongId() throws InterruptedException {
		setupServices();
		
		assertNotNull(repository);
		when(repository.getEObject(any(TOSDevicePackage.Literals.DEVICE.getClass()), any(String.class))).thenReturn(null);
		
		Device device = devService.getDeviceById("test");
		assertNull(device);
	}
	
	@Test
	public void testDeviceIdOK() throws InterruptedException {
		setupServices();
		
		Device device = createDeviceSample(DeviceActivationType.ACTIVE);
		assertNotNull(repository);
		when(repository.getEObject(any(TOSDevicePackage.Literals.DEVICE.getClass()), any(String.class))).thenReturn(device);
		
		Device dev = devService.getDeviceById("test01");
		assertNotNull(dev);
		assertEquals("test01", dev.getId());
	}
	
	@Test
	public void testUpdateDeviceNullId() throws InterruptedException {
		setupServices();
		
		Device device = createDeviceSample(DeviceActivationType.ACTIVE);
		device.setId(null);
		
		Device dev = devService.updateDevice(device);
		assertNull(dev);
	}
	
	/**
	 * 1. Update a non-existing Device through the DevicesService --> should save the new Device, trigger the ConfigAdmin
	 *    and activate the corresponding DeviceService
	 * 2. Retrieve the Device through the DeviceService --> the Device should be the same as the newly created one
	 * @throws InterruptedException
	 */
	@Test
	public void testUpdateDeviceNew() throws InterruptedException {
		setupServices();
		
		ServiceChecker<DeviceService> devChecker = createCheckerTrackedForCleanUp(DeviceService.class);
		devChecker.start();
		assertEquals(0, devChecker.getCurrentCreateCount(false));
		
		Device device = createDeviceSample(DeviceActivationType.ACTIVE);
		assertNotNull(repository);
		when(repository.getEObject(any(TOSDevicePackage.Literals.DEVICE.getClass()), any(String.class))).thenReturn(null)
		.thenReturn(device);
		
		Device dev = devService.updateDevice(device);
		assertNotNull(dev);
		assertEquals(device, dev);
		
		assertEquals(1, devChecker.getCurrentCreateCount(true));
		
		deviceService = getService(DeviceService.class);
		assertNotNull(deviceService);
						
		Device currDevice = deviceService.getDevice();
		
		assertEquals(device.getId(), currDevice.getId());
		assertEquals(device, currDevice);
		assertEquals(LifeCycleDeviceType.UNPROVISIONED, currDevice.getLifeCycleType());	
		
		devService.removeDevice("test01");		
		assertTrue(devChecker.awaitRemoval());		
		
	}
	
	@Test
	public void testUpdateDeviceNoUpdateNeeded() throws InterruptedException {
		setupServices();		
		
		Device device = createDeviceSample(DeviceActivationType.ACTIVE);
		assertNotNull(repository);
		when(repository.getEObject(any(TOSDevicePackage.Literals.DEVICE.getClass()), any(String.class))).thenReturn(device);
		
		Device dev = devService.updateDevice(device);
		assertNotNull(dev);
		assertEquals(device, dev);
	}
	
	/**
	 * 1. Updates a Device through the DevicesService --> this should trigger the ConfigAdmin and creates an instance of DeviceService
	 * 2. Retrieves the Device from the DeviceService --> this should give the same Device we updated
	 * 3. Re-updates the Device through the DevicesService --> this should trigger the modify method in the DeviceService
	 * 4. Re-retrieves the Device from the DeviceService --> the Device should be the same as the newly updated Device
	 * @throws InterruptedException
	 */
	@Test
	public void testUpdateDeviceOK() throws InterruptedException {
		setupServices();
		
		ServiceChecker<DeviceService> devChecker = createCheckerTrackedForCleanUp(DeviceService.class);
		devChecker.start();
		assertEquals(0, devChecker.getCurrentCreateCount(false));		
		
		Device newDevice = createDeviceSample(DeviceActivationType.ACTIVE);
		Device oldDevice = createDeviceSample(DeviceActivationType.INACTIVE);
		
		assertNotNull(repository);
		when(repository.getEObject(any(TOSDevicePackage.Literals.DEVICE.getClass()), any(String.class))).thenReturn(oldDevice)
		.thenReturn(newDevice).thenReturn(oldDevice);
		
		Device dev = devService.updateDevice(newDevice);
		assertNotNull(dev);
		assertEquals(newDevice, dev);
		
		logger.info("Starting waiting till DeviceService is injected");
		Thread.sleep(3000);
		
		assertNotNull(devChecker);
		assertTrue(devChecker.awaitCreation());
		assertEquals(1, devChecker.getCurrentCreateCount(true));
		
		deviceService = getService(DeviceService.class);
		assertNotNull(deviceService);
				
		Device currDevice = deviceService.getDevice();
		
		assertEquals(newDevice.getId(), currDevice.getId());
		assertEquals(newDevice, currDevice);
		assertEquals(LifeCycleDeviceType.UNPROVISIONED, currDevice.getLifeCycleType());		
		
		oldDevice.setActivationState(DeviceActivationType.INACTIVE);
		dev = devService.updateDevice(oldDevice);
		assertNotNull(dev);
		assertEquals(oldDevice, dev);
		
		logger.info("Starting waiting till DeviceService is modified");
		Thread.sleep(3000);
		
		assertNotNull(devChecker);
		assertEquals(1, devChecker.getCurrentCreateCount(true));
		
		
		deviceService = getService(DeviceService.class);
				
		currDevice = devService.getDeviceById("test01");
		assertEquals(LifeCycleDeviceType.UNPROVISIONED, currDevice.getLifeCycleType());	
		assertEquals(DeviceActivationType.INACTIVE, currDevice.getActivationState());	
		
		devService.removeDevice("test01");		
		assertTrue(devChecker.awaitRemoval());		
	}
	
	/**
	 * 1. Updates a Device through the DevicesService --> this triggers the activation of the corresponding DeviceService
	 * 2. Updates the DeviceConfiguration for that Device --> this delegates to the DeviceService
	 * 3. Retrieves the Device --> this should have the updated DeviceConfiguration as Configuration
	 * @throws InterruptedException
	 */
	@Test 
	public void testUpdateDevConfigOK() throws InterruptedException {
		setupServices();
		
		Device newDevice = createDeviceSample(DeviceActivationType.ACTIVE);
		Device oldDevice = createDeviceSample(DeviceActivationType.INACTIVE);
		
		assertNotNull(repository);
		when(repository.getEObject(any(TOSDevicePackage.Literals.DEVICE.getClass()), any(String.class))).thenReturn(oldDevice)
		.thenReturn(newDevice).thenReturn(newDevice).thenReturn(oldDevice);
		
		ServiceChecker<DeviceService> devChecker = createCheckerTrackedForCleanUp(DeviceService.class);
		devChecker.start();
		assertEquals(0, devChecker.getCurrentCreateCount(false));
		
		Device dev = devService.updateDevice(newDevice);
		assertNotNull(dev);
		assertEquals(newDevice, dev);
		
		verify(repository, times(1)).save(any(Device.class));
		
		assertNotNull(devChecker);
		assertEquals(1, devChecker.getCurrentCreateCount(true));
		
		deviceService = getService(DeviceService.class);
		assertNotNull(deviceService);
		
		logger.info("Starting waiting till DeviceService is injected");
		Thread.sleep(3000);
		
		DeviceConfiguration config = createConfigSample("conf_id");
		DeviceConfiguration c = devService.updateDeviceConfiguration("test01", config);
		
		verify(repository, times(3)).save(any(Device.class));
		
		assertNotNull(c);
		
		dev = deviceService.getDevice();
		assertEquals(config, dev.getConfiguration());
		
		devService.removeDevice("test01");		
		assertTrue(devChecker.awaitRemoval());		
	}
	
	
	
	
	/**
	 * 1. Updates a Device through the DevicesService --> this triggers the activation of the corresponding DeviceService
	 * 2. Updates the DeviceInfo for that Device --> this delegates to the DeviceService
	 * 3. Retrieves the Device --> this should have the updated DeviceInfo as DeviceInfo
	 * @throws InterruptedException 
	 */
	@Test
	public void testUpdatedDevInfoOK() throws InterruptedException {
		setupServices();
		
		Device newDevice = createDeviceSample(DeviceActivationType.ACTIVE);
		Device oldDevice = createDeviceSample(DeviceActivationType.INACTIVE);
		
		assertNotNull(repository);
		when(repository.getEObject(any(TOSDevicePackage.Literals.DEVICE.getClass()), any(String.class))).thenReturn(oldDevice)
		.thenReturn(newDevice).thenReturn(newDevice).thenReturn(oldDevice);
		
		ServiceChecker<DeviceService> devChecker = createCheckerTrackedForCleanUp(DeviceService.class);
		devChecker.start();
		assertEquals(0, devChecker.getCurrentCreateCount(false));
		
		Device dev = devService.updateDevice(newDevice);
		assertNotNull(dev);
		assertEquals(newDevice, dev);
		
		verify(repository, times(1)).save(any(Device.class));
		
		assertNotNull(devChecker);
		assertEquals(1, devChecker.getCurrentCreateCount(true));
				
		deviceService = getService(DeviceService.class);
		assertNotNull(deviceService);
				
		logger.info("Starting waiting till DeviceService is injected");
		Thread.sleep(3000);
		
		DeviceInfo info = createInfoSample();
		DeviceInfo updatedInfo = devService.updateDeviceInformation("test01", info);
		
		assertNotNull(updatedInfo);
		assertEquals(info, updatedInfo);
		verify(repository, times(2)).save(any(Device.class));
		
		dev = deviceService.getDevice();
		assertEquals(info, dev.getDeviceInformation());
		
		devService.removeDevice("test01");		
		assertTrue(devChecker.awaitRemoval());
	}
	
	
	/**
	 * Tests the implementation of getDataEntries method when the start and end dates are provided within 
	 * a valid range and there is an entry in the DB for such time range.
	 * @throws InterruptedException 
	 */
	@SuppressWarnings("unchecked")
	@Test
	public void testOKDataEntry() throws InterruptedException {
		
		setupServices();
		
		Device newDevice = createDeviceSample(DeviceActivationType.ACTIVE);
		Device oldDevice = createDeviceSample(DeviceActivationType.INACTIVE);
		
		ServiceChecker<EMFRepository> repoChecker = createCheckerTrackedForCleanUp(EMFRepository.class);
		repoChecker.start();
		assertEquals(1,  repoChecker.getCurrentCreateCount(false));
		
		assertNotNull(repository);
		when(repository.getEObject(any(TOSDevicePackage.Literals.DEVICE.getClass()), any(String.class))).thenReturn(oldDevice)
		.thenReturn(newDevice).thenReturn(newDevice).thenReturn(oldDevice);
		
		ServiceChecker<DeviceService> devChecker = createCheckerTrackedForCleanUp(DeviceService.class);
		devChecker.start();
		assertEquals(0, devChecker.getCurrentCreateCount(false));
		
		Device dev = devService.updateDevice(newDevice);
		assertNotNull(dev);
		assertEquals(newDevice, dev);
		
		verify(repository, times(1)).save(any(Device.class));
		
		assertNotNull(devChecker);
		assertEquals(1, devChecker.getCurrentCreateCount(true));
		assertTrue(devChecker.awaitCreation());
		
		deviceService = getService(DeviceService.class);
		assertNotNull(deviceService);	
				
		logger.info("Starting waiting till DeviceService is injected");
		Thread.sleep(3000);
		
		DataEntry dataEntry = createDataEntrySample();
		List<EObject> list = new ArrayList<EObject>();
		list.add(dataEntry);


		Mockito.when(repository.getEObjectsByQuery(Mockito.any(TOSDevicePackage.Literals.DATA_ENTRY.getClass()), 
				Mockito.any(IQuery.class), Mockito.anyMap())).thenReturn(list);		
		
		Date startDate = new Date(System.currentTimeMillis()-10000);
		Date endDate = new Date(System.currentTimeMillis() + 10000);
		String deviceId = "test01";
		List<DataEntry> result = devService.getDataEntries(deviceId, startDate, endDate);
//		
		assertNotNull(result);
		assertFalse(result.isEmpty());
		
		Mockito.verify(repository, times(1)).getEObjectsByQuery(Mockito.any(TOSDevicePackage.Literals.DATA_ENTRY.getClass()), 
				Mockito.any(IQuery.class), Mockito.anyMap());
		
		Mockito.verify(repository, times(1)).createQueryBuilder();
		Mockito.verify(builder, times(1)).column(TOSDevicePackage.Literals.DATA_ENTRY__TIMESTAMP);
		Mockito.verify(builder, times(1)).rangeQuery();
		Mockito.verify(builder, times(1)).build();

		devService.removeDevice("test01");		
		assertTrue(devChecker.awaitRemoval());		
	}
	
	
	/**
	 * Tests the implementation of the getDataEntries method when start and end dates are consistent, but 
	 * no entries has been found in the DB.
	 * @throws InterruptedException 
	 */
	@SuppressWarnings("unchecked")
	@Test
	public void testNoDataEntry() throws InterruptedException {
		
		setupServices();
		
		Device newDevice = createDeviceSample(DeviceActivationType.ACTIVE);
		Device oldDevice = createDeviceSample(DeviceActivationType.INACTIVE);
		
		assertNotNull(repository);
		when(repository.getEObject(any(TOSDevicePackage.Literals.DEVICE.getClass()), any(String.class))).thenReturn(oldDevice)
		.thenReturn(newDevice).thenReturn(newDevice).thenReturn(oldDevice);
			
		ServiceChecker<DeviceService> devChecker = createCheckerTrackedForCleanUp(DeviceService.class);
		devChecker.start();
		assertEquals(0, devChecker.getCurrentCreateCount(false));
		
		Device dev = devService.updateDevice(newDevice);
		assertNotNull(dev);
		assertEquals(newDevice, dev);
		
		verify(repository, times(1)).getEObject(any(TOSDevicePackage.Literals.DEVICE.getClass()), any(String.class));
		verify(repository, times(1)).save(any(Device.class));
		
		assertNotNull(devChecker);
		assertEquals(1, devChecker.getCurrentCreateCount(true));
		
		deviceService = getService(DeviceService.class);
		assertNotNull(deviceService);		
		
		logger.info("Starting waiting till DeviceService is injected");
		Thread.sleep(3000);
		
		DataEntry dataEntry = createDataEntrySample();
		List<EObject> list = new ArrayList<EObject>();
		list.add(dataEntry);

		Mockito.when(repository.getEObjectsByQuery(Mockito.any(TOSDevicePackage.Literals.DATA_ENTRY.getClass()), 
				Mockito.any(IQuery.class), Mockito.anyMap())).thenReturn(new LinkedList<DataEntry>());
		
		Date startDate = new Date(System.currentTimeMillis());
		Date endDate = new Date(System.currentTimeMillis() + 10000);
		String deviceId = "test01";
		List<DataEntry> result = devService.getDataEntries(deviceId, startDate, endDate);
		
		assertNotNull(result);
		assertTrue(result.isEmpty());
		
		Mockito.verify(repository, times(1)).getEObjectsByQuery(Mockito.any(TOSDevicePackage.Literals.DATA_ENTRY.getClass()), 
				Mockito.any(IQuery.class), Mockito.anyMap());
		
		Mockito.verify(repository, times(1)).createQueryBuilder();
		Mockito.verify(builder, times(1)).column(TOSDevicePackage.Literals.DATA_ENTRY__TIMESTAMP);
		Mockito.verify(builder, times(1)).rangeQuery();
		Mockito.verify(builder, times(1)).build();
		
		devService.removeDevice("test01");		
		assertTrue(devChecker.awaitRemoval());		
	}
	
	/**
	 * Tests the implementation of the getDataEntries when start and end dates are inconsistent.
	 * @throws InterruptedException 
	 */
	@SuppressWarnings("unchecked")
	@Test
	public void testInconsistentDate() throws InterruptedException {
		
		setupServices();
		
		Device newDevice = createDeviceSample(DeviceActivationType.ACTIVE);
		Device oldDevice = createDeviceSample(DeviceActivationType.INACTIVE);
		
		assertNotNull(repository);
		when(repository.getEObject(any(TOSDevicePackage.Literals.DEVICE.getClass()), any(String.class))).thenReturn(oldDevice)
		.thenReturn(newDevice).thenReturn(newDevice).thenReturn(oldDevice);
		
		ServiceChecker<DeviceService> devChecker = createCheckerTrackedForCleanUp(DeviceService.class);
		devChecker.start();
		assertEquals(0, devChecker.getCurrentCreateCount(false));
		
		Device dev = devService.updateDevice(newDevice);
		assertNotNull(dev);
		assertEquals(newDevice, dev);
		
		verify(repository, times(1)).save(any(Device.class));
		
		assertNotNull(devChecker);
		assertTrue(devChecker.awaitCreation());
		assertEquals(1, devChecker.getCurrentCreateCount(true));
		
		deviceService = getService(DeviceService.class);
		assertNotNull(deviceService);	
		
		logger.info("Starting waiting till DeviceService is injected");
		Thread.sleep(3000);
		
		DataEntry dataEntry = createDataEntrySample();
		List<EObject> list = new ArrayList<EObject>();
		list.add(dataEntry);

		Mockito.when(repository.getEObjectsByQuery(Mockito.any(TOSDevicePackage.Literals.DATA_ENTRY.getClass()), 
				Mockito.any(IQuery.class), Mockito.anyMap())).thenReturn(new LinkedList<DataEntry>());
		
		Date startDate = new Date(System.currentTimeMillis() + 10000);
		Date endDate = new Date(System.currentTimeMillis() - 10000);
		String deviceId = "test01";
		List<DataEntry> result = devService.getDataEntries(deviceId, startDate, endDate);
		
		assertNotNull(result);
		assertTrue(result.isEmpty());
		
		Mockito.verify(repository, times(0)).createQueryBuilder();
		Mockito.verify(builder, times(0)).column(TOSDevicePackage.Literals.DATA_ENTRY__TIMESTAMP);
		Mockito.verify(builder, times(0)).rangeQuery();
		Mockito.verify(builder, times(0)).build();
		
		
		devService.removeDevice("test01");		
		assertTrue(devChecker.awaitRemoval());		
	}
	
	@Test
	public void testSimulation() throws DeviceException, InterruptedException, InvalidSyntaxException {
		
		setupServices();
		
		Device newDevice = createDeviceSample(DeviceActivationType.ACTIVE);		
		Device oldDevice = createDeviceSample(DeviceActivationType.INACTIVE);
		
		assertNotNull(repository);
		when(repository.getEObject(any(TOSDevicePackage.Literals.DEVICE.getClass()), any(String.class))).thenReturn(oldDevice)
		.thenReturn(newDevice).thenReturn(newDevice).thenReturn(oldDevice);
		
		ServiceChecker<DeviceService> devChecker = createCheckerTrackedForCleanUp(DeviceService.class);
		devChecker.start();
		assertEquals(0, devChecker.getCurrentCreateCount(true));
		
		Device dev = devService.updateDevice(newDevice);
		assertNotNull(dev);
		assertEquals(newDevice, dev);
		
		verify(repository, times(1)).save(any(Device.class));	
		
		assertNotNull(devChecker);
		assertEquals(1, devChecker.getCurrentCreateCount(true));
		
		deviceService = getService(DeviceService.class);
		assertNotNull(deviceService);	
		
		logger.info("Starting waiting till DeviceService is injected");
		Thread.sleep(3000);
		
		DeviceConfiguration config = createConfigSample("conf_id");
		DeviceConfiguration c = devService.updateDeviceConfiguration("test01", config);
		verify(repository, times(3)).save(any(Device.class));
		
		assertNotNull(c);
		
		dev = deviceService.getDevice();
		assertEquals(config, dev.getConfiguration());
		
		devService.startDevice("test01");
		dev = deviceService.getDevice();
		assertEquals(LifeCycleDeviceType.RUNNING, dev.getLifeCycleType());
		
		devService.stopDevice("test01");
		dev = deviceService.getDevice();
		
		Thread.sleep(2000);
		assertEquals(LifeCycleDeviceType.PROVISIONED, dev.getLifeCycleType());
		
		devService.removeDevice("test01");
		assertTrue(devChecker.awaitRemoval());
	}
	
	private void setupServices() {
		
		Dictionary<String, Object> properties = new Hashtable<String, Object>();
		properties.put("repo_id", "tos.trafficos");
		registerServiceForCleanup(EMFRepository.class, new PrototypeServiceFactory<EMFRepository>() {

			@Override
			public EMFRepository getService(Bundle bundle, ServiceRegistration<EMFRepository> registration) {
				return repository;
			}

			@Override
			public void ungetService(Bundle bundle, ServiceRegistration<EMFRepository> registration, EMFRepository service) {
				repository.dispose();
			}
		}, properties);
		ServiceChecker<EMFRepository> repoChecker = createCheckerTrackedForCleanUp(EMFRepository.class);
		repoChecker.start();
		assertEquals(1,  repoChecker.getCurrentCreateCount(true));
		
		ServiceChecker<EventAdmin> adminChecker = createCheckerTrackedForCleanUp(EventAdmin.class);
		adminChecker.start();
		assertEquals(1,  adminChecker.getCurrentCreateCount(true));
				
		ServiceChecker<DeviceLifeCycleHandler> lifeCycleChecker = createCheckerTrackedForCleanUp(DeviceLifeCycleHandler.class);
		lifeCycleChecker.start();
		assertEquals(1,  lifeCycleChecker.getCurrentCreateCount(true));
		
		ServiceChecker<DevicesService> checker = createCheckerTrackedForCleanUp(DevicesService.class);
		checker.start();
		assertNotNull(checker);
		assertEquals(1, checker.getCurrentCreateCount(true));
		
		devService = getService(DevicesService.class);
		assertNotNull(devService);
		
		when(repository.createQueryBuilder()).thenReturn(builder);
		when(builder.column(TOSDevicePackage.Literals.DATA_ENTRY__TIMESTAMP)).thenReturn(builder);
		when(builder.rangeQuery()).thenReturn(builder);
		when(builder.build()).thenReturn(null);
	}
	
	private Device createDeviceSample(DeviceActivationType status) {
		
		Device device = TOSDeviceFactory.eINSTANCE.createDevice();
		device.setId("test01");
		device.setActivationState(status);
		device.setLifeCycleType(LifeCycleDeviceType.UNPROVISIONED);
		return device;
	}
	
	private DeviceConfiguration createConfigSample(String id) {
		
		DeviceConfiguration config = TOSDeviceFactory.eINSTANCE.createDeviceConfiguration();
		config.setId(id);
		Intersection intersection = TOSDeviceFactory.eINSTANCE.createIntersection();
		intersection.setId(UUID.randomUUID().toString());
		config.getIntersection().add(intersection);
		config.setCurrentIntersection(intersection);
		return config;
	}
	
	private DeviceInfo createInfoSample() {
		DeviceInfo info = TOSDeviceFactory.eINSTANCE.createDeviceInfo();
		info.setDescription("description");
		info.setHumanReadableName("human readable name");
		info.setShortName("alias");
		info.setTechnicalName("technical name");
		return info;
	}
	
	private DataEntry createDataEntrySample() {
		
		DataEntry dataEntry = TOSDeviceFactory.eINSTANCE.createDataEntry();
		dataEntry.setDevice("test01");
		dataEntry.setId("test_id");
		dataEntry.setIndex(1234);
		dataEntry.setTimestamp(new Date());
		
		DataValue dataValue = TOSDeviceFactory.eINSTANCE.createDataValue();
		dataValue.setValue("value_test");
		IdNameElement element = TOSDeviceFactory.eINSTANCE.createIdNameElement();
		element.setId("elId_test");
		element.setName("elName_test");
		dataValue.setElement(element);
		dataEntry.getValue().add(dataValue);
		
		return dataEntry;
	}
}
