/**
 * Copyright (c) 2012 - 2019 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package de.dim.trafficos.device.tests;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;
import static org.mockito.Matchers.any;
import static org.mockito.Mockito.when;

import java.io.IOException;
import java.util.Dictionary;
import java.util.Hashtable;
import java.util.UUID;

import org.gecko.core.tests.AbstractOSGiTest;
import org.gecko.core.tests.ServiceChecker;
import org.gecko.emf.repository.EMFRepository;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.Mock;
import org.mockito.runners.MockitoJUnitRunner;
import org.osgi.framework.Bundle;
import org.osgi.framework.FrameworkUtil;
import org.osgi.framework.InvalidSyntaxException;
import org.osgi.framework.PrototypeServiceFactory;
import org.osgi.framework.ServiceRegistration;
import org.osgi.service.cm.Configuration;
import org.osgi.service.cm.ConfigurationAdmin;
import org.osgi.service.event.EventAdmin;

import de.dim.trafficos.device.api.DeviceException;
import de.dim.trafficos.device.api.DeviceService;
import de.dim.trafficos.device.api.DevicesService;
import de.dim.trafficos.model.device.Device;
import de.dim.trafficos.model.device.DeviceActivationType;
import de.dim.trafficos.model.device.DeviceConfiguration;
import de.dim.trafficos.model.device.Intersection;
import de.dim.trafficos.model.device.LifeCycleDeviceType;
import de.dim.trafficos.model.device.TOSDeviceFactory;
import de.dim.trafficos.model.device.TOSDevicePackage;

/**
 * Tests the correct firing of LifeCycleDeviceType events
 * 
 * @author ilenia
 * @since Jun 28, 2019
 */
@RunWith(MockitoJUnitRunner.class)
public class LifeCycleEventsTest extends AbstractOSGiTest {
	
	@Mock
	private EMFRepository repository;
	
	private DevicesService devService;
	private DeviceService deviceService;

	/**
	 * Creates a new instance.
	 * @param bundleContext
	 */
	public LifeCycleEventsTest() {
		super(FrameworkUtil.getBundle(LifeCycleEventsTest.class).getBundleContext());
	}

	/* 
	 * (non-Javadoc)
	 * @see org.gecko.core.tests.AbstractOSGiTest#doBefore()
	 */
	@Override
	public void doBefore() {
		ConfigurationAdmin configAdmin = getConfigAdmin();
		try {
			Configuration[] configs = configAdmin.listConfigurations("("+DeviceService.PROP_DEVICE_ID+"=*)");
			if(configs != null) {
				for(Configuration c : configs) {
					c.delete();
				}
			}
			
		} catch (IOException | InvalidSyntaxException e) {
			e.printStackTrace();
		}
	}

	/* 
	 * (non-Javadoc)
	 * @see org.gecko.core.tests.AbstractOSGiTest#doAfter()
	 */
	@Override
	public void doAfter() {
		ConfigurationAdmin configAdmin = getConfigAdmin();
		try {
			Configuration[] configs = configAdmin.listConfigurations("("+DeviceService.PROP_DEVICE_ID+"=*)");
			if(configs != null) {
				for(Configuration c : configs) {
					c.delete();
				}
			}
			
		} catch (IOException | InvalidSyntaxException e) {
			e.printStackTrace();
		}
	}

	@Test
	public void testServiceCreation() {
		
		setupServices();
	}
	
	/**
	 * 1. Create a Device
	 * 2. Update the Device through the DevicesService 
	 * 3. Retrieve the Device Status with the DeviceService --> status should be UNPROVISIONED
	 * 4. Re-update the Device with a null Device
	 * 5. Retrieve the Device Status with the DeviceService --> status should be EXCEPTIONAL
	 * @throws InterruptedException 
	 * 
	 */
	@Test
	public void testUnprovisionedStatus() throws InterruptedException {
		
		setupServices();
		
		ServiceChecker<DeviceService> devChecker = createCheckerTrackedForCleanUp(DeviceService.class);
		devChecker.start();
		assertEquals(0, devChecker.getCurrentCreateCount(false));
		
		Device oldDevice = createDeviceSample(DeviceActivationType.INACTIVE);
		Device device = createDeviceSample(DeviceActivationType.ACTIVE);
		assertNotNull(repository);
		when(repository.getEObject(any(TOSDevicePackage.Literals.DEVICE.getClass()), any(String.class))).thenReturn(oldDevice);
		
		Device dev = devService.updateDevice(device);
		assertNotNull(dev);
		assertEquals(device, dev);
		
		assertEquals(1, devChecker.getCurrentCreateCount(true));
		
		deviceService = getService(DeviceService.class);
		assertNotNull(deviceService);
				
		Thread.sleep(3000);
		assertEquals(LifeCycleDeviceType.UNPROVISIONED, devService.getStatus("test01"));
	
		devService.removeDevice("test01");		
		assertTrue(devChecker.awaitRemoval());		
	}
	
	/**
	 * 1. Create a Device with a Configuration
	 * 2. Update the Device through the DevicesService 
	 * 3. Retrieve the Device Status with the DeviceService --> status should be PROVISIONED
	 * 4. Re-update the Device with a Device without Configuration
	 * 5. Retrieve the Device Status with the DeviceService --> status should be UNPROVISIONED
	 * @throws InterruptedException 
	 * 
	 */
	@Test
	public void testProvisionedStatus() throws InterruptedException {
		setupServices();
		
		ServiceChecker<DeviceService> devChecker = createCheckerTrackedForCleanUp(DeviceService.class);
		devChecker.start();
		assertEquals(0, devChecker.getCurrentCreateCount(false));
		
		Device oldDevice = createDeviceSample(DeviceActivationType.INACTIVE);
		Device device = createDeviceSample(DeviceActivationType.ACTIVE);
		DeviceConfiguration config = createConfigSample("config_id");
		device.setConfiguration(config);
		assertNotNull(repository);
		when(repository.getEObject(any(TOSDevicePackage.Literals.DEVICE.getClass()), any(String.class))).thenReturn(oldDevice)
		.thenReturn(device).thenReturn(device).thenReturn(oldDevice);
		
		Device dev = devService.updateDevice(device);
		assertNotNull(dev);
		assertEquals(device, dev);
		
		assertEquals(1, devChecker.getCurrentCreateCount(true));
		
		deviceService = getService(DeviceService.class);
		assertNotNull(deviceService);
				
		Thread.sleep(3000);
		assertEquals(LifeCycleDeviceType.PROVISIONED, devService.getStatus("test01"));
		
		dev = devService.updateDevice(oldDevice);
		Thread.sleep(3000);
		assertEquals(LifeCycleDeviceType.UNPROVISIONED, devService.getStatus("test01"));	
		
		devService.removeDevice("test01");		
		assertTrue(devChecker.awaitRemoval());		
	}
	
	/**
	 * 1. Create a Device with a Configuration
	 * 2. Update the Device through the DevicesService 
	 * 3. Retrieve the Device Status with the DeviceService --> status should be PROVISIONED
	 * 4. Start the Device (there is no Program to run)
	 * 5. Retrieve the Device Status with the DeviceService --> status should be PROVISIONED
	 * 6. Stop the Device (the simulation never actually started)
	 * 7. Retrieve the Device Status with the DeviceService --> status should be PROVISIONED
	 * @throws InterruptedException 
	 * @throws DeviceException 
	 */
	@Test
	public void testRunningStatus() throws InterruptedException, DeviceException {
		
		setupServices();
		
		ServiceChecker<DeviceService> devChecker = createCheckerTrackedForCleanUp(DeviceService.class);
		devChecker.start();
		assertEquals(0, devChecker.getCurrentCreateCount(false));
		
		Device oldDevice = createDeviceSample(DeviceActivationType.INACTIVE);
		Device device = createDeviceSample(DeviceActivationType.ACTIVE);
		DeviceConfiguration config = createConfigSample("config_id");
		device.setConfiguration(config);
		assertNotNull(repository);
		when(repository.getEObject(any(TOSDevicePackage.Literals.DEVICE.getClass()), any(String.class))).thenReturn(oldDevice)
		.thenReturn(device);
		
		Device dev = devService.updateDevice(device);
		assertNotNull(dev);
		assertEquals(device, dev);
		
		assertEquals(1, devChecker.getCurrentCreateCount(true));
		
		deviceService = getService(DeviceService.class);
		assertNotNull(deviceService);
				
		Thread.sleep(3000);
		assertEquals(LifeCycleDeviceType.PROVISIONED, devService.getStatus("test01"));
		dev = devService.getDeviceById("test01");
		assertNotNull(dev);
		
		assertTrue(deviceService.startDevice());
		assertEquals(LifeCycleDeviceType.RUNNING, devService.getStatus("test01"));		
		
		deviceService.stopDevice();
		assertEquals(LifeCycleDeviceType.PROVISIONED, devService.getStatus("test01"));
		
		devService.removeDevice("test01");		
		assertTrue(devChecker.awaitRemoval());		
	}
	
	private Device createDeviceSample(DeviceActivationType status) {
		
		Device device = TOSDeviceFactory.eINSTANCE.createDevice();
		device.setId("test01");
		device.setActivationState(status);
		device.setLifeCycleType(LifeCycleDeviceType.UNPROVISIONED);
		return device;
	}
	
	private DeviceConfiguration createConfigSample(String id) {
		
		DeviceConfiguration config = TOSDeviceFactory.eINSTANCE.createDeviceConfiguration();
		config.setId(id);
		Intersection intersection = TOSDeviceFactory.eINSTANCE.createIntersection();
		intersection.setId(UUID.randomUUID().toString());
		config.getIntersection().add(intersection);
		config.setCurrentIntersection(intersection);
		return config;
	}
	
	private void setupServices() {
		
		Dictionary<String, Object> properties = new Hashtable<String, Object>();
		properties.put("repo_id", "tos.trafficos");
		registerServiceForCleanup(EMFRepository.class, new PrototypeServiceFactory<EMFRepository>() {

			@Override
			public EMFRepository getService(Bundle bundle, ServiceRegistration<EMFRepository> registration) {
				return repository;
			}

			@Override
			public void ungetService(Bundle bundle, ServiceRegistration<EMFRepository> registration, EMFRepository service) {
				repository.dispose();
			}
		}, properties);
		ServiceChecker<EMFRepository> repoChecker = createCheckerTrackedForCleanUp(EMFRepository.class);
		repoChecker.start();
		assertEquals(1,  repoChecker.getCurrentCreateCount(true));
		
		ServiceChecker<EventAdmin> adminChecker = createCheckerTrackedForCleanUp(EventAdmin.class);
		adminChecker.start();
		assertEquals(1,  adminChecker.getCurrentCreateCount(true));		
		
		ServiceChecker<DevicesService> checker = createCheckerTrackedForCleanUp(DevicesService.class);
		checker.start();
		assertNotNull(checker);
		assertEquals(1, checker.getCurrentCreateCount(true));
		
		devService = getService(DevicesService.class);
		assertNotNull(devService);
	}
	
	
}
