/**
 * Copyright (c) 2012 - 2019 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package de.dim.trafficos.simulator.api;

import java.util.List;
import java.util.Map;

import de.dim.trafficos.model.device.ConflictingLane;
import de.dim.trafficos.model.device.Intersection;
import de.dim.trafficos.model.device.Phase;
import de.dim.trafficos.model.device.PhaseGroup;
import de.dim.trafficos.model.device.Program;
import de.dim.trafficos.model.device.ScheduleModeType;

/**
 * This interface provides some useful methods to build the SignalPrograms for the SignalGroup of an Intersection.
 * 
 * @author ilenia
 * @since Jun 17, 2019
 */
public interface SignalPlanService {
	
	/**
	 * The first step is to create the Phases. A Phase is an object which represent the lanes which can have the green
	 * light at the same time. Given an Intersection, this method creates a list of Phase EObject, defined by an ID 
	 * and the corresponding list of Lanes.
	 * The Phases are created as follow: we order the list of IncomingLane of an Intersection depending on their index,
	 * and then we add also all the PedestrianLane of the Intersection.
	 * Then we start from the first Lane in the list. This is added to the Lane of the current Phase we are creating.
	 * Then we start looping over the other lanes, adding them in case they are not in conflict with the ones already 
	 * in the Phase. After completing the loop we checks for other Phases, by starting again from the first IncomingLane
	 * but looking now at the other Lanes not starting from the second in the list, but from the third, and so on, till
	 * we have done all the possible loops.
	 * This is then repeated for each Lanes.
	 * This guarantees to find all the possible combinations of Phases. Then, depending on the provided strategy,
	 * all these Phases are returned, or the list if filtered accordingly to the provided strategy option.
	 * * 
	 * @param intersection the Intersection for which we want to compute the Phases
	 * @return a list of Phase EObject or an empty list, if some error occurred
	 */	
	public List<Phase> createPhases(Intersection intersection, String strategy);
	
	/**
	 * After creating the Phases, we would need to put them together in a consistent way, in order to be able to
	 * create Signal Programs. This method does that. It loops over the list of Phases of the provided Intersection
	 * and search for possible combinations of Phases, in such a way that all the Lanes are in at least one Phase
	 * at some point. It also assigns a "penalty" term to the PhaseGroup, which is computed as follow. 
	 * PhaseGroups in which no Lane is present in more than one Phase get a penalty term of 0. 
	 * If just one Lane is present in 2 Phases, the PhaseGroup gets a penalty term of 1.
	 * If a Lane is present in 3 phases, or 2 Lanes are present in 2 Phases, then the PhaseGroup gets a penalty term of 2,
	 * and so on. This penalty term can later be used to select the PhaseGroup to be used when constructing a 
	 * Signal Program. The PhaseGroups are appended to the provided Intersection. 
	 * 
	 * @param intersection the Intersection for which PhaseGroups have to be computed.
	 */
	public void createPhaseGroups(Intersection intersection);
	

	/**
	 * Provided a PhaseGroup, and a Duration, a SignalProgram is created. The Program will have the provided duration.
	 * The time will be split among the Phases in the PhaseGroup and the corresponding Transitions.
	 * Transitions duration depends on the some parameters, which can be defined through the <code>SignalPlanConstants<code>
	 * interface, such as the clearing time for pedestrians and for main and secondary roads.
	 * The time for the Phases will then be assigned according to the weight attribute of the Phases. Each Phase has a min and 
	 * max weight, which should be set as a percentage value (70 means the Phase will receive the 70% of the available program
	 * duration, etc). The min weight is the one used for the computation currently, but there are a min and a max to help
	 * decimal values to be properly rounded, since the duration has to be an Integer.
	 * The Program will then be built starting from the first Phase (based on the ID), then the corresponding Transition between 
	 * this Phase and the second one, and so on. 
	 *  
	 * @param phaseGroup the PhaseGroup
	 * @param programId the program ID
	 * @param duration the program duration
	 * @return the newly created Program, or null, if an error occurred.
	 */
	public Program createFixTimeProgram(PhaseGroup phaseGroup, String programId, int duration);
	
	
	/**
	 * Applies the provided Program to the Intersection TimeTable for the provided ScheduleModeType.
	 * These generates a list of DataEntry in the Program, which represent the cached DataEntry to look for during 
	 * a normal execution of the Program (namely without increase or decrease of a Phase duration based on some input 
	 * parameters). 
	 * 
	 * @param intersection
	 * @param program
	 * @param scheduleType
	 * @return a Map where the keys are the TX values (from 0 to the Program duration - 1), and the Values are a Maps with 
	 * the ConflictingLane as keys and the status of the corresponding SignalGroups as value.  
	 */
	public Map<Integer, Map<ConflictingLane, String>> applyProgram(Intersection intersection, Program program, ScheduleModeType scheduleType);

	
	/**
	 * Sets the attribute duration to the cached DataValue, in such a way to have an estimate (in sec) of 
	 * how long it will take for that SG to change status. This will be used to provide the user with info
	 * about the status of the Devices situated along his route.
	 * 
	 * @param program the Program for which we want to set the DataValue durations
	 */
	public void addCacheDataValueDurations(Program program);
}
