/**
 * Copyright (c) 2012 - 2019 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package de.dim.trafficos.simulator.api;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.List;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.BiConsumer;
import java.util.logging.Logger;
import java.util.stream.Collectors;

import org.eclipse.emf.ecore.util.EcoreUtil;

import de.dim.trafficos.model.device.CacheDataEntry;
import de.dim.trafficos.model.device.DataEntry;
import de.dim.trafficos.model.device.DeviceConfiguration;
import de.dim.trafficos.model.device.Intersection;
import de.dim.trafficos.model.device.Program;
import de.dim.trafficos.model.device.ScheduleModeType;
import de.dim.trafficos.model.device.SignalTable;
import de.dim.trafficos.model.device.TOSDeviceFactory;
import de.dim.trafficos.model.device.TimeTable;
import de.dim.trafficos.model.device.TimeTableEntry;

/**
 * Class to manage a Device Simulation
 * @author ilenia
 * @since Jun 26, 2019
 */
public class DeviceSimulator {
	
	private static final Logger logger = Logger.getLogger(DeviceSimulator.class.getName());
	
	
	private Intersection intersection;
	private Calendar lastEntry = null;
	private AtomicInteger numCycles = new AtomicInteger();
	private AtomicInteger counter = new AtomicInteger();
	private AtomicInteger cycleCounter = new AtomicInteger(-1);
	private BiConsumer<DataEntry, Integer> notifyConsumer;
	private final ScheduledExecutorService ses = Executors.newSingleThreadScheduledExecutor();
	private ScheduledFuture<?> simulationFuture;
	private ReentrantLock simLock = new ReentrantLock(true);
	private Date startTime;
	private Program program;
	private final DeviceStatusCallback statusCallback;
   
    
    /**
	 * Creates a new instance.
	 * @param deviceServiceImpl
	 */
	public DeviceSimulator(DeviceStatusCallback statusCallback) {
		this.statusCallback = statusCallback;
	}

		
	public void initializeSimulation(Intersection intersection) {
    	this.intersection = intersection;
    }
    
   
    public boolean startSimulation() {
		if (!isRunning()) {
			simulationFuture = ses.scheduleAtFixedRate(this::simulate, 0, 1, TimeUnit.SECONDS);
			statusCallback.statusChanged(true);
			return true;
		} else {
			logger.warning(String.format("[%s] Simulation for device is already running", intersection.getId()));
			return false;
		}
	}
	
    
	public boolean stopSimulation() {
		if (isRunning()) {
			simulationFuture.cancel(true);
			while(!simulationFuture.isDone()) {
				try {
					Thread.sleep(50l);
				} catch (InterruptedException e) {
					logger.severe(String.format("[%s] Simulation stopping was interrupted", intersection.getId()));
				}
			}
			ses.shutdown(); 
		    try {
		        if (!ses.awaitTermination(60, TimeUnit.SECONDS)) {
		        	ses.shutdownNow(); 
		        }
		    } catch (InterruptedException ie) {
		    	ses.shutdownNow();
//		        Thread.currentThread().interrupt();
		    }	
		  statusCallback.statusChanged(false);
		  return true;
		} else {
			logger.warning(String.format("[%s] Simulation for device is not running", intersection.getId()));
			return false;
		}
	}
	

	public void setNotifyConsumer(BiConsumer<DataEntry, Integer> notifyConsumer) {
		this.notifyConsumer = notifyConsumer;
	}
	
	public boolean isRunning() {
		return simulationFuture != null && !simulationFuture.isDone();
	}
	
	public Date getStartTime() {
		return this.startTime;
	}
	
	public int getNumCycles() {
		return numCycles.get();
	}
	
	public Program getRunningProgram() {
		return program;
	}
		
	private void simulate() {
		if (simLock.tryLock()) {
			try {
				doSimulate();
			} finally {
				simLock.unlock();
			}
		} else {
			logger.warning(String.format("[%s] Simulation step is currently in progress, waiting", intersection.getId()));
		}		
	}
	
	private void doSimulate() {
		Calendar current = GregorianCalendar.getInstance();
		TimeTableEntry tte = getEntry(current);
		if (tte == null) {
			logger.warning(String.format("[%s] There is no time table for the current ScheduleMode '%s'", intersection.getId(), current.getTimeInMillis()));
			return;
		}		
		long currentSecond = current.getTimeInMillis() / 1000;
		long lastSecond = lastEntry == null ? currentSecond : lastEntry.getTimeInMillis() / 1000;
		int diffSeconds = (int)(currentSecond - lastSecond);
		if (diffSeconds > 1) {
			for (int i = 0; i <= diffSeconds; i++) {
				doSimulateSecond(lastSecond + i, tte);
			}
		} else {
			doSimulateSecond(currentSecond, tte);
		}
	}
	
	private void doSimulateSecond(long currentSecond, TimeTableEntry entry) {
		Calendar current = GregorianCalendar.getInstance();		
		current.setTimeInMillis(currentSecond * 1000);
		Program p = entry.getProgram();
		if(p == null) {
			logger.severe(String.format("[%s] There is no Program to start for such Intersection", intersection.getId()));
			stopSimulation();
			return;
		}
		if(p.getLength() <= 0) {
			logger.severe(String.format("[%s] The Program has length <= 0. Cannot start a simulation for such Intersection", intersection.getId()));
			stopSimulation();			
			return;
		}
		program = p;
		runProgram(p, current);		
	}
	
	private void runProgram(Program program, Calendar current) {
		
		int cycleLength = program.getLength();
		if((counter.get() % cycleLength) == 0) {
			numCycles.incrementAndGet();
		}
		int cnt = counter.incrementAndGet();		
		int currentTx = cycleCounter.incrementAndGet() % cycleLength;
		cycleCounter.set(currentTx);
		SignalTable sigTab = program.getSignalTable();
		List<CacheDataEntry> cacheDataEntry = new ArrayList<CacheDataEntry>(sigTab.getCacheDataEntry());
		cacheDataEntry = cacheDataEntry.stream().filter(de->(String.valueOf(currentTx).equals(de.getId())))
				.collect(Collectors.toList());
		if(cacheDataEntry.isEmpty()) {
			logger.severe(String.format("[%s] No cached DataEntry for TX %d", intersection.getId(), currentTx));
			stopSimulation();
			return;
		}
		if(cacheDataEntry.size() > 1) {
			logger.severe(String.format("[%s] More than one cached DataEntry for TX %d", intersection.getId(), currentTx));
			stopSimulation();
			return;
		}
		CacheDataEntry cde = cacheDataEntry.get(0);
		if(currentTx == 0) {
			startTime = current.getTime();
		}
		
		DataEntry dataEntry = TOSDeviceFactory.eINSTANCE.createDataEntry();
		dataEntry.setIndex(cnt);
		dataEntry.setTimestamp(current.getTime());
		if(intersection.eContainer() instanceof DeviceConfiguration) {
			DeviceConfiguration config = (DeviceConfiguration) intersection.eContainer();
			dataEntry.setConfiguration(config.getId());
		}
		else {
			dataEntry.setConfiguration(intersection.getId());
		}		
		dataEntry.setDevice(intersection.getId());
		dataEntry.getValue().addAll(EcoreUtil.copyAll(cde.getValue()));		
				
		if (notifyConsumer != null) {
			notifyConsumer.accept(dataEntry, Integer.valueOf(currentTx));
		}
		else {
			logger.severe(String.format("[%s] NotifyConsumer is null", intersection.getId()));
		}
	}

	private TimeTableEntry getEntry(Calendar current) {
		TimeTable table = intersection.getTimeTable();
		ScheduleModeType schedMode = isWeekend(current) ? ScheduleModeType.WEEKEND : ScheduleModeType.WORKING_DAY;
		TimeTableEntry entry = table.getEntry().stream().filter(tte->schedMode.equals(tte.getMode())).findFirst().orElse(null);
		if(entry == null) {
			logger.severe(String.format("[%s] No TimeTableEntry found for current time", intersection.getId()));
		}
		return entry;
	}
	
	private boolean isWeekend(Calendar timestamp) {
		int weekday = timestamp.get(Calendar.DAY_OF_WEEK);
		return Calendar.SUNDAY == weekday || Calendar.SATURDAY == weekday;
	}
}
