/**
 * Copyright (c) 2012 - 2019 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package de.dim.trafficos.simulator.impl;

import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;
import java.util.logging.Logger;
import java.util.stream.Collectors;

import org.eclipse.emf.ecore.EObject;
import org.gecko.emf.repository.EMFRepository;
import org.gecko.emf.repository.query.IQuery;
import org.gecko.emf.repository.query.IQueryBuilder;
import org.gecko.emf.repository.query.QueryRepository;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;

import de.dim.trafficos.model.device.Intersection;
import de.dim.trafficos.model.device.Output;
import de.dim.trafficos.model.device.Position;
import de.dim.trafficos.model.device.TOSDeviceFactory;
import de.dim.trafficos.simulator.api.ConflictingAreaService;

/**
 * 
 * @author ilenia
 * @since Jul 29, 2019
 */
//@Component
public class ConflictingAreaServiceImpl implements ConflictingAreaService {
	
	@Reference
	private EMFRepository repo;
	
	private Logger logger = Logger.getLogger(ConflictingAreaServiceImpl.class.getName());
	private double dist = 30.; //distance in meters from the signal group position to the vertices of the rectangle around it
	private double earthR = 6371e3; // Earth radius [metres]

	
	/* 
	 * (non-Javadoc)
	 * @see de.dim.trafficos.simulator.api.ConflictingAreaService#computeConflictingAreas(de.dim.trafficos.model.device.Intersection)
	 */
	@Override
	public List<Position> computeConflictingAreas(Intersection intersection) {
		List<Position> vertices = new ArrayList<Position>();
		if(intersection == null) {
			logger.severe("Cannot compute conflicting areas for a null Intersection.");
			return vertices;
		}
		List<Output> outputs = intersection.getOutput();
		if(outputs.isEmpty()) {
			logger.warning("Cannot compute conflicting areas for an Intersection without any Output");
			return vertices;
		}
		for(Output out : outputs) {
			List<Position> positions = out.getLocation().stream().filter(l->(l instanceof Position))
					.map(l->((Position) l)).collect(Collectors.toList());
			if(positions.isEmpty()) {
				logger.warning(String.format("[%s] Cannot compute area around SignalGroup because of missing Position", out.getId()));
			}
			else {
				Position pos = positions.get(0);
				for(int i = 0; i < 4; i++) {
					vertices.add(computeVertex(pos, pos.getBearing() + 45. + i*90.));
				}
			}
		}		
		return vertices;
	}

	/**
	 * This computes the geo coordinates of the point distant <code>dist<code> from the provided Position and with the 
	 * provided bearing
	 * @param startingPoint the Position of the point from which we want to compute the other point
	 * @param bear the bearing of the final point
	 * @return the Position of a point which is distant <code>dist<code> from the provided point and with the provided bearing 
	 */
	private Position computeVertex(Position startingPoint, double bear) {		
		
		double startingLat = startingPoint.getLatitude();
		double startingLng = startingPoint.getLongitude();
		
		double lat = Math.asin(Math.sin(degToRad(startingLat))*Math.cos(dist/earthR)+Math.cos(degToRad(startingLat))
		*Math.sin(dist/earthR)*Math.cos(degToRad(bear)));
		
		double lng = degToRad(startingLng) + Math.atan2(Math.sin(degToRad(bear))*Math.sin(dist/earthR)
				*Math.cos(degToRad(startingLat)),
                Math.cos(dist/earthR)-Math.sin(degToRad(startingLat))*Math.sin(lat));
		
		Position pos = TOSDeviceFactory.eINSTANCE.createPosition();
		pos.setLatitude(radToDeg(lat));
		pos.setLongitude(radToDeg(lng));
		return pos;
	}
	
	/**
	 * @param pos the Position we want to know which ConflictingAreas include
	 * @return the list of ConflictingAreas which includes the provided Position
	 */
	private List<EObject> getConflictingAreas(Position pos) {
		QueryRepository queryRepository = (QueryRepository) repo;
		IQueryBuilder query = queryRepository.createQueryBuilder();
//		queryRepository.getEObjectsByQuery(eClass, query); //this should return the list of Rectangles which overlap with the given point
		return new LinkedList<EObject>();
	}
	
	private double degToRad(double deg) {
		return deg*Math.PI/180.;
	}
	
	private double radToDeg(double rad) {
		return rad*180./Math.PI;
	}
}
