/**
 * Copyright (c) 2012 - 2018 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package de.dim.trafficos.simulator.tests;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.gecko.core.tests.AbstractOSGiTest;
import org.gecko.core.tests.ServiceChecker;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.runners.MockitoJUnitRunner;
import org.osgi.framework.FrameworkUtil;

import de.dim.trafficos.model.device.Intersection;
import de.dim.trafficos.model.device.Phase;
import de.dim.trafficos.model.device.PhaseGroup;
import de.dim.trafficos.simulator.api.IntersectionConstants;
import de.dim.trafficos.simulator.api.IntersectionService;
import de.dim.trafficos.simulator.api.SignalPlanConstants;
import de.dim.trafficos.simulator.api.SignalPlanService;


/**
 * Tests the implementation of the SignalPlanService methods for creating the Phases for a 4-arms intersection.
 * @since 1.0
 */
@RunWith(MockitoJUnitRunner.class)
public class SigPlanPhases4ArmsTest extends AbstractOSGiTest {
	
	private IntersectionService service;
	private SignalPlanService sigService;
	private Map<Integer, String> options;
	
	/**
	 * Creates a new instance.
	 * @param bundleContext
	 */
	public SigPlanPhases4ArmsTest() {
		super(FrameworkUtil.getBundle(SigPlanPhases4ArmsTest.class).getBundleContext());
	}

	/**
	 * Here you can put everything you want to be executed before every test
	 */
	public void doBefore() {
		
		
		
		options = new HashMap<Integer, String>();
	}
	
	/**
	 * Here you can put everything you want to be executed after every test
	 */
	public void doAfter() {
		
	}
	
	private void setupServices() {
		ServiceChecker<IntersectionService> checker = createCheckerTrackedForCleanUp(IntersectionService.class);
		checker.start();
		assertNotNull(checker);
		assertEquals(1, checker.getCurrentCreateCount(true));
		
		service = getService(IntersectionService.class);
		assertNotNull(service);
		
		ServiceChecker<SignalPlanService> sigChecker = createCheckerTrackedForCleanUp(SignalPlanService.class);
		sigChecker.start();
		assertNotNull(sigChecker);
		assertEquals(1, sigChecker.getCurrentCreateCount(true));
		
		sigService = getService(SignalPlanService.class);
		assertNotNull(sigService);	
	}
	
	/**
	 * All the possible Phases for a 4 Arms Intersection of the where the roads are of types:
	 * <code>IntersectionConstants.MAIN_STRAIGHT_TURNS_SEP<code>
	 * <code>IntersectionConstants.SEC_STRAIGHT_TURNS_SEP<code>
	 * <code>IntersectionConstants.MAIN_STRAIGHT_TURNS_SEP<code>
	 * <code>IntersectionConstants.SEC_STRAIGHT_TURNS_SEP<code>
	 */
	@Test 
	public void test4ArmsAllSepAllPhases() {
		setupServices();
		options.put(0, IntersectionConstants.MAIN_STRAIGHT_TURNS_SEP);
		options.put(1, IntersectionConstants.SEC_STRAIGHT_TURNS_SEP);
		options.put(2, IntersectionConstants.MAIN_STRAIGHT_TURNS_SEP);
		options.put(3, IntersectionConstants.SEC_STRAIGHT_TURNS_SEP);
		Intersection intersection = service.createIntersection(options);
	
		List<Phase> phases = sigService.createPhases(intersection, SignalPlanConstants.ALL_PHASES);
		phases = phases.stream().sorted(Comparator.comparing(Phase::getId)).collect(Collectors.toList());
		assertFalse(phases.isEmpty());
		assertEquals(39, phases.size());		
		assertEquals(39, intersection.getPhase().size());
		
		sigService.createPhaseGroups(intersection);
		List<PhaseGroup> phGroups = intersection.getPhaseGroup().stream().sorted(Comparator.comparing(PhaseGroup::getPenalty))
				.collect(Collectors.toList());
		assertEquals(35, phGroups.size());		
	}
	
	/**
	 * All the possible Phases WITHOUT PEDESTRIAN for a 4 Arms Intersection of the where the roads are of types:
	 * <code>IntersectionConstants.MAIN_STRAIGHT_TURNS_SEP<code>
	 * <code>IntersectionConstants.SEC_STRAIGHT_TURNS_SEP<code>
	 * <code>IntersectionConstants.MAIN_STRAIGHT_TURNS_SEP<code>
	 * <code>IntersectionConstants.SEC_STRAIGHT_TURNS_SEP<code>
	 */
	@Test 
	public void test4ArmsAllSepNoPed() {
		setupServices();
		options.put(0, IntersectionConstants.MAIN_STRAIGHT_TURNS_SEP);
		options.put(1, IntersectionConstants.SEC_STRAIGHT_TURNS_SEP);
		options.put(2, IntersectionConstants.MAIN_STRAIGHT_TURNS_SEP);
		options.put(3, IntersectionConstants.SEC_STRAIGHT_TURNS_SEP);
		Intersection intersection = service.createIntersection(options);
	
		List<Phase> phases = sigService.createPhases(intersection, SignalPlanConstants.PHASES_WITHOUT_PEDESTRIAN);
		phases = phases.stream().sorted(Comparator.comparing(Phase::getId)).collect(Collectors.toList());
		assertFalse(phases.isEmpty());
		assertEquals(16, phases.size());		
		assertEquals(16, intersection.getPhase().size());
		
		for(Phase p : phases) {
			assertEquals(4, p.getLane().size());
			assertEquals(phases.size()-1, p.getTransition().size());
		}		
	}
	
	/**
	 * All the possible Phases WITHOUT LANES WITH ONLY PEDESTRIAN  for a 4 Arms Intersection of the where the roads are of types:
	 * <code>IntersectionConstants.MAIN_STRAIGHT_TURNS_SEP<code>
	 * <code>IntersectionConstants.SEC_STRAIGHT_TURNS_SEP<code>
	 * <code>IntersectionConstants.MAIN_STRAIGHT_TURNS_SEP<code>
	 * <code>IntersectionConstants.SEC_STRAIGHT_TURNS_SEP<code>
	 */
	@Test 
	public void test4ArmsAllSepNotOnlyPed() {
		setupServices();
		options.put(0, IntersectionConstants.MAIN_STRAIGHT_TURNS_SEP);
		options.put(1, IntersectionConstants.SEC_STRAIGHT_TURNS_SEP);
		options.put(2, IntersectionConstants.MAIN_STRAIGHT_TURNS_SEP);
		options.put(3, IntersectionConstants.SEC_STRAIGHT_TURNS_SEP);
		Intersection intersection = service.createIntersection(options);
	
		List<Phase> phases = sigService.createPhases(intersection, SignalPlanConstants.PHASES_NOT_ONLY_PEDESTRIAN);
		phases = phases.stream().sorted(Comparator.comparing(Phase::getId)).collect(Collectors.toList());
		assertFalse(phases.isEmpty());
		assertEquals(38, phases.size());		
		assertEquals(38, intersection.getPhase().size());
		
		for(Phase p : phases) {
			assertEquals(4, p.getLane().size());
			assertEquals(phases.size()-1, p.getTransition().size());
		}
	}
	
	/**
	 * All the possible Phases for a 4 Arms Intersection of the where the roads are of types:
	 * <code>IntersectionConstants.MAIN_STRAIGHT_TURNS_MERGE<code>
	 * <code>IntersectionConstants.SEC_STRAIGHT_TURNS_MERGE<code>
	 * <code>IntersectionConstants.MAIN_STRAIGHT_TURNS_MERGE<code>
	 * <code>IntersectionConstants.SEC_STRAIGHT_TURNS_MERGE<code>
	 */
	@Test 
	public void test4ArmsAllMergeAllPhases() {
		setupServices();
		options.put(0, IntersectionConstants.MAIN_STRAIGHT_TURNS_MERGE);
		options.put(1, IntersectionConstants.SEC_STRAIGHT_TURNS_MERGE);
		options.put(2, IntersectionConstants.MAIN_STRAIGHT_TURNS_MERGE);
		options.put(3, IntersectionConstants.SEC_STRAIGHT_TURNS_MERGE);
		Intersection intersection = service.createIntersection(options);
	
		List<Phase> phases = sigService.createPhases(intersection, SignalPlanConstants.ALL_PHASES);
		phases = phases.stream().sorted(Comparator.comparing(Phase::getId)).collect(Collectors.toList());
		assertFalse(phases.isEmpty());
		assertEquals(3, phases.size());		
		assertEquals(3, intersection.getPhase().size());
		
		for(Phase p : phases) {
			assertEquals(4, p.getLane().size());
			assertEquals(phases.size()-1, p.getTransition().size());
		}
	}
	
	/**
	 * All the possible Phases WITHOUT PEDESTRIAN for a 4 Arms Intersection of the where the roads are of types:
	 * <code>IntersectionConstants.MAIN_STRAIGHT_TURNS_MERGE<code>
	 * <code>IntersectionConstants.SEC_STRAIGHT_TURNS_MERGE<code>
	 * <code>IntersectionConstants.MAIN_STRAIGHT_TURNS_MERGE<code>
	 * <code>IntersectionConstants.SEC_STRAIGHT_TURNS_MERGE<code>
	 */
	@Test 
	public void test4ArmsAllMergeNoPed() {
		setupServices();
		options.put(0, IntersectionConstants.MAIN_STRAIGHT_TURNS_MERGE);
		options.put(1, IntersectionConstants.SEC_STRAIGHT_TURNS_MERGE);
		options.put(2, IntersectionConstants.MAIN_STRAIGHT_TURNS_MERGE);
		options.put(3, IntersectionConstants.SEC_STRAIGHT_TURNS_MERGE);
		Intersection intersection = service.createIntersection(options);
	
		List<Phase> phases = sigService.createPhases(intersection, SignalPlanConstants.PHASES_WITHOUT_PEDESTRIAN);
		phases = phases.stream().sorted(Comparator.comparing(Phase::getId)).collect(Collectors.toList());
		assertFalse(phases.isEmpty());
		assertEquals(2, phases.size());		
		assertEquals(2, intersection.getPhase().size());
		
		for(Phase p : phases) {
			assertEquals(2, p.getLane().size());
			assertEquals(phases.size()-1, p.getTransition().size());
		}
	}
	
	/**
	 * All the possible Phases WITHOUT LANES WITH ONLY PEDESTRIAN for a 4 Arms Intersection of the where 
	 * the roads are of types:
	 * <code>IntersectionConstants.MAIN_STRAIGHT_TURNS_MERGE<code>
	 * <code>IntersectionConstants.SEC_STRAIGHT_TURNS_MERGE<code>
	 * <code>IntersectionConstants.MAIN_STRAIGHT_TURNS_MERGE<code>
	 * <code>IntersectionConstants.SEC_STRAIGHT_TURNS_MERGE<code>
	 */
	@Test 
	public void test4ArmsAllMergeNotOnlyPed() {
		setupServices();
		options.put(0, IntersectionConstants.MAIN_STRAIGHT_TURNS_MERGE);
		options.put(1, IntersectionConstants.SEC_STRAIGHT_TURNS_MERGE);
		options.put(2, IntersectionConstants.MAIN_STRAIGHT_TURNS_MERGE);
		options.put(3, IntersectionConstants.SEC_STRAIGHT_TURNS_MERGE);
		Intersection intersection = service.createIntersection(options);
	
		List<Phase> phases = sigService.createPhases(intersection, SignalPlanConstants.PHASES_NOT_ONLY_PEDESTRIAN);
		phases = phases.stream().sorted(Comparator.comparing(Phase::getId)).collect(Collectors.toList());
		assertFalse(phases.isEmpty());
		assertEquals(2, phases.size());		
		assertEquals(2, intersection.getPhase().size());
		
		for(Phase p : phases) {
			assertEquals(4, p.getLane().size());
			assertEquals(phases.size()-1, p.getTransition().size());
		}
	}
	
	
	/**
	 * All the possible Phases for a 4 Arms Intersection of the where the roads are of types:
	 * <code>IntersectionConstants.MAIN_STRAIGHT_LEFT_MERGE<code>
	 * <code>IntersectionConstants.SEC_STRAIGHT_TURNS_SEP<code>
	 * <code>IntersectionConstants.MAIN_STRAIGHT_LEFT_MERGE<code>
	 * <code>IntersectionConstants.SEC_STRAIGHT_TURNS_SEP<code>
	 */
	@Test 
	public void test4ArmsMainLeftMergeAllPhases() {
		setupServices();
		options.put(0, IntersectionConstants.MAIN_STRAIGHT_LEFT_MERGE);
		options.put(1, IntersectionConstants.SEC_STRAIGHT_TURNS_SEP);
		options.put(2, IntersectionConstants.MAIN_STRAIGHT_LEFT_MERGE);
		options.put(3, IntersectionConstants.SEC_STRAIGHT_TURNS_SEP);
		Intersection intersection = service.createIntersection(options);
	
		List<Phase> phases = sigService.createPhases(intersection, SignalPlanConstants.ALL_PHASES);
		phases = phases.stream().sorted(Comparator.comparing(Phase::getId)).collect(Collectors.toList());
		assertFalse(phases.isEmpty());
		assertEquals(30, phases.size());		
		assertEquals(30, intersection.getPhase().size());
		List<String> p3Lanes = new ArrayList<String>();
		p3Lanes.add("PH_0");
		p3Lanes.add("PH_8");
		p3Lanes.add("PH_10");
		p3Lanes.add("PH_21");
		p3Lanes.add("PH_22");
		p3Lanes.add("PH_24");
		p3Lanes.add("PH_28");
		p3Lanes.add("PH_29");
		for(Phase p : phases) {
			if(p3Lanes.contains(p.getId())) {
				assertEquals(3, p.getLane().size());
			}
			else {
				assertEquals(4, p.getLane().size());
			}
			assertEquals(phases.size()-1, p.getTransition().size());
		}
	}
	
	/**
	 * All the possible Phases WITHOUT PEDESTRIAN for a 4 Arms Intersection of the where the roads are of types:
	 * <code>IntersectionConstants.MAIN_STRAIGHT_LEFT_MERGE<code>
	 * <code>IntersectionConstants.SEC_STRAIGHT_TURNS_SEP<code>
	 * <code>IntersectionConstants.MAIN_STRAIGHT_LEFT_MERGE<code>
	 * <code>IntersectionConstants.SEC_STRAIGHT_TURNS_SEP<code>
	 */
	@Test 
	public void test4ArmsMainLeftMergeNoPed() {
		setupServices();
		options.put(0, IntersectionConstants.MAIN_STRAIGHT_LEFT_MERGE);
		options.put(1, IntersectionConstants.SEC_STRAIGHT_TURNS_SEP);
		options.put(2, IntersectionConstants.MAIN_STRAIGHT_LEFT_MERGE);
		options.put(3, IntersectionConstants.SEC_STRAIGHT_TURNS_SEP);
		Intersection intersection = service.createIntersection(options);
	
		List<Phase> phases = sigService.createPhases(intersection, SignalPlanConstants.PHASES_WITHOUT_PEDESTRIAN);
		phases = phases.stream().sorted(Comparator.comparing(Phase::getId)).collect(Collectors.toList());
		assertFalse(phases.isEmpty());
		assertEquals(12, phases.size());		
		assertEquals(12, intersection.getPhase().size());
		List<String> p3Lanes = new ArrayList<String>();
		p3Lanes.add("PH_0");
		p3Lanes.add("PH_9");
		for(Phase p : phases) {
			if(p.getId().equals("PH_5")) {
				assertEquals(2, p.getLane().size());
			}
			else if(p3Lanes.contains(p.getId())) {
				assertEquals(3, p.getLane().size());
			}
			else {
				assertEquals(4, p.getLane().size());
			}
			assertEquals(phases.size()-1, p.getTransition().size());
		}
	}
	
	/**
	 * All the possible Phases WITHOUT LANES WITH ONLY PEDESTRIAN for a 4 Arms Intersection of the where 
	 * the roads are of types:
	 * <code>IntersectionConstants.MAIN_STRAIGHT_LEFT_MERGE<code>
	 * <code>IntersectionConstants.SEC_STRAIGHT_TURNS_SEP<code>
	 * <code>IntersectionConstants.MAIN_STRAIGHT_LEFT_MERGE<code>
	 * <code>IntersectionConstants.SEC_STRAIGHT_TURNS_SEP<code>
	 */
	@Test 
	public void test4ArmsMainLeftMergeNotOnlyPed() {
		setupServices();
		options.put(0, IntersectionConstants.MAIN_STRAIGHT_LEFT_MERGE);
		options.put(1, IntersectionConstants.SEC_STRAIGHT_TURNS_SEP);
		options.put(2, IntersectionConstants.MAIN_STRAIGHT_LEFT_MERGE);
		options.put(3, IntersectionConstants.SEC_STRAIGHT_TURNS_SEP);
		Intersection intersection = service.createIntersection(options);
	
		List<Phase> phases = sigService.createPhases(intersection, SignalPlanConstants.PHASES_NOT_ONLY_PEDESTRIAN);
		phases = phases.stream().sorted(Comparator.comparing(Phase::getId)).collect(Collectors.toList());
		assertFalse(phases.isEmpty());
		assertEquals(29, phases.size());		
		assertEquals(29, intersection.getPhase().size());
		List<String> p3Lanes = new ArrayList<String>();
		p3Lanes.add("PH_0");
		p3Lanes.add("PH_8");
		p3Lanes.add("PH_10");
		p3Lanes.add("PH_21");
		p3Lanes.add("PH_22");
		p3Lanes.add("PH_24");
		p3Lanes.add("PH_28");
		p3Lanes.add("PH_27");
		for(Phase p : phases) {
			if(p3Lanes.contains(p.getId())) {
				assertEquals(3, p.getLane().size());
			}
			else {
				assertEquals(4, p.getLane().size());
			}
			assertEquals(phases.size()-1, p.getTransition().size());
		}
	}
	
	/**
	 * All the possible Phases for a 4 Arms Intersection of the where the roads are of types:
	 * <code>IntersectionConstants.MAIN_STRAIGHT_LEFT_MERGE<code>
	 * <code>IntersectionConstants.SEC_STRAIGHT_LEFT_MERGE<code>
	 * <code>IntersectionConstants.MAIN_STRAIGHT_LEFT_MERGE<code>
	 * <code>IntersectionConstants.SEC_STRAIGHT_LEFT_MERGE<code>
	 */
	@Test 
	public void test4ArmsAllLeftMergeAllPhases() {
		setupServices();
		options.put(0, IntersectionConstants.MAIN_STRAIGHT_LEFT_MERGE);
		options.put(1, IntersectionConstants.SEC_STRAIGHT_LEFT_MERGE);
		options.put(2, IntersectionConstants.MAIN_STRAIGHT_LEFT_MERGE);
		options.put(3, IntersectionConstants.SEC_STRAIGHT_LEFT_MERGE);
		Intersection intersection = service.createIntersection(options);
	
		List<Phase> phases = sigService.createPhases(intersection, SignalPlanConstants.ALL_PHASES);
		phases = phases.stream().sorted(Comparator.comparing(Phase::getId)).collect(Collectors.toList());
		assertFalse(phases.isEmpty());
		assertEquals(24, phases.size());		
		assertEquals(24, intersection.getPhase().size());
		List<String> p4Lanes = new ArrayList<String>();
		p4Lanes.add("PH_1");
		p4Lanes.add("PH_12");
		p4Lanes.add("PH_21");
		p4Lanes.add("PH_7");
		for(Phase p : phases) {
			if(p4Lanes.contains(p.getId())) {
				assertEquals(4, p.getLane().size());
			}
			else {
				assertEquals(3, p.getLane().size());
			}
			assertEquals(phases.size()-1, p.getTransition().size());
		}	
	}
	
	/**
	 * All the possible Phases WITHOUT PEDESTRIAN for a 4 Arms Intersection of the where the roads are of types:
	 * <code>IntersectionConstants.MAIN_STRAIGHT_LEFT_MERGE<code>
	 * <code>IntersectionConstants.SEC_STRAIGHT_LEFT_MERGE<code>
	 * <code>IntersectionConstants.MAIN_STRAIGHT_LEFT_MERGE<code>
	 * <code>IntersectionConstants.SEC_STRAIGHT_LEFT_MERGE<code>
	 */
	@Test 
	public void test4ArmsAllLeftMergeNoPed() {
		setupServices();
		options.put(0, IntersectionConstants.MAIN_STRAIGHT_LEFT_MERGE);
		options.put(1, IntersectionConstants.SEC_STRAIGHT_LEFT_MERGE);
		options.put(2, IntersectionConstants.MAIN_STRAIGHT_LEFT_MERGE);
		options.put(3, IntersectionConstants.SEC_STRAIGHT_LEFT_MERGE);
		Intersection intersection = service.createIntersection(options);
	
		List<Phase> phases = sigService.createPhases(intersection, SignalPlanConstants.PHASES_WITHOUT_PEDESTRIAN);
		phases = phases.stream().sorted(Comparator.comparing(Phase::getId)).collect(Collectors.toList());
		assertFalse(phases.isEmpty());
		assertEquals(7, phases.size());		
		assertEquals(7, intersection.getPhase().size());
		List<String> p2Lanes = new ArrayList<String>();
		p2Lanes.add("PH_3");
		p2Lanes.add("PH_5");
		for(Phase p : phases) {
			if(p.getId().equals("PH_1")) {
				assertEquals(4, p.getLane().size());
			}
			else if(p2Lanes.contains(p.getId())) {
				assertEquals(2, p.getLane().size());
			}
			else {
				assertEquals(3, p.getLane().size());
			}
			assertEquals(phases.size()-1, p.getTransition().size());
		}	
	}
	
	/**
	 * All the possible Phases WITHOUT LANES WITH ONLY PEDESTRIAN for a 4 Arms Intersection of the where 
	 * the roads are of types:
	 * <code>IntersectionConstants.MAIN_STRAIGHT_LEFT_MERGE<code>
	 * <code>IntersectionConstants.SEC_STRAIGHT_LEFT_MERGE<code>
	 * <code>IntersectionConstants.MAIN_STRAIGHT_LEFT_MERGE<code>
	 * <code>IntersectionConstants.SEC_STRAIGHT_LEFT_MERGE<code>
	 */
	@Test 
	public void test4ArmsAllLeftMergeNotOnlyPed() {
		setupServices();
		options.put(0, IntersectionConstants.MAIN_STRAIGHT_LEFT_MERGE);
		options.put(1, IntersectionConstants.SEC_STRAIGHT_LEFT_MERGE);
		options.put(2, IntersectionConstants.MAIN_STRAIGHT_LEFT_MERGE);
		options.put(3, IntersectionConstants.SEC_STRAIGHT_LEFT_MERGE);
		Intersection intersection = service.createIntersection(options);
	
		List<Phase> phases = sigService.createPhases(intersection, SignalPlanConstants.PHASES_NOT_ONLY_PEDESTRIAN);
		phases = phases.stream().sorted(Comparator.comparing(Phase::getId)).collect(Collectors.toList());
		assertFalse(phases.isEmpty());
		assertEquals(23, phases.size());		
		assertEquals(23, intersection.getPhase().size());
		List<String> p4Lanes = new ArrayList<String>();
		p4Lanes.add("PH_1");
		p4Lanes.add("PH_12");
		p4Lanes.add("PH_7");
		for(Phase p : phases) {
			if(p4Lanes.contains(p.getId())) {
				assertEquals(4, p.getLane().size());
			}
			else {
				assertEquals(3, p.getLane().size());
			}
			assertEquals(phases.size()-1, p.getTransition().size());
		}	
	}
	
	
	/**
	 * All the possible Phases for a 4 Arms Intersection of the where the roads are of types:
	 * <code>IntersectionConstants.MAIN_STRAIGHT_RIGHT_MERGE<code>
	 * <code>IntersectionConstants.SEC_STRAIGHT_RIGHT_MERGE<code>
	 * <code>IntersectionConstants.MAIN_STRAIGHT_RIGHT_MERGE<code>
	 * <code>IntersectionConstants.SEC_STRAIGHT_RIGHT_MERGE<code>
	 */
	@Test 
	public void test4ArmsAllRightMergeAllPhases() {
		setupServices();
		options.put(0, IntersectionConstants.MAIN_STRAIGHT_RIGHT_MERGE);
		options.put(1, IntersectionConstants.SEC_STRAIGHT_RIGHT_MERGE);
		options.put(2, IntersectionConstants.MAIN_STRAIGHT_RIGHT_MERGE);
		options.put(3, IntersectionConstants.SEC_STRAIGHT_RIGHT_MERGE);
		Intersection intersection = service.createIntersection(options);
	
		List<Phase> phases = sigService.createPhases(intersection, SignalPlanConstants.ALL_PHASES);
		phases = phases.stream().sorted(Comparator.comparing(Phase::getId)).collect(Collectors.toList());
		assertFalse(phases.isEmpty());
		assertEquals(13, phases.size());		
		assertEquals(13, intersection.getPhase().size());
		List<String> p4Lanes = new ArrayList<String>();
		p4Lanes.add("PH_1");
		p4Lanes.add("PH_12");
		p4Lanes.add("PH_4");
		for(Phase p : phases) {
			if(p.getId().equals("PH_2") || p.getId().equals("PH_7")) {
				assertEquals(2,  p.getLane().size());
			}
			else if(p4Lanes.contains(p.getId())) {
				assertEquals(4, p.getLane().size());
			}
			else {
				assertEquals(3, p.getLane().size());
			}
			assertEquals(phases.size()-1, p.getTransition().size());
		}	
	}
	
	/**
	 * All the possible Phases WITHOUT PEDESTRIAN for a 4 Arms Intersection of the where the roads are of types:
	 * <code>IntersectionConstants.MAIN_STRAIGHT_RIGHT_MERGE<code>
	 * <code>IntersectionConstants.SEC_STRAIGHT_RIGHT_MERGE<code>
	 * <code>IntersectionConstants.MAIN_STRAIGHT_RIGHT_MERGE<code>
	 * <code>IntersectionConstants.SEC_STRAIGHT_RIGHT_MERGE<code>
	 */
	@Test 
	public void test4ArmsAllRightMergeNoPed() {
		setupServices();
		options.put(0, IntersectionConstants.MAIN_STRAIGHT_RIGHT_MERGE);
		options.put(1, IntersectionConstants.SEC_STRAIGHT_RIGHT_MERGE);
		options.put(2, IntersectionConstants.MAIN_STRAIGHT_RIGHT_MERGE);
		options.put(3, IntersectionConstants.SEC_STRAIGHT_RIGHT_MERGE);
		Intersection intersection = service.createIntersection(options);
	
		List<Phase> phases = sigService.createPhases(intersection, SignalPlanConstants.PHASES_WITHOUT_PEDESTRIAN);
		phases = phases.stream().sorted(Comparator.comparing(Phase::getId)).collect(Collectors.toList());
		assertFalse(phases.isEmpty());
		assertEquals(8, phases.size());		
		assertEquals(8, intersection.getPhase().size());
		for(Phase p : phases) {
			assertEquals(2,  p.getLane().size());
			assertEquals(phases.size()-1, p.getTransition().size());
		}	
	}
	
	/**
	 * All the possible Phases WITHOUT LANES WITH ONLY PEDESTRIAN for a 4 Arms Intersection of the where 
	 * the roads are of types:
	 * <code>IntersectionConstants.MAIN_STRAIGHT_RIGHT_MERGE<code>
	 * <code>IntersectionConstants.SEC_STRAIGHT_RIGHT_MERGE<code>
	 * <code>IntersectionConstants.MAIN_STRAIGHT_RIGHT_MERGE<code>
	 * <code>IntersectionConstants.SEC_STRAIGHT_RIGHT_MERGE<code>
	 */
	@Test 
	public void test4ArmsAllRightMergeNotOnlyPed() {
		setupServices();
		options.put(0, IntersectionConstants.MAIN_STRAIGHT_RIGHT_MERGE);
		options.put(1, IntersectionConstants.SEC_STRAIGHT_RIGHT_MERGE);
		options.put(2, IntersectionConstants.MAIN_STRAIGHT_RIGHT_MERGE);
		options.put(3, IntersectionConstants.SEC_STRAIGHT_RIGHT_MERGE);
		Intersection intersection = service.createIntersection(options);
	
		List<Phase> phases = sigService.createPhases(intersection, SignalPlanConstants.PHASES_NOT_ONLY_PEDESTRIAN);
		phases = phases.stream().sorted(Comparator.comparing(Phase::getId)).collect(Collectors.toList());
		assertFalse(phases.isEmpty());
		assertEquals(12, phases.size());		
		assertEquals(12, intersection.getPhase().size());
		List<String> p4Lanes = new ArrayList<String>();
		p4Lanes.add("PH_1");
		p4Lanes.add("PH_4");
		for(Phase p : phases) {
			if(p.getId().equals("PH_2") || p.getId().equals("PH_7")) {
				assertEquals(2,  p.getLane().size());
			}
			else if(p4Lanes.contains(p.getId())) {
				assertEquals(4, p.getLane().size());
			}
			else {
				assertEquals(3, p.getLane().size());
			}
			assertEquals(phases.size()-1, p.getTransition().size());
		}	
	}
	
	/**
	 * All the possible Phases for a 4 Arms Intersection of the where the roads are of types:
	 * <code>IntersectionConstants.MAIN_STRAIGHT_RIGHT_MERGE<code>
	 * <code>IntersectionConstants.SEC_STRAIGHT_TURNS_SEP<code>
	 * <code>IntersectionConstants.MAIN_STRAIGHT_RIGHT_MERGE<code>
	 * <code>IntersectionConstants.SEC_STRAIGHT_TURNS_SEP<code>
	 */
	@Test 
	public void test4ArmsMainRightMergeAllPhases() {
		setupServices();
		options.put(0, IntersectionConstants.MAIN_STRAIGHT_RIGHT_MERGE);
		options.put(1, IntersectionConstants.SEC_STRAIGHT_TURNS_SEP);
		options.put(2, IntersectionConstants.MAIN_STRAIGHT_RIGHT_MERGE);
		options.put(3, IntersectionConstants.SEC_STRAIGHT_TURNS_SEP);
		Intersection intersection = service.createIntersection(options);
	
		List<Phase> phases = sigService.createPhases(intersection, SignalPlanConstants.ALL_PHASES);
		phases = phases.stream().sorted(Comparator.comparing(Phase::getId)).collect(Collectors.toList());
		assertFalse(phases.isEmpty());
		assertEquals(20, phases.size());		
		assertEquals(20, intersection.getPhase().size());
		List<String> p3Lanes = new ArrayList<String>();
		p3Lanes.add("PH_11");
		p3Lanes.add("PH_15");
		p3Lanes.add("PH_16");
		p3Lanes.add("PH_17");
		p3Lanes.add("PH_18");
		p3Lanes.add("PH_4");
		p3Lanes.add("PH_8");
		p3Lanes.add("PH_5");
		for(Phase p : phases) {
			if(p.getId().equals("PH_12")) {
				assertEquals(2,  p.getLane().size());
			}
			else if(p3Lanes.contains(p.getId())) {
				assertEquals(3, p.getLane().size());
			}
			else {
				assertEquals(4, p.getLane().size());
			}
			assertEquals(phases.size()-1, p.getTransition().size());
		}	
	}
	
	/**
	 * All the possible Phases WITHOUT PEDESTRIAN for a 4 Arms Intersection of the where the roads are of types:
	 * <code>IntersectionConstants.MAIN_STRAIGHT_RIGHT_MERGE<code>
	 * <code>IntersectionConstants.SEC_STRAIGHT_TURNS_SEP<code>
	 * <code>IntersectionConstants.MAIN_STRAIGHT_RIGHT_MERGE<code>
	 * <code>IntersectionConstants.SEC_STRAIGHT_TURNS_SEP<code>
	 */
	@Test 
	public void test4ArmsMainRightMergeNoPed() {
		setupServices();
		options.put(0, IntersectionConstants.MAIN_STRAIGHT_RIGHT_MERGE);
		options.put(1, IntersectionConstants.SEC_STRAIGHT_TURNS_SEP);
		options.put(2, IntersectionConstants.MAIN_STRAIGHT_RIGHT_MERGE);
		options.put(3, IntersectionConstants.SEC_STRAIGHT_TURNS_SEP);
		Intersection intersection = service.createIntersection(options);
	
		List<Phase> phases = sigService.createPhases(intersection, SignalPlanConstants.PHASES_WITHOUT_PEDESTRIAN);
		phases = phases.stream().sorted(Comparator.comparing(Phase::getId)).collect(Collectors.toList());
		assertFalse(phases.isEmpty());
		assertEquals(8, phases.size());		
		assertEquals(8, intersection.getPhase().size());
		List<String> p3Lanes = new ArrayList<String>();
		p3Lanes.add("PH_0");
		p3Lanes.add("PH_3");
		p3Lanes.add("PH_6");
		p3Lanes.add("PH_7");
		for(Phase p : phases) {
			if(p.getId().equals("PH_1") || p.getId().equals("PH_5")) {
				assertEquals(2,  p.getLane().size());
			}
			else if(p3Lanes.contains(p.getId())) {
				assertEquals(3, p.getLane().size());
			}
			else {
				assertEquals(4, p.getLane().size());
			}
			assertEquals(phases.size()-1, p.getTransition().size());
		}	
	}
	
	/**
	 * All the possible Phases WITHOUT LANES WITH ONLY PEDESTRIAN for a 4 Arms Intersection of the where 
	 * the roads are of types:
	 * <code>IntersectionConstants.MAIN_STRAIGHT_RIGHT_MERGE<code>
	 * <code>IntersectionConstants.SEC_STRAIGHT_TURNS_SEP<code>
	 * <code>IntersectionConstants.MAIN_STRAIGHT_RIGHT_MERGE<code>
	 * <code>IntersectionConstants.SEC_STRAIGHT_TURNS_SEP<code>
	 */
	@Test 
	public void test4ArmsMainRightMergeNotOnlyPed() {
		setupServices();
		options.put(0, IntersectionConstants.MAIN_STRAIGHT_RIGHT_MERGE);
		options.put(1, IntersectionConstants.SEC_STRAIGHT_TURNS_SEP);
		options.put(2, IntersectionConstants.MAIN_STRAIGHT_RIGHT_MERGE);
		options.put(3, IntersectionConstants.SEC_STRAIGHT_TURNS_SEP);
		Intersection intersection = service.createIntersection(options);
	
		List<Phase> phases = sigService.createPhases(intersection, SignalPlanConstants.PHASES_NOT_ONLY_PEDESTRIAN);
		phases = phases.stream().sorted(Comparator.comparing(Phase::getId)).collect(Collectors.toList());
		assertFalse(phases.isEmpty());
		assertEquals(19, phases.size());		
		assertEquals(19, intersection.getPhase().size());
		List<String> p3Lanes = new ArrayList<String>();
		p3Lanes.add("PH_11");
		p3Lanes.add("PH_15");
		p3Lanes.add("PH_16");
		p3Lanes.add("PH_17");
		p3Lanes.add("PH_18");
		p3Lanes.add("PH_4");
		p3Lanes.add("PH_8");
		p3Lanes.add("PH_5");
		for(Phase p : phases) {
			if(p.getId().equals("PH_12")) {
				assertEquals(2,  p.getLane().size());
			}
			else if(p3Lanes.contains(p.getId())) {
				assertEquals(3, p.getLane().size());
			}
			else {
				assertEquals(4, p.getLane().size());
			}
			assertEquals(phases.size()-1, p.getTransition().size());
		}	
	}
	
	/**
	 * All the possible Phases for a 4 Arms Intersection of the where the roads are of types:
	 * <code>IntersectionConstants.MAIN_STRAIGHT_RIGHT_MERGE<code>
	 * <code>IntersectionConstants.SEC_STRAIGHT_LEFT_MERGE<code>
	 * <code>IntersectionConstants.MAIN_STRAIGHT_RIGHT_MERGE<code>
	 * <code>IntersectionConstants.SEC_STRAIGHT_LEFT_MERGE<code>
	 */
	@Test 
	public void test4ArmsSecLeftMergeMainRightMergeAllPhases() {
		setupServices();
		options.put(0, IntersectionConstants.MAIN_STRAIGHT_RIGHT_MERGE);
		options.put(1, IntersectionConstants.SEC_STRAIGHT_LEFT_MERGE);
		options.put(2, IntersectionConstants.MAIN_STRAIGHT_RIGHT_MERGE);
		options.put(3, IntersectionConstants.SEC_STRAIGHT_LEFT_MERGE);
		Intersection intersection = service.createIntersection(options);
	
		List<Phase> phases = sigService.createPhases(intersection, SignalPlanConstants.ALL_PHASES);
		phases = phases.stream().sorted(Comparator.comparing(Phase::getId)).collect(Collectors.toList());
		assertFalse(phases.isEmpty());
		assertEquals(10, phases.size());		
		assertEquals(10, intersection.getPhase().size());
		List<String> p3Lanes = new ArrayList<String>();
		p3Lanes.add("PH_4");
		p3Lanes.add("PH_8");		
		for(Phase p : phases) {
			if(p3Lanes.contains(p.getId())) {
				assertEquals(3, p.getLane().size());
			}
			else {
				assertEquals(4, p.getLane().size());
			}
			assertEquals(phases.size()-1, p.getTransition().size());
		}	
	}
	
	/**
	 * All the possible Phases WITHOUT PEDESTRIAN for a 4 Arms Intersection of the where the roads are of types:
	 * <code>IntersectionConstants.MAIN_STRAIGHT_RIGHT_MERGE<code>
	 * <code>IntersectionConstants.SEC_STRAIGHT_LEFT_MERGE<code>
	 * <code>IntersectionConstants.MAIN_STRAIGHT_RIGHT_MERGE<code>
	 * <code>IntersectionConstants.SEC_STRAIGHT_LEFT_MERGE<code>
	 */
	@Test 
	public void test4ArmsSecLeftMergeMainRightMergeNoPed() {
		setupServices();
		options.put(0, IntersectionConstants.MAIN_STRAIGHT_RIGHT_MERGE);
		options.put(1, IntersectionConstants.SEC_STRAIGHT_LEFT_MERGE);
		options.put(2, IntersectionConstants.MAIN_STRAIGHT_RIGHT_MERGE);
		options.put(3, IntersectionConstants.SEC_STRAIGHT_LEFT_MERGE);
		Intersection intersection = service.createIntersection(options);
	
		List<Phase> phases = sigService.createPhases(intersection, SignalPlanConstants.PHASES_WITHOUT_PEDESTRIAN);
		phases = phases.stream().sorted(Comparator.comparing(Phase::getId)).collect(Collectors.toList());
		assertFalse(phases.isEmpty());
		assertEquals(7, phases.size());		
		assertEquals(7, intersection.getPhase().size());
		List<String> p3Lanes = new ArrayList<String>();
		p3Lanes.add("PH_0");
		p3Lanes.add("PH_5");		
		for(Phase p : phases) {
			if(p.getId().equals("PH_2")) {
				assertEquals(4, p.getLane().size());
			}
			else if(p3Lanes.contains(p.getId())) {
				assertEquals(3, p.getLane().size());
			}
			else {
				assertEquals(2, p.getLane().size());
			}
			assertEquals(phases.size()-1, p.getTransition().size());
		}	
	}
	
	/**
	 * All the possible Phases WITHOUT LANES WITH ONLY PEDESTRIAN for a 4 Arms Intersection of the where 
	 * the roads are of types:
	 * <code>IntersectionConstants.MAIN_STRAIGHT_RIGHT_MERGE<code>
	 * <code>IntersectionConstants.SEC_STRAIGHT_LEFT_MERGE<code>
	 * <code>IntersectionConstants.MAIN_STRAIGHT_RIGHT_MERGE<code>
	 * <code>IntersectionConstants.SEC_STRAIGHT_LEFT_MERGE<code>
	 */
	@Test 
	public void test4ArmsSecLeftMergeMainRightMergeNotOnlyPed() {
		setupServices();
		options.put(0, IntersectionConstants.MAIN_STRAIGHT_RIGHT_MERGE);
		options.put(1, IntersectionConstants.SEC_STRAIGHT_LEFT_MERGE);
		options.put(2, IntersectionConstants.MAIN_STRAIGHT_RIGHT_MERGE);
		options.put(3, IntersectionConstants.SEC_STRAIGHT_LEFT_MERGE);
		Intersection intersection = service.createIntersection(options);
	
		List<Phase> phases = sigService.createPhases(intersection, SignalPlanConstants.PHASES_NOT_ONLY_PEDESTRIAN);
		phases = phases.stream().sorted(Comparator.comparing(Phase::getId)).collect(Collectors.toList());
		assertFalse(phases.isEmpty());
		assertEquals(9, phases.size());		
		assertEquals(9, intersection.getPhase().size());
		List<String> p3Lanes = new ArrayList<String>();
		p3Lanes.add("PH_4");
		p3Lanes.add("PH_8");		
		for(Phase p : phases) {
			if(p3Lanes.contains(p.getId())) {
				assertEquals(3, p.getLane().size());
			}
			else {
				assertEquals(4, p.getLane().size());
			}
			assertEquals(phases.size()-1, p.getTransition().size());
		}	
	}	
}
