/**
 * Copyright (c) 2012 - 2018 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package de.dim.trafficos.simulator.tests;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.stream.Collectors;

import org.gecko.core.tests.AbstractOSGiTest;
import org.gecko.core.tests.ServiceChecker;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.runners.MockitoJUnitRunner;
import org.osgi.framework.FrameworkUtil;

import de.dim.trafficos.model.device.CacheDataEntry;
import de.dim.trafficos.model.device.ConflictingLane;
import de.dim.trafficos.model.device.DataValue;
import de.dim.trafficos.model.device.Intersection;
import de.dim.trafficos.model.device.Output;
import de.dim.trafficos.model.device.Phase;
import de.dim.trafficos.model.device.PhaseGroup;
import de.dim.trafficos.model.device.Program;
import de.dim.trafficos.model.device.ScheduleModeType;
import de.dim.trafficos.simulator.api.IntersectionConstants;
import de.dim.trafficos.simulator.api.IntersectionService;
import de.dim.trafficos.simulator.api.SignalPlanConstants;
import de.dim.trafficos.simulator.api.SignalPlanService;


/**
 * Tests the implementation of the SignalPlanService when creating a Program for a 3 arms intersection
 * @since 1.0
 */
@RunWith(MockitoJUnitRunner.class)
public class SigPlanProgram3ArmsTest extends AbstractOSGiTest {
	
	private IntersectionService service;
	private SignalPlanService sigService;
	private Map<Integer, String> options;
	private File file = new File("data/TestFixProgram3Arms.txt");
	
	/**
	 * Creates a new instance.
	 * @param bundleContext
	 */
	public SigPlanProgram3ArmsTest() {
		super(FrameworkUtil.getBundle(SigPlanProgram3ArmsTest.class).getBundleContext());
	}

	/**
	 * Here you can put everything you want to be executed before every test
	 */
	public void doBefore() {
		if(file.exists()) {
			file.delete();
		}
		options = new HashMap<Integer, String>();
	}
	
	/**
	 * Here you can put everything you want to be executed after every test
	 */
	public void doAfter() {
		
	}
	
	private void setupServices() {
		ServiceChecker<IntersectionService> checker = createCheckerTrackedForCleanUp(IntersectionService.class);
		checker.start();
		assertNotNull(checker);
		assertEquals(1, checker.getCurrentCreateCount(true));
		
		service = getService(IntersectionService.class);
		assertNotNull(service);
		
		ServiceChecker<SignalPlanService> sigChecker = createCheckerTrackedForCleanUp(SignalPlanService.class);
		sigChecker.start();
		assertNotNull(sigChecker);
		assertEquals(1, sigChecker.getCurrentCreateCount(true));
		
		sigService = getService(SignalPlanService.class);
		assertNotNull(sigService);	
	}
	
	/**
	 * This test would create a table in a file with the status of the SignalGroup of each IncomingLane
	 * for each value of TX. 
	 * 
	 * @throws IOException
	 */
	@Test
	public void test3ArmsDIMFixTSigProg() throws IOException {
		setupServices();
		options.put(0, IntersectionConstants.MAIN_STRAIGHT_RIGHT_MERGE);
		options.put(1, IntersectionConstants.MAIN_STRAIGHT_LEFT_SEP);
		options.put(2, IntersectionConstants.SEC_LEFT_RIGHT_MERGE);
		Intersection intersection = service.createIntersection(options);
		
		assertNotNull(intersection);
		assertFalse(intersection.getRoad().isEmpty());
		assertEquals(options.size(), intersection.getRoad().size());		
		
		List<Phase> phases = sigService.createPhases(intersection, SignalPlanConstants.ALL_PHASES);
		assertFalse(phases.isEmpty());
		
		sigService.createPhaseGroups(intersection);
		assertFalse(intersection.getPhaseGroup().isEmpty());
		List<PhaseGroup> phGroups = intersection.getPhaseGroup().stream().sorted(Comparator.comparing(PhaseGroup::getPenalty))
				.collect(Collectors.toList());
		PhaseGroup pg = phGroups.get(0);
		List<Phase> groupPh = pg.getPhase();
		for(Phase p : groupPh) {			
			if(p.getId().equals("PH_0")) {
				p.setWeightMax(70);
				p.setWeightMin(60);
			}
			else if(p.getId().equals("PH_2")) {
				p.setWeightMax(50);
				p.setWeightMin(30);
			}
			else if(p.getId().equals("PH_3")) {
				p.setWeightMax(20);
				p.setWeightMin(10);
			}
		}
		
		Program program = sigService.createFixTimeProgram(pg, UUID.randomUUID().toString(), 100);

		Map<Integer, Map<ConflictingLane, String>> programMap = sigService.applyProgram(intersection, program, ScheduleModeType.WORKING_DAY);
		assertNotNull(programMap);
		assertFalse(programMap.isEmpty());
		file.createNewFile();
		BufferedWriter writer = new BufferedWriter(new FileWriter(file.getAbsolutePath(), true));
		for(int m = 0; m < programMap.size(); m++) {
			if(m == 0) {
				Iterator<ConflictingLane> iter = programMap.get(m).keySet().iterator();
				while(iter.hasNext()) {
					String id = iter.next().getId();
					if(id.startsWith("P")) {
						writer.append("\t" + id);
					}
					else {
						writer.append("\t\t" + id);
					}					
				}
				writer.append("\n");
			}
			writer.append(String.valueOf(m));
			Map<ConflictingLane, String> values = programMap.get(m);
			values.entrySet().forEach(e->{
				try {
					writer.append("\t\t" + e.getValue());
				} catch (IOException e1) {
					e1.printStackTrace();
				}
			});
			writer.append("\n");	
		}	     
	    writer.close();
	}	
	
	@Test
	public void testDataValueDuration() throws IOException {
		setupServices();
		options.put(0, IntersectionConstants.MAIN_STRAIGHT_RIGHT_MERGE);
		options.put(1, IntersectionConstants.MAIN_STRAIGHT_LEFT_SEP);
		options.put(2, IntersectionConstants.SEC_LEFT_RIGHT_MERGE);
		Intersection intersection = service.createIntersection(options);
		
		assertNotNull(intersection);
		assertFalse(intersection.getRoad().isEmpty());
		assertEquals(options.size(), intersection.getRoad().size());		
		
		List<Phase> phases = sigService.createPhases(intersection, SignalPlanConstants.ALL_PHASES);
		assertFalse(phases.isEmpty());
		
		sigService.createPhaseGroups(intersection);
		assertFalse(intersection.getPhaseGroup().isEmpty());
		List<PhaseGroup> phGroups = intersection.getPhaseGroup().stream().sorted(Comparator.comparing(PhaseGroup::getPenalty))
				.collect(Collectors.toList());
		PhaseGroup pg = phGroups.get(0);
		List<Phase> groupPh = pg.getPhase();
		for(Phase p : groupPh) {			
			if(p.getId().equals("PH_0")) {
				p.setWeightMax(70);
				p.setWeightMin(60);
			}
			else if(p.getId().equals("PH_2")) {
				p.setWeightMax(50);
				p.setWeightMin(30);
			}
			else if(p.getId().equals("PH_3")) {
				p.setWeightMax(20);
				p.setWeightMin(10);
			}
		}
		
		Program program = sigService.createFixTimeProgram(pg, UUID.randomUUID().toString(), 100);

		Map<Integer, Map<ConflictingLane, String>> programMap = sigService.applyProgram(intersection, program, ScheduleModeType.WORKING_DAY);
		assertNotNull(programMap);
		assertFalse(programMap.isEmpty());
		file.createNewFile();
		BufferedWriter writer = new BufferedWriter(new FileWriter(file.getAbsolutePath(), true));
		for(int m = 0; m < programMap.size(); m++) {
			if(m == 0) {
				Iterator<ConflictingLane> iter = programMap.get(m).keySet().iterator();
				while(iter.hasNext()) {
					String id = iter.next().getId();
					if(id.startsWith("P")) {
						writer.append("\t" + id);
					}
					else {
						writer.append("\t\t" + id);
					}					
				}
				writer.append("\n");
			}
			writer.append(String.valueOf(m));
			Map<ConflictingLane, String> values = programMap.get(m);
			values.entrySet().forEach(e->{
				try {
					writer.append("\t\t" + e.getValue());
				} catch (IOException e1) {
					e1.printStackTrace();
				}
			});
			writer.append("\n");	
		}	     
	    writer.close();
		sigService.addCacheDataValueDurations(program);
		
		for(CacheDataEntry de : program.getSignalTable().getCacheDataEntry()) {
			for(DataValue dv : de.getValue()) {
				if(dv.getElement() instanceof Output) {
					assertTrue(dv.getDuration() != 0);
					if(de.getId().equals("99")) {
						if(dv.getElement().getId().equals("SG-0_1") || 
								dv.getElement().getId().equals("SG-1_0")) {
							assertEquals(1, dv.getDuration());
						}
					}
				}				
			}
		}
	}
}
