/**
 * Copyright (c) 2012 - 2018 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package de.dim.trafficos.simulator.tests;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import java.util.Comparator;
import java.util.Dictionary;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;

import org.gecko.core.tests.AbstractOSGiTest;
import org.gecko.core.tests.ServiceChecker;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.runners.MockitoJUnitRunner;
import org.osgi.framework.FrameworkUtil;
import org.osgi.service.event.Event;
import org.osgi.service.event.EventAdmin;
import org.osgi.service.event.EventConstants;
import org.osgi.service.event.EventHandler;

import de.dim.trafficos.device.api.DeviceService;
import de.dim.trafficos.model.device.ConflictingLane;
import de.dim.trafficos.model.device.DataEntry;
import de.dim.trafficos.model.device.DataValue;
import de.dim.trafficos.model.device.Intersection;
import de.dim.trafficos.model.device.Output;
import de.dim.trafficos.model.device.Parameter;
import de.dim.trafficos.model.device.Phase;
import de.dim.trafficos.model.device.PhaseGroup;
import de.dim.trafficos.model.device.Program;
import de.dim.trafficos.model.device.ScheduleModeType;
import de.dim.trafficos.simulator.api.DeviceSimulator;
import de.dim.trafficos.simulator.api.DeviceStatusCallback;
import de.dim.trafficos.simulator.api.IntersectionConstants;
import de.dim.trafficos.simulator.api.IntersectionService;
import de.dim.trafficos.simulator.api.SignalPlanConstants;
import de.dim.trafficos.simulator.api.SignalPlanService;


/**
 *Tests the implementation of the SignalPlanService
 * @since 1.0
 */
@RunWith(MockitoJUnitRunner.class)
public class SimulatorIntegrationTest extends AbstractOSGiTest {
	
	private EventAdmin eventAdmin;
	private IntersectionService interService;
	private SignalPlanService sigService;
	private Map<Integer, String> options;
	private Intersection intersection;
	
	/**
	 * Creates a new instance.
	 * @param bundleContext
	 */
	public SimulatorIntegrationTest() {
		super(FrameworkUtil.getBundle(SimulatorIntegrationTest.class).getBundleContext());
	}

	/**
	 * Here you can put everything you want to be executed before every test
	 */
	public void doBefore() {
		
		options = new HashMap<Integer, String>();
	}
	
	/**
	 * Here you can put everything you want to be executed after every test
	 */
	public void doAfter() {

	}
	
	private void setupSimulationServices() {
		
		ServiceChecker<EventAdmin> adminChecker = createCheckerTrackedForCleanUp(EventAdmin.class);
		adminChecker.start();
		assertEquals(1,  adminChecker.getCurrentCreateCount(true));
		eventAdmin = getService(EventAdmin.class);
		assertNotNull(eventAdmin);
		
		ServiceChecker<IntersectionService> checker = createCheckerTrackedForCleanUp(IntersectionService.class);
		checker.start();
		assertNotNull(checker);
		assertEquals(1, checker.getCurrentCreateCount(true));
		
		interService = getService(IntersectionService.class);
		assertNotNull(interService);
		
		ServiceChecker<SignalPlanService> sigChecker = createCheckerTrackedForCleanUp(SignalPlanService.class);
		sigChecker.start();
		assertNotNull(sigChecker);
		assertEquals(1, sigChecker.getCurrentCreateCount(true));
		
		sigService = getService(SignalPlanService.class);
		assertNotNull(sigService);	
		
	
	}
	
	/**
	 * Test the implementation of the DeviceSimulator which, given an Intersection launches the corresponding 
	 * Signal Program. 
	 * @throws InterruptedException 
	 */
	@Test
	public void testSimulation() throws InterruptedException {
		setupSimulationServices();
		
		final AtomicInteger ai = new AtomicInteger(-1);
		CountDownLatch cdl = new CountDownLatch(1);
		EventHandler handler = new EventHandler() {
			
			@Override
			public void handleEvent(Event event) {
				ai.incrementAndGet();
				if (ai.get() == 20) {
					cdl.countDown();
				}
				DataEntry entry = (DataEntry) event.getProperty(DeviceService.PROP_DATA_ENTRY);
				Integer cycleCounter = (Integer) event.getProperty(DeviceService.PROP_CYCLE_COUNTER);
				if (entry == null) {
					fail();
				} else {
					System.out.println("Simulation step " + ai.get());
					assertEquals(ai.get(), cycleCounter.intValue());
					assertEquals(ai.get() + 1, entry.getIndex());
					assertEquals(8, entry.getValue().size());
					List<DataValue> values = entry.getValue();
					List<DataValue> txValue = values.stream().filter(v->(v.getElement() instanceof Parameter))
							.collect(Collectors.toList());
					assertFalse(txValue.isEmpty());
					assertEquals(1, txValue.size());
					DataValue tx = txValue.get(0);
					assertEquals(String.valueOf(ai.get()), tx.getValue());
					List<DataValue> sigGroups = values.stream().filter(v->(v.getElement() instanceof Output))
							.collect(Collectors.toList());
					assertFalse(sigGroups.isEmpty());
					assertEquals(7, sigGroups.size());
				}
			}			
		};
		Dictionary<String, Object> properties = new Hashtable<String, Object>();
		properties.put(EventConstants.EVENT_TOPIC, "dataEntry/*");
		registerServiceForCleanup(EventHandler.class, handler, properties);
		
		ServiceChecker<EventHandler> trackedHandler = createCheckerTrackedForCleanUp(EventHandler.class);
		trackedHandler.start();
		assertEquals(1,  trackedHandler.getCurrentCreateCount(true));
		
		options.put(0, IntersectionConstants.MAIN_STRAIGHT_RIGHT_MERGE);
		options.put(1, IntersectionConstants.MAIN_STRAIGHT_LEFT_SEP);
		options.put(2, IntersectionConstants.SEC_LEFT_RIGHT_MERGE);
		intersection = interService.createIntersection(options);
		
		assertNotNull(intersection);
		
		List<Phase> phases = sigService.createPhases(intersection, SignalPlanConstants.ALL_PHASES);
		assertFalse(phases.isEmpty());
		
		sigService.createPhaseGroups(intersection);
		assertFalse(intersection.getPhaseGroup().isEmpty());
		List<PhaseGroup> phGroups = intersection.getPhaseGroup().stream().sorted(Comparator.comparing(PhaseGroup::getPenalty))
				.collect(Collectors.toList());
		PhaseGroup pg = phGroups.get(0);
		List<Phase> groupPh = pg.getPhase();
		for(Phase p : groupPh) {
			if(p.getId().equals("PH_0")) {
				p.setWeightMax(70);
				p.setWeightMin(60);
			}
			else if(p.getId().equals("PH_2")) {
				p.setWeightMax(50);
				p.setWeightMin(30);
			}
			else if(p.getId().equals("PH_3")) {
				p.setWeightMax(20);
				p.setWeightMin(10);
			}
		}
		
		Program program = sigService.createFixTimeProgram(pg, UUID.randomUUID().toString(), 100);
		assertNotNull(program);
		
		Map<Integer, Map<ConflictingLane, String>> programMap = sigService.applyProgram(intersection, program, ScheduleModeType.WORKING_DAY);
		assertNotNull(programMap);
		assertFalse(programMap.isEmpty());
		
		DeviceSimulator simulator = new DeviceSimulator(new TestCallback());
		simulator.initializeSimulation(intersection);
		simulator.setNotifyConsumer(this::handleDataEntry);
		simulator.startSimulation();
		
		assertTrue(cdl.await(30, TimeUnit.SECONDS));
	}
	
	private void handleDataEntry(DataEntry entry, Integer tx) {
		String topic = String.format(DeviceService.TOPIC_DATA_ENTRY, intersection.getId());
		Map<String, Object> properties = new HashMap<String, Object>();
		properties.put(DeviceService.PROP_DATA_ENTRY, entry);
		properties.put(DeviceService.PROP_CYCLE_COUNTER, tx);
		Event event = new Event(topic, properties);
		eventAdmin.postEvent(event);
	}
	
	
	public class TestCallback implements DeviceStatusCallback {

		/* 
		 * (non-Javadoc)
		 * @see de.dim.trafficos.simulator.api.DeviceStatusCallback#statusChanged(boolean)
		 */
		@Override
		public void statusChanged(boolean running) {
			if(!running) {
				System.out.println("Simulation has been stopped");
			}
			
		}	
	}
	
}
