/**
 * Copyright (c) 2012 - 2018 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package de.dim.trafficos.simulator.tests;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;

import java.util.Comparator;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.gecko.core.tests.AbstractOSGiTest;
import org.gecko.core.tests.ServiceChecker;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.runners.MockitoJUnitRunner;
import org.osgi.framework.FrameworkUtil;

import de.dim.trafficos.model.device.ConflictingLane;
import de.dim.trafficos.model.device.IncomingLane;
import de.dim.trafficos.model.device.Intersection;
import de.dim.trafficos.model.device.Lane;
import de.dim.trafficos.model.device.Link;
import de.dim.trafficos.model.device.PedestrianLane;
import de.dim.trafficos.model.device.Road;
import de.dim.trafficos.simulator.api.IntersectionConstants;
import de.dim.trafficos.simulator.api.IntersectionService;


/**
 *Tests the implementation of the IntersectionService
 * @since 1.0
 */
@RunWith(MockitoJUnitRunner.class)
public class IntersectionServiceIntegrationTest extends AbstractOSGiTest {
	
	/**
	 * Creates a new instance.
	 * @param bundleContext
	 */
	public IntersectionServiceIntegrationTest() {
		super(FrameworkUtil.getBundle(IntersectionServiceIntegrationTest.class).getBundleContext());
	}

	/**
	 * Here you can put everything you want to be executed before every test
	 */
	public void doBefore() {
		
	}
	
	/**
	 * Here you can put everything you want to be executed after every test
	 */
	public void doAfter() {
		
	}
	
	/**
	 * Tests that the Service is created
	 */
	@Test
	public void testServiceCreation() {
		
		ServiceChecker<IntersectionService> checker = createCheckerTrackedForCleanUp(IntersectionService.class);
		checker.start();
		assertNotNull(checker);
		assertEquals(1, checker.getCurrentCreateCount(true));
		
		IntersectionService service = getService(IntersectionService.class);
		assertNotNull(service);
	}
	
	@Test
	public void test2RoadOptions() {
		
		ServiceChecker<IntersectionService> checker = createCheckerTrackedForCleanUp(IntersectionService.class);
		checker.start();
		assertNotNull(checker);
		assertEquals(1, checker.getCurrentCreateCount(true));
		
		IntersectionService service = getService(IntersectionService.class);
		assertNotNull(service);
		
		Map<Integer, String> options = new HashMap<Integer, String>();
		options.put(0, IntersectionConstants.MAIN_STRAIGHT);
		options.put(1, IntersectionConstants.MAIN_STRAIGHT);
		Intersection intersection = service.createIntersection(options);
		
		assertNotNull(intersection);
		
		options = new HashMap<Integer, String>();
		options.put(0, IntersectionConstants.MAIN_STRAIGHT_LEFT_MERGE);
		options.put(1, IntersectionConstants.MAIN_STRAIGHT);
		intersection = service.createIntersection(options);
		
		assertNull(intersection);
		
		options = new HashMap<Integer, String>();
		options.put(0, IntersectionConstants.MAIN_STRAIGHT);
		options.put(1, IntersectionConstants.MAIN_STRAIGHT_RIGHT_SEP);
		intersection = service.createIntersection(options);
		
		assertNull(intersection);
	}
	
	@Test
	public void test3RoadOptions() {
		
		ServiceChecker<IntersectionService> checker = createCheckerTrackedForCleanUp(IntersectionService.class);
		checker.start();
		assertNotNull(checker);
		assertEquals(1, checker.getCurrentCreateCount(true));
		
		IntersectionService service = getService(IntersectionService.class);
		assertNotNull(service);
		
		Map<Integer, String> options = new HashMap<Integer, String>();
		options.put(0, IntersectionConstants.MAIN_STRAIGHT_LEFT_MERGE);
		options.put(1, IntersectionConstants.MAIN_LEFT_RIGHT_SEP);
		options.put(2, IntersectionConstants.MAIN_STRAIGHT_RIGHT_MERGE);
		Intersection intersection = service.createIntersection(options);
		
		assertNotNull(intersection);
		
		options = new HashMap<Integer, String>();
		options.put(0, IntersectionConstants.MAIN_LEFT_RIGHT_SEP);
		options.put(1, IntersectionConstants.MAIN_STRAIGHT_LEFT_MERGE);
		options.put(2, IntersectionConstants.MAIN_STRAIGHT_RIGHT_MERGE);
		intersection = service.createIntersection(options);
		
		assertNotNull(intersection);
		
	}
	
	
	@Test 
	public void test3ArmsRoadsCreation() {
		ServiceChecker<IntersectionService> checker = createCheckerTrackedForCleanUp(IntersectionService.class);
		checker.start();
		assertNotNull(checker);
		assertEquals(1, checker.getCurrentCreateCount(true));
		
		IntersectionService service = getService(IntersectionService.class);
		assertNotNull(service);
		
		Map<Integer, String> options = new HashMap<Integer, String>();
		options.put(0, IntersectionConstants.MAIN_STRAIGHT_RIGHT_MERGE);
		options.put(1, IntersectionConstants.MAIN_STRAIGHT_LEFT_MERGE);
		options.put(2, IntersectionConstants.SEC_LEFT_RIGHT_MERGE);
		Intersection intersection = service.createIntersection(options);
		
		assertNotNull(intersection);
		assertFalse(intersection.getRoad().isEmpty());
		assertEquals(options.size(), intersection.getRoad().size());
		for(int i = 0; i < intersection.getRoad().size(); i++) {
			Road road = intersection.getRoad().get(i);
			assertEquals(String.valueOf(i), road.getId());
			assertNotNull(road.getOutgoingLane());
			assertEquals(1, road.getOutgoingLane().size());
			assertEquals("-"+road.getId()+"_0", road.getOutgoingLane().get(0).getId());
			assertEquals(road.getId(), road.getOutgoingLane().get(0).getRefRoadId());
			assertEquals(road, road.getOutgoingLane().get(0).getRoad());
			assertNotNull(road.getPedestrianLane());
			assertEquals(1, road.getPedestrianLane().size());
			assertEquals("P_"+road.getId()+"_0", road.getPedestrianLane().get(0).getId());
			assertEquals(road.getId(), road.getPedestrianLane().get(0).getRefRoadId());
			assertEquals(road, road.getPedestrianLane().get(0).getRoad());			
			assertFalse(road.getIncomingLane().isEmpty());
			assertEquals(1, road.getIncomingLane().size());
			assertFalse(road.getIncomingLane().get(0).getSubLane().isEmpty());
			assertEquals(1, road.getIncomingLane().get(0).getSubLane().size());
		}
	}
	
	@Test 
	public void test4ArmsRoadsCreation() {
		ServiceChecker<IntersectionService> checker = createCheckerTrackedForCleanUp(IntersectionService.class);
		checker.start();
		assertNotNull(checker);
		assertEquals(1, checker.getCurrentCreateCount(true));
		
		IntersectionService service = getService(IntersectionService.class);
		assertNotNull(service);
		
		Map<Integer, String> options = new HashMap<Integer, String>();
		options.put(0, IntersectionConstants.MAIN_STRAIGHT_TURNS_SEP);
		options.put(1, IntersectionConstants.SEC_STRAIGHT_TURNS_SEP);
		options.put(2, IntersectionConstants.MAIN_STRAIGHT_TURNS_SEP);
		options.put(3, IntersectionConstants.SEC_STRAIGHT_TURNS_SEP);
		Intersection intersection = service.createIntersection(options);
		
		assertNotNull(intersection);
		assertFalse(intersection.getRoad().isEmpty());
		assertEquals(options.size(), intersection.getRoad().size());
		for(int i = 0; i < intersection.getRoad().size(); i++) {
			Road road = intersection.getRoad().get(i);
			assertEquals(String.valueOf(i), road.getId());
			assertNotNull(road.getOutgoingLane());
			assertEquals(1, road.getOutgoingLane().size());
			assertEquals("-"+road.getId()+"_0", road.getOutgoingLane().get(0).getId());
			assertEquals(road.getId(), road.getOutgoingLane().get(0).getRefRoadId());
			assertEquals(road, road.getOutgoingLane().get(0).getRoad());
			assertNotNull(road.getPedestrianLane());
			assertEquals(1, road.getPedestrianLane().size());
			assertEquals("P_"+road.getId()+"_0", road.getPedestrianLane().get(0).getId());
			assertEquals(road.getId(), road.getPedestrianLane().get(0).getRefRoadId());
			assertEquals(road, road.getPedestrianLane().get(0).getRoad());			
			assertFalse(road.getIncomingLane().isEmpty());
			assertEquals(3, road.getIncomingLane().size());
			assertTrue(road.getIncomingLane().get(0).getSubLane().isEmpty());
		}
	}
	
	@Test
	public void test3ArmsLanesCreation() {
		ServiceChecker<IntersectionService> checker = createCheckerTrackedForCleanUp(IntersectionService.class);
		checker.start();
		assertNotNull(checker);
		assertEquals(1, checker.getCurrentCreateCount(true));
		
		IntersectionService service = getService(IntersectionService.class);
		assertNotNull(service);
		
		Map<Integer, String> options = new HashMap<Integer, String>();
		options.put(0, IntersectionConstants.MAIN_STRAIGHT_RIGHT_MERGE);
		options.put(1, IntersectionConstants.MAIN_STRAIGHT_LEFT_MERGE);
		options.put(2, IntersectionConstants.SEC_LEFT_RIGHT_MERGE);
		Intersection intersection = service.createIntersection(options);
		
		assertNotNull(intersection);
		assertFalse(intersection.getRoad().isEmpty());
		assertEquals(3, intersection.getRoad().size());
		List<ConflictingLane> confLanes = new LinkedList<ConflictingLane>();
		for(int i = 0; i < intersection.getRoad().size(); i++) {
			Road road = intersection.getRoad().get(i);
			for(IncomingLane il : road.getIncomingLane()) {
				confLanes.add(il);
				assertEquals(road.getId(), il.getRefRoadId());
				assertEquals(road, il.getRoad());
				if(!il.getSubLane().isEmpty()) {
					for(Lane sl : il.getSubLane()) {
						assertTrue(sl instanceof ConflictingLane);
						assertEquals(sl.getParentLane(), il);
						confLanes.add((ConflictingLane) sl);
					}					
				}				
			}
			for(PedestrianLane pl : road.getPedestrianLane()) {
				confLanes.add(pl);
			}
			
		}
		confLanes = confLanes.stream().sorted(Comparator.comparing(ConflictingLane::getIndex)).collect(Collectors.toList());
		for(int c = 0; c < confLanes.size(); c++) {			
			ConflictingLane cl = confLanes.get(c);
			assertEquals(c, cl.getIndex());
		}
	}
	
	@Test
	public void test4ArmsLanesCreation() {
		ServiceChecker<IntersectionService> checker = createCheckerTrackedForCleanUp(IntersectionService.class);
		checker.start();
		assertNotNull(checker);
		assertEquals(1, checker.getCurrentCreateCount(true));
		
		IntersectionService service = getService(IntersectionService.class);
		assertNotNull(service);
		
		Map<Integer, String> options = new HashMap<Integer, String>();
		options.put(0, IntersectionConstants.MAIN_STRAIGHT_TURNS_SEP);
		options.put(1, IntersectionConstants.SEC_STRAIGHT_TURNS_SEP);
		options.put(2, IntersectionConstants.MAIN_STRAIGHT_TURNS_SEP);
		options.put(3, IntersectionConstants.SEC_STRAIGHT_TURNS_SEP);
		Intersection intersection = service.createIntersection(options);
		
		assertNotNull(intersection);
		assertFalse(intersection.getRoad().isEmpty());
		assertEquals(options.size(), intersection.getRoad().size());
		List<ConflictingLane> confLanes = new LinkedList<ConflictingLane>();
		for(int i = 0; i < intersection.getRoad().size(); i++) {
			Road road = intersection.getRoad().get(i);
			
			for(IncomingLane il : road.getIncomingLane()) {
				confLanes.add(il);
				if(!il.getSubLane().isEmpty()) {
					for(Lane sl : il.getSubLane()) {
						assertTrue(sl instanceof ConflictingLane);
						assertEquals(sl.getParentLane(), il);
						confLanes.add((ConflictingLane) sl);
					}					
				}				
			}
			for(PedestrianLane pl : road.getPedestrianLane()) {
				confLanes.add(pl);
			}
		}
		confLanes = confLanes.stream().sorted(Comparator.comparing(ConflictingLane::getIndex)).collect(Collectors.toList());
		for(int c = 0; c < confLanes.size(); c++) {
			ConflictingLane cl = confLanes.get(c);
			assertEquals(c, cl.getIndex());
		}
	}
	
	@Test
	public void test3ArmsLinksCreation() {
		ServiceChecker<IntersectionService> checker = createCheckerTrackedForCleanUp(IntersectionService.class);
		checker.start();
		assertNotNull(checker);
		assertEquals(1, checker.getCurrentCreateCount(true));
		
		IntersectionService service = getService(IntersectionService.class);
		assertNotNull(service);
		
		Map<Integer, String> options = new HashMap<Integer, String>();
		options.put(0, IntersectionConstants.MAIN_STRAIGHT_RIGHT_MERGE);
		options.put(1, IntersectionConstants.MAIN_STRAIGHT_LEFT_MERGE);
		options.put(2, IntersectionConstants.SEC_LEFT_RIGHT_MERGE);
		Intersection intersection = service.createIntersection(options);
		
		assertNotNull(intersection);
		assertFalse(intersection.getLink().isEmpty());
		assertEquals(9, intersection.getLink().size());
		for(Link l : intersection.getLink()) {
			assertNotNull(l.getRefIncomingLane());
			assertNotNull(l.getRefOutgoingLane());
			assertEquals(intersection.getId()+"_"+l.getRefIncomingLane().getId()+"_"+l.getRefOutgoingLane().getId(), l.getIndex());
		}
	}
	
	@Test
	public void test4ArmsLinksCreation() {
		ServiceChecker<IntersectionService> checker = createCheckerTrackedForCleanUp(IntersectionService.class);
		checker.start();
		assertNotNull(checker);
		assertEquals(1, checker.getCurrentCreateCount(true));
		
		IntersectionService service = getService(IntersectionService.class);
		assertNotNull(service);
		
		Map<Integer, String> options = new HashMap<Integer, String>();
		options.put(0, IntersectionConstants.MAIN_STRAIGHT_TURNS_SEP);
		options.put(1, IntersectionConstants.SEC_STRAIGHT_TURNS_SEP);
		options.put(2, IntersectionConstants.MAIN_STRAIGHT_TURNS_SEP);
		options.put(3, IntersectionConstants.SEC_STRAIGHT_TURNS_SEP);
		Intersection intersection = service.createIntersection(options);
		
		assertNotNull(intersection);
		assertFalse(intersection.getLink().isEmpty());
		assertEquals(16, intersection.getLink().size());
		for(Link l : intersection.getLink()) {
			assertNotNull(l.getRefIncomingLane());
			assertNotNull(l.getRefOutgoingLane());
			assertEquals(intersection.getId()+"_"+l.getRefIncomingLane().getId()+"_"+l.getRefOutgoingLane().getId(), l.getIndex());
		}
	}
	
	@Test
	public void test3ArmsConflictsCreation() {
		ServiceChecker<IntersectionService> checker = createCheckerTrackedForCleanUp(IntersectionService.class);
		checker.start();
		assertNotNull(checker);
		assertEquals(1, checker.getCurrentCreateCount(true));
		
		IntersectionService service = getService(IntersectionService.class);
		assertNotNull(service);
		
		Map<Integer, String> options = new HashMap<Integer, String>();
		options.put(0, IntersectionConstants.MAIN_STRAIGHT_RIGHT_MERGE);
		options.put(1, IntersectionConstants.MAIN_STRAIGHT_LEFT_MERGE);
		options.put(2, IntersectionConstants.SEC_LEFT_RIGHT_MERGE);
		Intersection intersection = service.createIntersection(options);
		
		assertNotNull(intersection);
		for(int i = 0; i < intersection.getRoad().size(); i++) {
			Road road = intersection.getRoad().get(i);
			List<ConflictingLane> confLanes = new LinkedList<ConflictingLane>();
			for(IncomingLane il : road.getIncomingLane()) {
				confLanes.add(il);
				if(!il.getSubLane().isEmpty()) {
					for(Lane sl : il.getSubLane()) {
						assertTrue(sl instanceof ConflictingLane);
						assertEquals(sl.getParentLane(), il);
						confLanes.add((ConflictingLane) sl);
					}					
				}				
			}
			for(PedestrianLane pl : road.getPedestrianLane()) {
				confLanes.add(pl);
			}
			confLanes = confLanes.stream().sorted(Comparator.comparing(ConflictingLane::getIndex)).collect(Collectors.toList());
			for(int c = 0; c < confLanes.size(); c++) {
				ConflictingLane cl = confLanes.get(c);				
				assertFalse(cl.getConflictingLane().isEmpty());
				for(ConflictingLane conf : cl.getConflictingLane()) {
					assertFalse(conf.getLink().isEmpty());
					assertEquals(conf, conf.getLink().get(0).getRefIncomingLane());						
				}
			}
		}
		
	}
	
	@Test
	public void test4ArmsConflictsCreation() {
		ServiceChecker<IntersectionService> checker = createCheckerTrackedForCleanUp(IntersectionService.class);
		checker.start();
		assertNotNull(checker);
		assertEquals(1, checker.getCurrentCreateCount(true));
		
		IntersectionService service = getService(IntersectionService.class);
		assertNotNull(service);
		
		Map<Integer, String> options = new HashMap<Integer, String>();
		options.put(0, IntersectionConstants.MAIN_STRAIGHT_TURNS_SEP);
		options.put(1, IntersectionConstants.SEC_STRAIGHT_TURNS_SEP);
		options.put(2, IntersectionConstants.MAIN_STRAIGHT_TURNS_SEP);
		options.put(3, IntersectionConstants.SEC_STRAIGHT_TURNS_SEP);
		Intersection intersection = service.createIntersection(options);
		
		assertNotNull(intersection);
		for(int i = 0; i < intersection.getRoad().size(); i++) {
			Road road = intersection.getRoad().get(i);
			List<ConflictingLane> confLanes = new LinkedList<ConflictingLane>();
			for(IncomingLane il : road.getIncomingLane()) {
				confLanes.add(il);
				if(!il.getSubLane().isEmpty()) {
					for(Lane sl : il.getSubLane()) {
						assertTrue(sl instanceof ConflictingLane);
						assertEquals(sl.getParentLane(), il);
						confLanes.add((ConflictingLane) sl);
					}					
				}
			}
			for(PedestrianLane pl : road.getPedestrianLane()) {
				confLanes.add(pl);
			}
			confLanes = confLanes.stream().sorted(Comparator.comparing(ConflictingLane::getIndex)).collect(Collectors.toList());
			for(int c = 0; c < confLanes.size(); c++) {
				ConflictingLane cl = confLanes.get(c);			
				assertFalse(cl.getConflictingLane().isEmpty());
				for(ConflictingLane conf : cl.getConflictingLane()) {
					assertFalse(conf.getLink().isEmpty());
					assertEquals(conf, conf.getLink().get(0).getRefIncomingLane());						
				}
			}
		}
		
	}
	
	@Test
	public void test4ArmsAllRightMergeConflictsCreation() {
		ServiceChecker<IntersectionService> checker = createCheckerTrackedForCleanUp(IntersectionService.class);
		checker.start();
		assertNotNull(checker);
		assertEquals(1, checker.getCurrentCreateCount(true));
		
		IntersectionService service = getService(IntersectionService.class);
		assertNotNull(service);
		
		Map<Integer, String> options = new HashMap<Integer, String>();
		options.put(0, IntersectionConstants.MAIN_STRAIGHT_RIGHT_MERGE);
		options.put(1, IntersectionConstants.SEC_STRAIGHT_RIGHT_MERGE);
		options.put(2, IntersectionConstants.MAIN_STRAIGHT_RIGHT_MERGE);
		options.put(3, IntersectionConstants.SEC_STRAIGHT_RIGHT_MERGE);
		Intersection intersection = service.createIntersection(options);
		
		assertNotNull(intersection);
		for(int i = 0; i < intersection.getRoad().size(); i++) {
			Road road = intersection.getRoad().get(i);
			List<ConflictingLane> confLanes = new LinkedList<ConflictingLane>();
			for(IncomingLane il : road.getIncomingLane()) {
				confLanes.add(il);
				if(!il.getSubLane().isEmpty()) {
					for(Lane sl : il.getSubLane()) {
						assertTrue(sl instanceof ConflictingLane);
						assertEquals(sl.getParentLane(), il);
						confLanes.add((ConflictingLane) sl);
					}					
				}
			}
			confLanes = confLanes.stream().sorted(Comparator.comparing(ConflictingLane::getIndex)).collect(Collectors.toList());
			for(int c = 0; c < confLanes.size(); c++) {
				ConflictingLane cl = confLanes.get(c);				
				assertFalse(cl.getConflictingLane().isEmpty());
				for(ConflictingLane conf : cl.getConflictingLane()) {
					assertFalse(conf.getLink().isEmpty());
					assertEquals(conf, conf.getLink().get(0).getRefIncomingLane());						
				}
			}
		}
		
	}
	
}
