/**
 * Copyright (c) 2012 - 2018 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package de.dim.trafficos.simulator.tests;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;

import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.gecko.core.tests.AbstractOSGiTest;
import org.gecko.core.tests.ServiceChecker;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.runners.MockitoJUnitRunner;
import org.osgi.framework.FrameworkUtil;

import de.dim.trafficos.model.device.Intersection;
import de.dim.trafficos.model.device.Phase;
import de.dim.trafficos.model.device.PhaseGroup;
import de.dim.trafficos.simulator.api.IntersectionConstants;
import de.dim.trafficos.simulator.api.IntersectionService;
import de.dim.trafficos.simulator.api.SignalPlanConstants;
import de.dim.trafficos.simulator.api.SignalPlanService;


/**
 * Tests the implementation of the SignalPlanService methods for creating the Phases for a 3-arms intersection.
 * All possible combinations of Lanes are taken into account for this type of Intersection. 
 * 
 * @since 1.0
 */
@RunWith(MockitoJUnitRunner.class)
public class SigPlanPhases3ArmsTest extends AbstractOSGiTest {
	
	private IntersectionService service;
	private SignalPlanService sigService;
	private Map<Integer, String> options;
	
	/**
	 * Creates a new instance.
	 * @param bundleContext
	 */
	public SigPlanPhases3ArmsTest() {
		super(FrameworkUtil.getBundle(SigPlanPhases3ArmsTest.class).getBundleContext());
	}

	/**
	 * Here you can put everything you want to be executed before every test
	 */
	public void doBefore() {

		options = new HashMap<Integer, String>();
	}
	
	/**
	 * Here you can put everything you want to be executed after every test
	 */
	public void doAfter() {
		
	}
	
	private void setupServices() {
		ServiceChecker<IntersectionService> checker = createCheckerTrackedForCleanUp(IntersectionService.class);
		checker.start();
		assertNotNull(checker);
		assertEquals(1, checker.getCurrentCreateCount(true));
		
		service = getService(IntersectionService.class);
		assertNotNull(service);
		
		ServiceChecker<SignalPlanService> sigChecker = createCheckerTrackedForCleanUp(SignalPlanService.class);
		sigChecker.start();
		assertNotNull(sigChecker);
		assertEquals(1, sigChecker.getCurrentCreateCount(true));
		
		sigService = getService(SignalPlanService.class);
		assertNotNull(sigService);	
	}
	
	/**
	 * All the possible Phases for a 3 Arms Intersection of the where the roads are of types:
	 * <code>IntersectionConstants.MAIN_STRAIGHT_RIGHT_MERGE<code>
	 * <code>IntersectionConstants.MAIN_STRAIGHT_LEFT_SEP<code>
	 * <code>IntersectionConstants.SEC_LEFT_RIGHT_MERGE<code>
	 */
	@Test
	public void test3ArmsDIMAllPhases() {
		setupServices();
		
		options.put(0, IntersectionConstants.MAIN_STRAIGHT_RIGHT_MERGE);
		options.put(1, IntersectionConstants.MAIN_STRAIGHT_LEFT_SEP);
		options.put(2, IntersectionConstants.SEC_LEFT_RIGHT_MERGE);
		Intersection intersection = service.createIntersection(options);
		
		assertNotNull(intersection);
		assertFalse(intersection.getRoad().isEmpty());
		assertEquals(options.size(), intersection.getRoad().size());
		
		List<Phase> phases = sigService.createPhases(intersection, SignalPlanConstants.ALL_PHASES);
		phases = phases.stream().sorted(Comparator.comparing(Phase::getId)).collect(Collectors.toList());
		assertFalse(phases.isEmpty());
		assertEquals(5, phases.size());
		assertEquals(3, phases.get(0).getLane().size());
		assertEquals(2, phases.get(1).getLane().size());
		assertEquals(2, phases.get(2).getLane().size());
		assertEquals(3, phases.get(3).getLane().size());
		assertEquals(3, phases.get(4).getLane().size());
		assertEquals(5, intersection.getPhase().size());
		
		for(Phase p : phases) {
			assertEquals(phases.size()-1, p.getTransition().size());
		}
		sigService.createPhaseGroups(intersection);
		List<PhaseGroup> phGroups = intersection.getPhaseGroup().stream().sorted(Comparator.comparing(PhaseGroup::getPenalty))
				.collect(Collectors.toList());
		assertEquals(3, phGroups.size());
	}
	
	
	/**
	 * All the possible Phases WITHOUT PEDESTRIANS for a 3 Arms Intersection of the where the roads are of types:
	 * <code>IntersectionConstants.MAIN_STRAIGHT_RIGHT_MERGE<code>
	 * <code>IntersectionConstants.MAIN_STRAIGHT_LEFT_SEP<code>
	 * <code>IntersectionConstants.SEC_LEFT_RIGHT_MERGE<code>
	 */
	@Test
	public void test3ArmsDIMNoPed() {		
		setupServices();
		options.put(0, IntersectionConstants.MAIN_STRAIGHT_RIGHT_MERGE);
		options.put(1, IntersectionConstants.MAIN_STRAIGHT_LEFT_SEP);
		options.put(2, IntersectionConstants.SEC_LEFT_RIGHT_MERGE);
		Intersection intersection = service.createIntersection(options);
		
		assertNotNull(intersection);
		assertFalse(intersection.getRoad().isEmpty());
		assertEquals(options.size(), intersection.getRoad().size());
		
		List<Phase> phases = sigService.createPhases(intersection, SignalPlanConstants.PHASES_WITHOUT_PEDESTRIAN);
		phases = phases.stream().sorted(Comparator.comparing(Phase::getId)).collect(Collectors.toList());
		assertFalse(phases.isEmpty());
		assertEquals(3, phases.size());
		assertEquals(2, phases.get(0).getLane().size());
		assertEquals(2, phases.get(1).getLane().size());
		assertEquals(1, phases.get(2).getLane().size());
		assertEquals(3, intersection.getPhase().size());
		
		for(Phase p : phases) {
			assertEquals(phases.size()-1, p.getTransition().size());
		}
	}
	
	/**
	 * All the possible Phases WITHOUT THE PHASES WITH ONLY PEDESTRIANS for a 3 Arms Intersection of the where 
	 * the roads are of types:
	 * <code>IntersectionConstants.MAIN_STRAIGHT_RIGHT_MERGE<code>
	 * <code>IntersectionConstants.MAIN_STRAIGHT_LEFT_SEP<code>
	 * <code>IntersectionConstants.SEC_LEFT_RIGHT_MERGE<code>
	 */
	@Test
	public void test3ArmsDIMNotOnlyPed() {
		setupServices();
		options.put(0, IntersectionConstants.MAIN_STRAIGHT_RIGHT_MERGE);
		options.put(1, IntersectionConstants.MAIN_STRAIGHT_LEFT_SEP);
		options.put(2, IntersectionConstants.SEC_LEFT_RIGHT_MERGE);
		Intersection intersection = service.createIntersection(options);
		
		assertNotNull(intersection);
		assertFalse(intersection.getRoad().isEmpty());
		assertEquals(options.size(), intersection.getRoad().size());
		
		List<Phase> phases = sigService.createPhases(intersection, SignalPlanConstants.PHASES_NOT_ONLY_PEDESTRIAN);
		phases = phases.stream().sorted(Comparator.comparing(Phase::getId)).collect(Collectors.toList());
		assertFalse(phases.isEmpty());
		assertEquals(4, phases.size());
		assertEquals(3, phases.get(0).getLane().size());
		assertEquals(2, phases.get(1).getLane().size());
		assertEquals(2, phases.get(2).getLane().size());
		assertEquals(3, phases.get(3).getLane().size());
		assertEquals(4, intersection.getPhase().size());
		
		for(Phase p : phases) {
			assertEquals(phases.size()-1, p.getTransition().size());
		}
	}
	
	
	/**
	 * All the possible Phases for a 3 Arms Intersection of the where the roads are of types:
	 * <code>IntersectionConstants.MAIN_STRAIGHT_RIGHT_MERGE<code>
	 * <code>IntersectionConstants.MAIN_STRAIGHT_LEFT_MERGE<code>
	 * <code>IntersectionConstants.SEC_LEFT_RIGHT_MERGE<code>
	 */
	@Test 
	public void test3ArmsMergeAllPhases() {
		setupServices();
		options.put(0, IntersectionConstants.MAIN_STRAIGHT_RIGHT_MERGE);
		options.put(1, IntersectionConstants.MAIN_STRAIGHT_LEFT_MERGE);
		options.put(2, IntersectionConstants.SEC_LEFT_RIGHT_MERGE);
		Intersection intersection = service.createIntersection(options);
		
		assertNotNull(intersection);
		assertFalse(intersection.getRoad().isEmpty());
		assertEquals(options.size(), intersection.getRoad().size());
		
		List<Phase> phases = sigService.createPhases(intersection, SignalPlanConstants.ALL_PHASES);
		phases = phases.stream().sorted(Comparator.comparing(Phase::getId)).collect(Collectors.toList());
		assertFalse(phases.isEmpty());
		assertEquals(3, phases.size());
		assertEquals(3, phases.get(0).getLane().size());
		assertEquals(3, phases.get(1).getLane().size());
		assertEquals(3, phases.get(2).getLane().size());
		assertEquals(3, intersection.getPhase().size());
		
		for(Phase p : phases) {
			assertEquals(phases.size()-1, p.getTransition().size());
		}		
	}
	
	
	/**
	 * All the possible Phases WITHOUT PEDESTRIAN for a 3 Arms Intersection of the where the roads are of types:
	 * <code>IntersectionConstants.MAIN_STRAIGHT_RIGHT_MERGE<code>
	 * <code>IntersectionConstants.MAIN_STRAIGHT_LEFT_MERGE<code>
	 * <code>IntersectionConstants.SEC_LEFT_RIGHT_MERGE<code>
	 */
	@Test 
	public void test3ArmsMergeNoPed() {
		setupServices();
		options.put(0, IntersectionConstants.MAIN_STRAIGHT_RIGHT_MERGE);
		options.put(1, IntersectionConstants.MAIN_STRAIGHT_LEFT_MERGE);
		options.put(2, IntersectionConstants.SEC_LEFT_RIGHT_MERGE);
		Intersection intersection = service.createIntersection(options);
		
		assertNotNull(intersection);
		assertFalse(intersection.getRoad().isEmpty());
		assertEquals(options.size(), intersection.getRoad().size());
		
		List<Phase> phases = sigService.createPhases(intersection, SignalPlanConstants.PHASES_WITHOUT_PEDESTRIAN);
		phases = phases.stream().sorted(Comparator.comparing(Phase::getId)).collect(Collectors.toList());
		assertFalse(phases.isEmpty());
		assertEquals(2, phases.size());
		assertEquals(2, phases.get(0).getLane().size());
		assertEquals(1, phases.get(1).getLane().size());
		assertEquals(2, intersection.getPhase().size());
		
		for(Phase p : phases) {
			assertEquals(phases.size()-1, p.getTransition().size());
		}
		
	}
	
	/**
	 * All the possible Phases WITHOUT LANES WITH ONLY PEDESTRIAN for a 3 Arms Intersection of the where 
	 * the roads are of types:
	 * <code>IntersectionConstants.MAIN_STRAIGHT_RIGHT_MERGE<code>
	 * <code>IntersectionConstants.MAIN_STRAIGHT_LEFT_MERGE<code>
	 * <code>IntersectionConstants.SEC_LEFT_RIGHT_MERGE<code>
	 */
	@Test 
	public void test3ArmsMergeNotOnlyPed() {		
		setupServices();
		options.put(0, IntersectionConstants.MAIN_STRAIGHT_RIGHT_MERGE);
		options.put(1, IntersectionConstants.MAIN_STRAIGHT_LEFT_MERGE);
		options.put(2, IntersectionConstants.SEC_LEFT_RIGHT_MERGE);
		Intersection intersection = service.createIntersection(options);
		
		assertNotNull(intersection);
		assertFalse(intersection.getRoad().isEmpty());
		assertEquals(options.size(), intersection.getRoad().size());
		
		List<Phase> phases = sigService.createPhases(intersection, SignalPlanConstants.PHASES_NOT_ONLY_PEDESTRIAN);
		phases = phases.stream().sorted(Comparator.comparing(Phase::getId)).collect(Collectors.toList());
		assertFalse(phases.isEmpty());
		assertEquals(2, phases.size());
		assertEquals(3, phases.get(0).getLane().size());
		assertEquals(3, phases.get(1).getLane().size());
		assertEquals(2, intersection.getPhase().size());
		
		for(Phase p : phases) {
			assertEquals(phases.size()-1, p.getTransition().size());
		}		
	}
	
	/**
	 * All the possible Phases for a 3 Arms Intersection of the where the roads are of types:
	 * <code>IntersectionConstants.MAIN_STRAIGHT_RIGHT_SEP<code>
	 * <code>IntersectionConstants.MAIN_STRAIGHT_LEFT_SEP<code>
	 * <code>IntersectionConstants.SEC_LEFT_RIGHT_SEP<code>
	 */
	@Test 
	public void test3ArmsAllSepAllPhases() {
		setupServices();
		options.put(0, IntersectionConstants.MAIN_STRAIGHT_RIGHT_SEP);
		options.put(1, IntersectionConstants.MAIN_STRAIGHT_LEFT_SEP);
		options.put(2, IntersectionConstants.SEC_LEFT_RIGHT_SEP);
		Intersection intersection = service.createIntersection(options);
		
		assertNotNull(intersection);
		assertFalse(intersection.getRoad().isEmpty());
		assertEquals(options.size(), intersection.getRoad().size());
		
		List<Phase> phases = sigService.createPhases(intersection, SignalPlanConstants.ALL_PHASES);
		phases = phases.stream().sorted(Comparator.comparing(Phase::getId)).collect(Collectors.toList());
		assertFalse(phases.isEmpty());
		assertEquals(8, phases.size());
		assertEquals(3, phases.get(0).getLane().size());
		assertEquals(3, phases.get(1).getLane().size());
		assertEquals(3, phases.get(2).getLane().size());
		assertEquals(3, phases.get(3).getLane().size());
		assertEquals(3, phases.get(4).getLane().size());
		assertEquals(3, phases.get(5).getLane().size());
		assertEquals(3, phases.get(6).getLane().size());
		assertEquals(3, phases.get(7).getLane().size());
		assertEquals(8, intersection.getPhase().size());
		
		for(Phase p : phases) {
			assertEquals(phases.size()-1, p.getTransition().size());
		}		
	}
	
	
	/**
	 * All the possible Phases WITHOUT PEDESTRIAN for a 3 Arms Intersection of the where the roads are of types:
	 * <code>IntersectionConstants.MAIN_STRAIGHT_RIGHT_SEP<code>
	 * <code>IntersectionConstants.MAIN_STRAIGHT_LEFT_SEP<code>
	 * <code>IntersectionConstants.SEC_LEFT_RIGHT_SEP<code>
	 */
	@Test 
	public void test3ArmsAllSepNoPed() {
		setupServices();
		options.put(0, IntersectionConstants.MAIN_STRAIGHT_RIGHT_SEP);
		options.put(1, IntersectionConstants.MAIN_STRAIGHT_LEFT_SEP);
		options.put(2, IntersectionConstants.SEC_LEFT_RIGHT_SEP);
		Intersection intersection = service.createIntersection(options);
		
		assertNotNull(intersection);
		assertFalse(intersection.getRoad().isEmpty());
		assertEquals(options.size(), intersection.getRoad().size());
		
		List<Phase> phases = sigService.createPhases(intersection, SignalPlanConstants.PHASES_WITHOUT_PEDESTRIAN);
		phases = phases.stream().sorted(Comparator.comparing(Phase::getId)).collect(Collectors.toList());
		assertFalse(phases.isEmpty());
		assertEquals(4, phases.size());
		assertEquals(3, phases.get(0).getLane().size());
		assertEquals(3, phases.get(1).getLane().size());
		assertEquals(3, phases.get(2).getLane().size());
		assertEquals(3, phases.get(3).getLane().size());		
		assertEquals(4, intersection.getPhase().size());
		
		for(Phase p : phases) {
			assertEquals(phases.size()-1, p.getTransition().size());
		}		
	}
	
	/**
	 * All the possible Phases WITHOUT LANES WITH ONLY PEDESTRIAN for a 3 Arms Intersection of the where 
	 * the roads are of types:
	 * <code>IntersectionConstants.MAIN_STRAIGHT_RIGHT_SEP<code>
	 * <code>IntersectionConstants.MAIN_STRAIGHT_LEFT_SEP<code>
	 * <code>IntersectionConstants.SEC_LEFT_RIGHT_SEP<code>
	 */
	@Test 
	public void test3ArmsAllSepNotOnlyPed() {
		setupServices();
		options.put(0, IntersectionConstants.MAIN_STRAIGHT_RIGHT_SEP);
		options.put(1, IntersectionConstants.MAIN_STRAIGHT_LEFT_SEP);
		options.put(2, IntersectionConstants.SEC_LEFT_RIGHT_SEP);
		Intersection intersection = service.createIntersection(options);
		
		assertNotNull(intersection);
		assertFalse(intersection.getRoad().isEmpty());
		assertEquals(options.size(), intersection.getRoad().size());
		
		List<Phase> phases = sigService.createPhases(intersection, SignalPlanConstants.PHASES_NOT_ONLY_PEDESTRIAN);
		phases = phases.stream().sorted(Comparator.comparing(Phase::getId)).collect(Collectors.toList());
		assertFalse(phases.isEmpty());
		assertEquals(7, phases.size());
		assertEquals(3, phases.get(0).getLane().size());
		assertEquals(3, phases.get(1).getLane().size());
		assertEquals(3, phases.get(2).getLane().size());
		assertEquals(3, phases.get(3).getLane().size());
		assertEquals(3, phases.get(4).getLane().size());
		assertEquals(3, phases.get(5).getLane().size());
		assertEquals(3, phases.get(6).getLane().size());
		assertEquals(7, intersection.getPhase().size());
		
		for(Phase p : phases) {
			assertEquals(phases.size()-1, p.getTransition().size());
		}		
	}
	
	/**
	 * All the possible Phases for a 3 Arms Intersection of the where the roads are of types:
	 * <code>IntersectionConstants.MAIN_STRAIGHT_RIGHT_SEP<code>
	 * <code>IntersectionConstants.MAIN_STRAIGHT_LEFT_MERGE<code>
	 * <code>IntersectionConstants.SEC_LEFT_RIGHT_MERGE<code>
	 */
	@Test 
	public void test3ArmsRightSepAllPhases() {
		setupServices();
		options.put(0, IntersectionConstants.MAIN_STRAIGHT_RIGHT_SEP);
		options.put(1, IntersectionConstants.MAIN_STRAIGHT_LEFT_MERGE);
		options.put(2, IntersectionConstants.SEC_LEFT_RIGHT_MERGE);
		Intersection intersection = service.createIntersection(options);
		
		assertNotNull(intersection);
		assertFalse(intersection.getRoad().isEmpty());
		assertEquals(options.size(), intersection.getRoad().size());
		
		List<Phase> phases = sigService.createPhases(intersection, SignalPlanConstants.ALL_PHASES);
		phases = phases.stream().sorted(Comparator.comparing(Phase::getId)).collect(Collectors.toList());
		assertFalse(phases.isEmpty());
		assertEquals(6, phases.size());
		assertEquals(2, phases.get(0).getLane().size());
		assertEquals(3, phases.get(1).getLane().size());
		assertEquals(2, phases.get(2).getLane().size());
		assertEquals(2, phases.get(3).getLane().size());
		assertEquals(3, phases.get(4).getLane().size());
		assertEquals(3, phases.get(5).getLane().size());
		assertEquals(6, intersection.getPhase().size());
		
		for(Phase p : phases) {
			assertEquals(phases.size()-1, p.getTransition().size());
		}	
	}
	
	
	/**
	 * All the possible Phases WITHOUT PEDESTRIAN for a 3 Arms Intersection of the where the roads are of types:
	 * <code>IntersectionConstants.MAIN_STRAIGHT_RIGHT_SEP<code>
	 * <code>IntersectionConstants.MAIN_STRAIGHT_LEFT_MERGE<code>
	 * <code>IntersectionConstants.SEC_LEFT_RIGHT_MERGE<code>
	 */
	@Test 
	public void test3ArmsRightSepNoPed() {		
		setupServices();
		options.put(0, IntersectionConstants.MAIN_STRAIGHT_RIGHT_SEP);
		options.put(1, IntersectionConstants.MAIN_STRAIGHT_LEFT_MERGE);
		options.put(2, IntersectionConstants.SEC_LEFT_RIGHT_MERGE);
		Intersection intersection = service.createIntersection(options);
		
		assertNotNull(intersection);
		assertFalse(intersection.getRoad().isEmpty());
		assertEquals(options.size(), intersection.getRoad().size());
		
		List<Phase> phases = sigService.createPhases(intersection, SignalPlanConstants.PHASES_WITHOUT_PEDESTRIAN);
		phases = phases.stream().sorted(Comparator.comparing(Phase::getId)).collect(Collectors.toList());
		assertFalse(phases.isEmpty());
		assertEquals(3, phases.size());
		assertEquals(2, phases.get(0).getLane().size());
		assertEquals(2, phases.get(1).getLane().size());
		assertEquals(1, phases.get(2).getLane().size());
		assertEquals(3, intersection.getPhase().size());
		
		for(Phase p : phases) {
			assertEquals(phases.size()-1, p.getTransition().size());
		}	
	}
	
	/**
	 * All the possible Phases WITHOUT LANES WITH ONLY PEDESTRIN for a 3 Arms Intersection of the where 
	 * the roads are of types:
	 * <code>IntersectionConstants.MAIN_STRAIGHT_RIGHT_SEP<code>
	 * <code>IntersectionConstants.MAIN_STRAIGHT_LEFT_MERGE<code>
	 * <code>IntersectionConstants.SEC_LEFT_RIGHT_MERGE<code>
	 */
	@Test 
	public void test3ArmsRightSepNotOnlyPed() {		
		setupServices();
		options.put(0, IntersectionConstants.MAIN_STRAIGHT_RIGHT_SEP);
		options.put(1, IntersectionConstants.MAIN_STRAIGHT_LEFT_MERGE);
		options.put(2, IntersectionConstants.SEC_LEFT_RIGHT_MERGE);
		Intersection intersection = service.createIntersection(options);
		
		assertNotNull(intersection);
		assertFalse(intersection.getRoad().isEmpty());
		assertEquals(options.size(), intersection.getRoad().size());
		
		List<Phase> phases = sigService.createPhases(intersection, SignalPlanConstants.PHASES_NOT_ONLY_PEDESTRIAN);
		phases = phases.stream().sorted(Comparator.comparing(Phase::getId)).collect(Collectors.toList());
		assertFalse(phases.isEmpty());
		assertEquals(5, phases.size());
		assertEquals(2, phases.get(0).getLane().size());
		assertEquals(3, phases.get(1).getLane().size());
		assertEquals(2, phases.get(2).getLane().size());
		assertEquals(2, phases.get(3).getLane().size());
		assertEquals(3, phases.get(4).getLane().size());
		assertEquals(5, intersection.getPhase().size());
		
		for(Phase p : phases) {
			assertEquals(phases.size()-1, p.getTransition().size());
		}	
	}
	
	/**
	 * All the possible Phases for a 3 Arms Intersection of the where the roads are of types:
	 * <code>IntersectionConstants.MAIN_STRAIGHT_RIGHT_MERGE<code>
	 * <code>IntersectionConstants.MAIN_STRAIGHT_LEFT_MERGE<code>
	 * <code>IntersectionConstants.SEC_LEFT_RIGHT_SEP<code>
	 */
	@Test 
	public void test3ArmsTurnsSepAllPhases() {
		setupServices();
		options.put(0, IntersectionConstants.MAIN_STRAIGHT_RIGHT_MERGE);
		options.put(1, IntersectionConstants.MAIN_STRAIGHT_LEFT_MERGE);
		options.put(2, IntersectionConstants.SEC_LEFT_RIGHT_SEP);
		Intersection intersection = service.createIntersection(options);
		
		assertNotNull(intersection);
		assertFalse(intersection.getRoad().isEmpty());
		assertEquals(options.size(), intersection.getRoad().size());
		
		List<Phase> phases = sigService.createPhases(intersection, SignalPlanConstants.ALL_PHASES);
		phases = phases.stream().sorted(Comparator.comparing(Phase::getId)).collect(Collectors.toList());
		assertFalse(phases.isEmpty());
		assertEquals(6, phases.size());
		assertEquals(3, phases.get(0).getLane().size());
		assertEquals(2, phases.get(1).getLane().size());
		assertEquals(2, phases.get(2).getLane().size());
		assertEquals(2, phases.get(3).getLane().size());
		assertEquals(2, phases.get(4).getLane().size());
		assertEquals(3, phases.get(5).getLane().size());
		assertEquals(6, intersection.getPhase().size());
		
		for(Phase p : phases) {
			assertEquals(phases.size()-1, p.getTransition().size());
		}	
	}
	
	/**
	 * All the possible Phases WITHOUT PEDESTRIAN for a 3 Arms Intersection of the where the roads are of types:
	 * <code>IntersectionConstants.MAIN_STRAIGHT_RIGHT_MERGE<code>
	 * <code>IntersectionConstants.MAIN_STRAIGHT_LEFT_MERGE<code>
	 * <code>IntersectionConstants.SEC_LEFT_RIGHT_SEP<code>
	 */
	@Test 
	public void test3ArmsTurnsSepNoPed() {
		setupServices();
		options.put(0, IntersectionConstants.MAIN_STRAIGHT_RIGHT_MERGE);
		options.put(1, IntersectionConstants.MAIN_STRAIGHT_LEFT_MERGE);
		options.put(2, IntersectionConstants.SEC_LEFT_RIGHT_SEP);
		Intersection intersection = service.createIntersection(options);
		
		assertNotNull(intersection);
		assertFalse(intersection.getRoad().isEmpty());
		assertEquals(options.size(), intersection.getRoad().size());
		
		List<Phase> phases = sigService.createPhases(intersection, SignalPlanConstants.PHASES_WITHOUT_PEDESTRIAN);
		phases = phases.stream().sorted(Comparator.comparing(Phase::getId)).collect(Collectors.toList());
		assertFalse(phases.isEmpty());
		assertEquals(3, phases.size());
		assertEquals(2, phases.get(0).getLane().size());
		assertEquals(2, phases.get(1).getLane().size());
		assertEquals(2, phases.get(2).getLane().size());		
		assertEquals(3, intersection.getPhase().size());
		
		for(Phase p : phases) {
			assertEquals(phases.size()-1, p.getTransition().size());
		}	
	}
	
	/**
	 * All the possible Phases WITHOUT LANES WITH ONLY PEDESTRIAN for a 3 Arms Intersection of the where 
	 * the roads are of types:
	 * <code>IntersectionConstants.MAIN_STRAIGHT_RIGHT_MERGE<code>
	 * <code>IntersectionConstants.MAIN_STRAIGHT_LEFT_MERGE<code>
	 * <code>IntersectionConstants.SEC_LEFT_RIGHT_SEP<code>
	 */
	@Test 
	public void test3ArmsTurnsSepNotOnlyPed() {
		setupServices();
		options.put(0, IntersectionConstants.MAIN_STRAIGHT_RIGHT_MERGE);
		options.put(1, IntersectionConstants.MAIN_STRAIGHT_LEFT_MERGE);
		options.put(2, IntersectionConstants.SEC_LEFT_RIGHT_SEP);
		Intersection intersection = service.createIntersection(options);
		
		assertNotNull(intersection);
		assertFalse(intersection.getRoad().isEmpty());
		assertEquals(options.size(), intersection.getRoad().size());
		
		List<Phase> phases = sigService.createPhases(intersection, SignalPlanConstants.PHASES_NOT_ONLY_PEDESTRIAN);
		phases = phases.stream().sorted(Comparator.comparing(Phase::getId)).collect(Collectors.toList());
		assertFalse(phases.isEmpty());
		assertEquals(5, phases.size());
		assertEquals(3, phases.get(0).getLane().size());
		assertEquals(2, phases.get(1).getLane().size());
		assertEquals(2, phases.get(2).getLane().size());
		assertEquals(2, phases.get(3).getLane().size());
		assertEquals(2, phases.get(4).getLane().size());
		assertEquals(5, intersection.getPhase().size());
		
		for(Phase p : phases) {
			assertEquals(phases.size()-1, p.getTransition().size());
		}	
	}
	
	/**
	 * All the possible Phases for a 3 Arms Intersection of the where the roads are of types:
	 * <code>IntersectionConstants.MAIN_STRAIGHT_RIGHT_SEP<code>
	 * <code>IntersectionConstants.MAIN_STRAIGHT_LEFT_MERGE<code>
	 * <code>IntersectionConstants.SEC_LEFT_RIGHT_SEP<code>
	 */
	@Test 
	public void test3ArmsLeftMergeAllPhases() {		
		setupServices();
		options.put(0, IntersectionConstants.MAIN_STRAIGHT_RIGHT_SEP);
		options.put(1, IntersectionConstants.MAIN_STRAIGHT_LEFT_MERGE);
		options.put(2, IntersectionConstants.SEC_LEFT_RIGHT_SEP);
		Intersection intersection = service.createIntersection(options);
		
		assertNotNull(intersection);
		assertFalse(intersection.getRoad().isEmpty());
		assertEquals(options.size(), intersection.getRoad().size());
		
		List<Phase> phases = sigService.createPhases(intersection, SignalPlanConstants.ALL_PHASES);
		phases = phases.stream().sorted(Comparator.comparing(Phase::getId)).collect(Collectors.toList());
		assertFalse(phases.isEmpty());
		assertEquals(8, phases.size());
		assertEquals(2, phases.get(0).getLane().size());
		assertEquals(3, phases.get(1).getLane().size());
		assertEquals(3, phases.get(2).getLane().size());
		assertEquals(2, phases.get(3).getLane().size());
		assertEquals(2, phases.get(4).getLane().size());
		assertEquals(2, phases.get(5).getLane().size());
		assertEquals(2, phases.get(6).getLane().size());
		assertEquals(3, phases.get(7).getLane().size());
		assertEquals(8, intersection.getPhase().size());
		
		for(Phase p : phases) {
			assertEquals(phases.size()-1, p.getTransition().size());
		}	
	}
	
	
	/**
	 * All the possible Phases WITHOUT PEDESTRIAN for a 3 Arms Intersection of the where the roads are of types:
	 * <code>IntersectionConstants.MAIN_STRAIGHT_RIGHT_SEP<code>
	 * <code>IntersectionConstants.MAIN_STRAIGHT_LEFT_MERGE<code>
	 * <code>IntersectionConstants.SEC_LEFT_RIGHT_SEP<code>
	 */
	@Test 
	public void test3ArmsLeftMergeNoPed() {
		setupServices();
		options.put(0, IntersectionConstants.MAIN_STRAIGHT_RIGHT_SEP);
		options.put(1, IntersectionConstants.MAIN_STRAIGHT_LEFT_MERGE);
		options.put(2, IntersectionConstants.SEC_LEFT_RIGHT_SEP);
		Intersection intersection = service.createIntersection(options);
		
		assertNotNull(intersection);
		assertFalse(intersection.getRoad().isEmpty());
		assertEquals(options.size(), intersection.getRoad().size());
		
		List<Phase> phases = sigService.createPhases(intersection, SignalPlanConstants.PHASES_WITHOUT_PEDESTRIAN);
		phases = phases.stream().sorted(Comparator.comparing(Phase::getId)).collect(Collectors.toList());
		assertFalse(phases.isEmpty());
		assertEquals(3, phases.size());
		assertEquals(2, phases.get(0).getLane().size());
		assertEquals(3, phases.get(1).getLane().size());
		assertEquals(2, phases.get(2).getLane().size());		
		assertEquals(3, intersection.getPhase().size());
		
		for(Phase p : phases) {
			assertEquals(phases.size()-1, p.getTransition().size());
		}	
	}
	
	/**
	 * All the possible Phases WITHOUT LANES WITH ONLY PEDESTRIAN for a 3 Arms Intersection of the where 
	 * the roads are of types:
	 * <code>IntersectionConstants.MAIN_STRAIGHT_RIGHT_SEP<code>
	 * <code>IntersectionConstants.MAIN_STRAIGHT_LEFT_MERGE<code>
	 * <code>IntersectionConstants.SEC_LEFT_RIGHT_SEP<code>
	 */
	@Test 
	public void test3ArmsLeftMergeNotOnlyPed() {		
		setupServices();
		options.put(0, IntersectionConstants.MAIN_STRAIGHT_RIGHT_SEP);
		options.put(1, IntersectionConstants.MAIN_STRAIGHT_LEFT_MERGE);
		options.put(2, IntersectionConstants.SEC_LEFT_RIGHT_SEP);
		Intersection intersection = service.createIntersection(options);
		
		assertNotNull(intersection);
		assertFalse(intersection.getRoad().isEmpty());
		assertEquals(options.size(), intersection.getRoad().size());
		
		List<Phase> phases = sigService.createPhases(intersection, SignalPlanConstants.PHASES_NOT_ONLY_PEDESTRIAN);
		phases = phases.stream().sorted(Comparator.comparing(Phase::getId)).collect(Collectors.toList());
		assertFalse(phases.isEmpty());
		assertEquals(7, phases.size());
		assertEquals(2, phases.get(0).getLane().size());
		assertEquals(3, phases.get(1).getLane().size());
		assertEquals(3, phases.get(2).getLane().size());
		assertEquals(2, phases.get(3).getLane().size());
		assertEquals(2, phases.get(4).getLane().size());
		assertEquals(2, phases.get(5).getLane().size());
		assertEquals(2, phases.get(6).getLane().size());
		assertEquals(7, intersection.getPhase().size());
		
		for(Phase p : phases) {
			assertEquals(phases.size()-1, p.getTransition().size());
		}	
	}
	
	/**
	 * All the possible Phases for a 3 Arms Intersection of the where the roads are of types:
	 * <code>IntersectionConstants.MAIN_STRAIGHT_RIGHT_SEP<code>
	 * <code>IntersectionConstants.MAIN_STRAIGHT_LEFT_SEP<code>
	 * <code>IntersectionConstants.SEC_LEFT_RIGHT_MERGE<code>
	 */
	@Test 
	public void test3ArmsTurnsMergeAllPhases() {		
		setupServices();
		options.put(0, IntersectionConstants.MAIN_STRAIGHT_RIGHT_SEP);
		options.put(1, IntersectionConstants.MAIN_STRAIGHT_LEFT_SEP);
		options.put(2, IntersectionConstants.SEC_LEFT_RIGHT_MERGE);
		Intersection intersection = service.createIntersection(options);
		
		assertNotNull(intersection);
		assertFalse(intersection.getRoad().isEmpty());
		assertEquals(options.size(), intersection.getRoad().size());
		
		List<Phase> phases = sigService.createPhases(intersection, SignalPlanConstants.ALL_PHASES);
		phases = phases.stream().sorted(Comparator.comparing(Phase::getId)).collect(Collectors.toList());
		assertFalse(phases.isEmpty());
		assertEquals(7, phases.size());
		assertEquals(3, phases.get(0).getLane().size());
		assertEquals(3, phases.get(1).getLane().size());
		assertEquals(3, phases.get(2).getLane().size());
		assertEquals(2, phases.get(3).getLane().size());
		assertEquals(2, phases.get(4).getLane().size());
		assertEquals(3, phases.get(5).getLane().size());
		assertEquals(3, phases.get(6).getLane().size());
		assertEquals(7, intersection.getPhase().size());
		
		for(Phase p : phases) {
			assertEquals(phases.size()-1, p.getTransition().size());
		}	
	}
	
	
	/**
	 * All the possible Phases WITHOUT PEDESTRIAN for a 3 Arms Intersection of the where the roads are of types:
	 * <code>IntersectionConstants.MAIN_STRAIGHT_RIGHT_SEP<code>
	 * <code>IntersectionConstants.MAIN_STRAIGHT_LEFT_SEP<code>
	 * <code>IntersectionConstants.SEC_LEFT_RIGHT_MERGE<code>
	 */
	@Test 
	public void test3ArmsTurnsMergeNoPed() {
		setupServices();
		options.put(0, IntersectionConstants.MAIN_STRAIGHT_RIGHT_SEP);
		options.put(1, IntersectionConstants.MAIN_STRAIGHT_LEFT_SEP);
		options.put(2, IntersectionConstants.SEC_LEFT_RIGHT_MERGE);
		Intersection intersection = service.createIntersection(options);
		
		assertNotNull(intersection);
		assertFalse(intersection.getRoad().isEmpty());
		assertEquals(options.size(), intersection.getRoad().size());
		
		List<Phase> phases = sigService.createPhases(intersection, SignalPlanConstants.PHASES_WITHOUT_PEDESTRIAN);
		phases = phases.stream().sorted(Comparator.comparing(Phase::getId)).collect(Collectors.toList());
		assertFalse(phases.isEmpty());
		assertEquals(3, phases.size());
		assertEquals(3, phases.get(0).getLane().size());
		assertEquals(2, phases.get(1).getLane().size());
		assertEquals(2, phases.get(2).getLane().size());		
		assertEquals(3, intersection.getPhase().size());
		
		for(Phase p : phases) {
			assertEquals(phases.size()-1, p.getTransition().size());
		}		
	}
	
	/**
	 * All the possible Phases WITHOUT LANES WITH ONLY PEDESTRIAN for a 3 Arms Intersection of the where 
	 * the roads are of types:
	 * <code>IntersectionConstants.MAIN_STRAIGHT_RIGHT_SEP<code>
	 * <code>IntersectionConstants.MAIN_STRAIGHT_LEFT_SEP<code>
	 * <code>IntersectionConstants.SEC_LEFT_RIGHT_MERGE<code>
	 */
	@Test 
	public void test3ArmsTurnsMergeNotOnlyPed() {		
		setupServices();
		options.put(0, IntersectionConstants.MAIN_STRAIGHT_RIGHT_SEP);
		options.put(1, IntersectionConstants.MAIN_STRAIGHT_LEFT_SEP);
		options.put(2, IntersectionConstants.SEC_LEFT_RIGHT_MERGE);
		Intersection intersection = service.createIntersection(options);
		
		assertNotNull(intersection);
		assertFalse(intersection.getRoad().isEmpty());
		assertEquals(options.size(), intersection.getRoad().size());
		
		List<Phase> phases = sigService.createPhases(intersection, SignalPlanConstants.PHASES_NOT_ONLY_PEDESTRIAN);
		phases = phases.stream().sorted(Comparator.comparing(Phase::getId)).collect(Collectors.toList());
		assertFalse(phases.isEmpty());
		assertEquals(6, phases.size());
		assertEquals(3, phases.get(0).getLane().size());
		assertEquals(3, phases.get(1).getLane().size());
		assertEquals(3, phases.get(2).getLane().size());
		assertEquals(2, phases.get(3).getLane().size());
		assertEquals(2, phases.get(4).getLane().size());
		assertEquals(3, phases.get(5).getLane().size());
		assertEquals(6, intersection.getPhase().size());
		
		for(Phase p : phases) {
			assertEquals(phases.size()-1, p.getTransition().size());
		}
	}
	
	/**
	 * All the possible Phases for a 3 Arms Intersection of the where the roads are of types:
	 * <code>IntersectionConstants.MAIN_STRAIGHT_RIGHT_MERGE<code>
	 * <code>IntersectionConstants.MAIN_STRAIGHT_LEFT_SEP<code>
	 * <code>IntersectionConstants.SEC_LEFT_RIGHT_SEP<code>
	 */
	@Test 
	public void test3ArmsRightMergeAllPhases() {
		setupServices();
		options.put(0, IntersectionConstants.MAIN_STRAIGHT_RIGHT_MERGE);
		options.put(1, IntersectionConstants.MAIN_STRAIGHT_LEFT_SEP);
		options.put(2, IntersectionConstants.SEC_LEFT_RIGHT_SEP);
		Intersection intersection = service.createIntersection(options);
		
		assertNotNull(intersection);
		assertFalse(intersection.getRoad().isEmpty());
		assertEquals(options.size(), intersection.getRoad().size());
		
		List<Phase> phases = sigService.createPhases(intersection, SignalPlanConstants.ALL_PHASES);
		phases = phases.stream().sorted(Comparator.comparing(Phase::getId)).collect(Collectors.toList());
		assertFalse(phases.isEmpty());
		assertEquals(6, phases.size());
		assertEquals(3, phases.get(0).getLane().size());
		assertEquals(3, phases.get(1).getLane().size());
		assertEquals(3, phases.get(2).getLane().size());
		assertEquals(2, phases.get(3).getLane().size());
		assertEquals(2, phases.get(4).getLane().size());
		assertEquals(3, phases.get(5).getLane().size());
		assertEquals(6, intersection.getPhase().size());
		
		for(Phase p : phases) {
			assertEquals(phases.size()-1, p.getTransition().size());
		}	
	}
	
	/**
	 * All the possible Phases WITHOUT PEDESTRIAN for a 3 Arms Intersection of the where the roads are of types:
	 * <code>IntersectionConstants.MAIN_STRAIGHT_RIGHT_MERGE<code>
	 * <code>IntersectionConstants.MAIN_STRAIGHT_LEFT_SEP<code>
	 * <code>IntersectionConstants.SEC_LEFT_RIGHT_SEP<code>
	 */
	@Test 
	public void test3ArmsRightMergeNoPed() {
		setupServices();
		options.put(0, IntersectionConstants.MAIN_STRAIGHT_RIGHT_MERGE);
		options.put(1, IntersectionConstants.MAIN_STRAIGHT_LEFT_SEP);
		options.put(2, IntersectionConstants.SEC_LEFT_RIGHT_SEP);
		Intersection intersection = service.createIntersection(options);
		
		assertNotNull(intersection);
		assertFalse(intersection.getRoad().isEmpty());
		assertEquals(options.size(), intersection.getRoad().size());
		
		List<Phase> phases = sigService.createPhases(intersection, SignalPlanConstants.PHASES_WITHOUT_PEDESTRIAN);
		phases = phases.stream().sorted(Comparator.comparing(Phase::getId)).collect(Collectors.toList());
		assertFalse(phases.isEmpty());
		assertEquals(3, phases.size());
		assertEquals(2, phases.get(0).getLane().size());
		assertEquals(3, phases.get(1).getLane().size());
		assertEquals(2, phases.get(2).getLane().size());		
		assertEquals(3, intersection.getPhase().size());
		
		for(Phase p : phases) {
			assertEquals(phases.size()-1, p.getTransition().size());
		}	
	}
	
	/**
	 * All the possible Phases WITHOUT LANES WITH ONLY PEDESTRIAN for a 3 Arms Intersection of the where 
	 * the roads are of types:
	 * <code>IntersectionConstants.MAIN_STRAIGHT_RIGHT_MERGE<code>
	 * <code>IntersectionConstants.MAIN_STRAIGHT_LEFT_SEP<code>
	 * <code>IntersectionConstants.SEC_LEFT_RIGHT_SEP<code>
	 */
	@Test 
	public void test3ArmsRightMergeNotOnlyPed() {
		setupServices();
		options.put(0, IntersectionConstants.MAIN_STRAIGHT_RIGHT_MERGE);
		options.put(1, IntersectionConstants.MAIN_STRAIGHT_LEFT_SEP);
		options.put(2, IntersectionConstants.SEC_LEFT_RIGHT_SEP);
		Intersection intersection = service.createIntersection(options);
		
		assertNotNull(intersection);
		assertFalse(intersection.getRoad().isEmpty());
		assertEquals(options.size(), intersection.getRoad().size());
		
		List<Phase> phases = sigService.createPhases(intersection, SignalPlanConstants.PHASES_NOT_ONLY_PEDESTRIAN);
		phases = phases.stream().sorted(Comparator.comparing(Phase::getId)).collect(Collectors.toList());
		assertFalse(phases.isEmpty());
		assertEquals(5, phases.size());
		assertEquals(3, phases.get(0).getLane().size());
		assertEquals(3, phases.get(1).getLane().size());
		assertEquals(3, phases.get(2).getLane().size());
		assertEquals(2, phases.get(3).getLane().size());
		assertEquals(2, phases.get(4).getLane().size());
		assertEquals(5, intersection.getPhase().size());
		
		for(Phase p : phases) {
			assertEquals(phases.size()-1, p.getTransition().size());
		}	
	}
}
